"""This is a dialog for specifying a rubble mound's properties."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QLabel, QMessageBox

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.cmsflow.data.rm_structures_data import CALCULATION_METHODS, PARAMETER_TYPE_CONSTANT_IDX, PARAMETER_TYPES
from xms.cmsflow.gui.rubble_mound_dlg_ui import Ui_RubbleMoundDlg
from xms.cmsflow.gui.text_converter import TextConverter


class RubbleMoundDlg(XmsDlg):
    """A dialog for viewing a single rubble mound jetty's data."""
    def __init__(self, rm_structures_data, multi_msg, pe_tree, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            rm_structures_data (RMStructuresData): The rubble mound data to view
            multi_msg (str): Warning/multi-select message, if any
            pe_tree (TreeNode): The project explorer tree
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.cmsflow.gui.rubble_mound_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/CMS-Flow/RubbleMounds'
        self.multi_msg = multi_msg
        self.ui = Ui_RubbleMoundDlg()
        self.ui.setupUi(self)

        self.parameter_types = PARAMETER_TYPES
        self.calc_methods = CALCULATION_METHODS
        self.pe_tree = pe_tree
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.porosity_dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_validator.setDecimals(10)
        self.dbl_validator.setBottom(0.0)
        self.porosity_dbl_validator.setDecimals(10)
        self.porosity_dbl_validator.setBottom(0.0)
        self.porosity_dbl_validator.setTop(1.0)

        self.ui.porosity.setValidator(self.porosity_dbl_validator)
        self.ui.rock_diameter.setValidator(self.dbl_validator)
        self.ui.base_depth.setValidator(self.dbl_validator)

        self.ui.calculation_method.addItems(self.calc_methods)
        self.ui.cbx_base_depth_type.addItems(self.parameter_types)
        self.ui.cbx_porosity_type.addItems(self.parameter_types)
        self.ui.cbx_rock_diameter_type.addItems(self.parameter_types)

        self.rm_structures_data = rm_structures_data
        self.rock_diameter_uuid = '(none selected)'
        self.porosity_uuid = '(none selected)'
        self.base_depth_uuid = '(none selected)'
        self._add_multi_select_warning()
        self._connect_slots()
        self._load_data()

    def _connect_slots(self):
        """Connect slot/signals for various widgets."""
        self.ui.btn_box.helpRequested.connect(self.help_requested)
        self.ui.cbx_rock_diameter_type.currentIndexChanged.connect(self.rock_diameter_type_changed)
        self.ui.cbx_porosity_type.currentIndexChanged.connect(self.porosity_type_changed)
        self.ui.cbx_base_depth_type.currentIndexChanged.connect(self.base_depth_type_changed)
        self.ui.btn_base_depth_dset.clicked.connect(self.select_base_depth)
        self.ui.btn_porosity_dset.clicked.connect(self.select_porosity)
        self.ui.btn_rock_diameter_dset.clicked.connect(self.select_rock_diameter)

    @staticmethod
    def _parameter_type_changed(idx, constant, constant_lbl, dataset, dataset_lbl):
        """This method is called when the parameter type selection is changed in the RubbleMoundDlg class.

        Args:
            idx (int): The index of the selected parameter type
            constant (QLineEdit): The QLineEdit widget for the constant value
            constant_lbl (QLabel): The QLabel widget for the constant value label
            dataset (QComboBox): The QComboBox widget for the dataset selection
            dataset_lbl (QLabel): The QLabel widget for the dataset selection label
        """
        if idx == PARAMETER_TYPE_CONSTANT_IDX:
            constant.setVisible(True)
            constant_lbl.setVisible(True)
            dataset.setVisible(False)
            dataset_lbl.setVisible(False)
        else:  # idx == PARAMETER_TYPE_DATASET_IDX
            constant.setVisible(False)
            constant_lbl.setVisible(False)
            dataset.setVisible(True)
            dataset_lbl.setVisible(True)

    def _add_multi_select_warning(self):
        """Add a label for the multi-select warning, if applicable."""
        if self.multi_msg:
            label = QLabel(self.multi_msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.layout().insertWidget(0, label)

    def _initialize_state(self):
        """Initialize the state of the widgets."""
        self.rock_diameter_type_changed(self.ui.cbx_rock_diameter_type.currentIndex())
        self.base_depth_type_changed(self.ui.cbx_base_depth_type.currentIndex())
        self.porosity_type_changed(self.ui.cbx_porosity_type.currentIndex())

    def _load_data(self):
        """Loads the rubble mound structures data."""
        if not self.rm_structures_data.name.size == 0:
            self.ui.name.setText(TextConverter.get_text_for_field(self.rm_structures_data['name'].item()))
            self.ui.rock_diameter.setText(str(self.rm_structures_data['rock_diameter'].item()))
            self.ui.cbx_rock_diameter_type.setCurrentIndex(self.rm_structures_data['rock_diameter_type'].item())
            self.ui.porosity.setText(str(self.rm_structures_data['porosity'].item()))
            self.ui.cbx_porosity_type.setCurrentIndex(self.rm_structures_data['porosity_type'].item())
            self.ui.base_depth.setText(str(self.rm_structures_data['base_depth'].item()))
            self.ui.cbx_base_depth_type.setCurrentIndex(self.rm_structures_data['base_depth_type'].item())
            self.ui.calculation_method.setCurrentIndex(
                self.calc_methods.index(self.rm_structures_data['calculation_method'].item())
            )
            self.base_depth_uuid = self.rm_structures_data['STRUCTURE_BASE_DEPTH_DATASET'].item()
            tree_path = tree_util.build_tree_path(self.pe_tree, self.base_depth_uuid)
            self.ui.lbl_base_depth_dset.setText(tree_path if tree_path else '(none selected)')
            self.porosity_uuid = self.rm_structures_data['STRUCTURE_POROSITY_DATASET'].item()
            tree_path = tree_util.build_tree_path(self.pe_tree, self.porosity_uuid)
            self.ui.lbl_porosity_dset.setText(tree_path if tree_path else '(none selected)')
            self.rock_diameter_uuid = self.rm_structures_data['ROCK_DIAMETER_DATASET'].item()
            tree_path = tree_util.build_tree_path(self.pe_tree, self.rock_diameter_uuid)
            self.ui.lbl_rock_diameter_dset.setText(tree_path if tree_path else '(none selected)')
            self._initialize_state()

    def _save_rm_structures_data(self):
        """Save data from the rubble mound jetty attributes."""
        self.rm_structures_data['name'] = TextConverter.get_text_from_field(self.ui.name.text())
        self.rm_structures_data['rock_diameter'] = float(self.ui.rock_diameter.text())
        self.rm_structures_data['rock_diameter_type'] = self.ui.cbx_rock_diameter_type.currentIndex()
        self.rm_structures_data['porosity'] = float(self.ui.porosity.text())
        self.rm_structures_data['porosity_type'] = self.ui.cbx_porosity_type.currentIndex()
        self.rm_structures_data['base_depth'] = float(self.ui.base_depth.text())
        self.rm_structures_data['base_depth_type'] = self.ui.cbx_base_depth_type.currentIndex()
        self.rm_structures_data['calculation_method'] = self.ui.calculation_method.currentText()

        self.rm_structures_data['ROCK_DIAMETER_DATASET'] = self.rock_diameter_uuid
        self.rm_structures_data['STRUCTURE_POROSITY_DATASET'] = self.porosity_uuid
        self.rm_structures_data['STRUCTURE_BASE_DEPTH_DATASET'] = self.base_depth_uuid

    def _select_polygon_dataset(self, dlg_title, label_widget, previous_uuid):
        """Display a dataset picker and store the new selection if one is made.

        Args:
            dlg_title (:obj:`str`): Title of the dataset selector dialog
            label_widget (:obj:`QLabel`): The label associated with the dataset selector push button
            previous_uuid (str): UUID of the previous selection

        Returns:
            str: The selected dataset's UUID or None if not selected
        """
        tree_copy = tree_util.copy_tree(tree_node=self.pe_tree)  # Create copy of the project explorer tree to filter.
        if tree_copy:
            tree_util.filter_project_explorer(tree_copy, DatasetSelector.is_scalar_and_steady_state_if_dset)
        if not tree_copy.children:  # Have a geometry but no selectable datasets
            msg = QMessageBox(
                QMessageBox.Information, 'SMS', 'No selectable datasets found. Rubble mound datasets '
                'must be scalar and steady state.', QMessageBox.Ok, self
            )
            msg.exec()
            return

        dlg = TreeItemSelectorDlg(
            title=dlg_title,
            target_type=xmd.DatasetItem,
            pe_tree=tree_copy,
            previous_selection=previous_uuid,
            show_root=True,
            parent=self
        )
        dset_uuid = None
        if dlg.exec():
            dset_uuid = dlg.get_selected_item_uuid()
            if dset_uuid:
                dset_path = tree_util.build_tree_path(tree_copy, dset_uuid)
            else:
                dset_path = '(none selected)'
            label_widget.setText(dset_path)
        return dset_uuid

    def rock_diameter_type_changed(self, idx):
        """Called when the rock diameter type changes.

        Args:
            idx (int): The type combobox's current index
        """
        self._parameter_type_changed(
            idx, self.ui.rock_diameter, self.ui.rock_diameter_label, self.ui.btn_rock_diameter_dset,
            self.ui.lbl_rock_diameter_dset
        )

    def porosity_type_changed(self, idx):
        """Called when the porosity type changes.

        Args:
            idx (int): The type combobox's current index
        """
        self._parameter_type_changed(
            idx, self.ui.porosity, self.ui.porosity_label, self.ui.btn_porosity_dset, self.ui.lbl_porosity_dset
        )

    def base_depth_type_changed(self, idx):
        """Called when the base depth type changes.

        Args:
            idx (int): The type combobox's current index
        """
        self._parameter_type_changed(
            idx, self.ui.base_depth, self.ui.base_depth_label, self.ui.btn_base_depth_dset, self.ui.lbl_base_depth_dset
        )

    def select_base_depth(self):
        """Select a base depth dataset."""
        dset_uuid = self._select_polygon_dataset(
            dlg_title='Select a Base Depth Dataset',
            previous_uuid=self.base_depth_uuid,
            label_widget=self.ui.lbl_base_depth_dset
        )
        if dset_uuid:
            self.base_depth_uuid = dset_uuid

    def select_porosity(self):
        """Select a porosity dataset."""
        dset_uuid = self._select_polygon_dataset(
            dlg_title='Select a Porosity Dataset',
            previous_uuid=self.porosity_uuid,
            label_widget=self.ui.lbl_porosity_dset
        )
        if dset_uuid:
            self.porosity_uuid = dset_uuid

    def select_rock_diameter(self):
        """Select a rock diameter dataset."""
        dset_uuid = self._select_polygon_dataset(
            dlg_title='Select a Rock Diameter Dataset',
            previous_uuid=self.rock_diameter_uuid,
            label_widget=self.ui.lbl_rock_diameter_dset
        )
        if dset_uuid:
            self.rock_diameter_uuid = dset_uuid

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_rm_structures_data()
        super().accept()
