"""Dialog for assigning Structures coverage properties."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Qt
from PySide2.QtWidgets import (
    QAbstractItemView, QDialogButtonBox, QHBoxLayout, QHeaderView, QStyle, QToolBar, QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.delegates.texture_button_delegate import TextureButtonDelegate
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.resources.resources_util import get_resource_path
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.guipy.widgets.widget_builder import setup_toolbar

# 4. Local modules
from xms.cmswave.data.structures_data import StructuresData
from xms.cmswave.gui.structures_filter_model import StructuresFilterModel
from xms.cmswave.gui.structures_mod_display_delegate import ModDisplayDelegate


class StructuresDialog(XmsDlg):
    """A dialog to define structures and their properties."""
    def __init__(self, title, parent, structure_data):
        """Initializes the structure list and properties dialog.

        Args:
            title (:obj:`str`): Title of the dialog
            parent (:obj:`QWidget`): Parent Qt dialog
            structure_data (StructuresData): The structure data to edit
        """
        super().__init__(parent, 'xms.cmswave.gui.structures_dialog')
        self.parent = parent
        self.help_url = 'https://cirpwiki.info/wiki/CMS-Wave/Structures_v2'
        self.widgets = {}
        self.structures_model = None  # QxPandasTableModel for structures spreadsheet
        self.actions = None  # Disable delete of unassigned structure
        self.structure_data = structure_data
        self.struct_att_structure_id = -1  # init to -1 so it will be set to UNASSIGNED when the dialog starts
        self.deleted_structure_ids = set()
        self.change_all_textures = PolygonOptions()
        self.delegates = dict()
        self.delegates['structure_display'] = TextureButtonDelegate(self)
        self.delegates['structure_type'] = QxCbxDelegate(self)
        strings = [
            'Bathymetry modification', 'Floating breakwater', 'Highly permeable', 'Semi-permeable', 'Rubble-mound',
            'Wall breakwater', 'Wave runup'
        ]
        self.delegates['structure_type'].set_strings(strings)
        self.delegates['structure_type'].state_changed.connect(self.on_combo_changed)
        self.delegates['structure_use_mod'] = CheckBoxNoTextDelegate(self)
        self.delegates['structure_use_mod'].state_changed.connect(self.on_toggle_changed)
        self.delegates['structure_mod_type'] = ModDisplayDelegate(self)
        self.delegates['structure_mod_val'] = EditFieldValidator(self)
        self.add_icon = get_resource_path(':/resources/icons/add.svg')
        self.delete_icon = get_resource_path(':/resources/icons/delete.svg')
        self.change_all_icon = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'change-all-struct.svg'
        )

        # Setup the dialog
        self.setWindowTitle(title)
        self.setup_ui()
        self.adjustSize()

        # Calculate a reasonable width for the dialog.
        self.dlg_width = self.widgets['table_view'].horizontalHeader().length()
        self.dlg_width += self.widgets['table_view'].style().pixelMetric(QStyle.PM_ScrollBarExtent)
        self.dlg_width += self.widgets['table_view'].frameWidth() * 2
        self.dlg_width += 50
        self.resize(self.dlg_width, self.size().height())

    def setup_ui(self):
        """Setup dialog widgets."""
        # Add a main vertical layout.
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])

        # set up the structures table view
        self._setup_ui_structures_view()

        # add buttons to add, delete, change all textures for structures
        self._setup_ui_structures_buttons()

        # Connect a slots to the structure list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        selection_model.selectionChanged.connect(self.on_structure_changed)
        if self.parent.selected_structure is not None:  # Assign Structures command
            current_row = self.parent.selected_structure
        else:  # Structures List and Properties command
            current_row = StructuresData.UNASSIGNED_STRUCT  # Select the first row
        current_index = self.widgets['table_view'].model().index(current_row, StructuresData.COL_NAME)
        selection_model.setCurrentIndex(current_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)

        # add the ok, cancel, help... buttons at the bottom of the dialog
        self._setup_ui_bottom_button_box()

    def _setup_ui_structures_view(self):
        i = 0
        """Sets up the table veiw for the structures."""
        self.widgets['table_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['table_horiz_layout'])

        # Add the structure list table view.
        self.widgets['table_view'] = QxTableView()
        df = self.structure_data.structures.to_dataframe()
        if not df.index.empty and df.index[0] == 0:
            df.index = df.index + 1  # Start index at 1, not 0
        self.structures_model = QxPandasTableModel(df)
        self.widgets['table_horiz_layout'].addWidget(self.widgets['table_view'])

        # Override filter model to enable editing of description.
        self.widgets['table_view'].filter_model = StructuresFilterModel(self)
        self.widgets['table_view'].filter_model.setSourceModel(self.structures_model)
        self.widgets['table_view'].setModel(self.widgets['table_view'].filter_model)

        # Set up the color and texture column to be a button.
        self.widgets['table_view'].setItemDelegateForColumn(
            StructuresData.COL_COLOR, self.delegates['structure_display']
        )

        # Hide the ID column.
        self.widgets['table_view'].setColumnHidden(StructuresData.COL_ID, True)

        # Set up the type column to be a combo box
        self.widgets['table_view'].setItemDelegateForColumn(
            StructuresData.COL_CBX_TYPE, self.delegates['structure_type']
        )

        # Set up the use mod column to be a toggle
        self.widgets['table_view'].setItemDelegateForColumn(
            StructuresData.COL_TOG_MOD, self.delegates['structure_use_mod']
        )

        # Set up the mod type column to display the text
        self.widgets['table_view'].setItemDelegateForColumn(
            StructuresData.COL_TXT_MOD, self.delegates['structure_mod_type']
        )

        self.widgets['table_view'].horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.widgets['table_view'].horizontalHeader().setStretchLastSection(True)
        self.widgets['table_view'].resizeColumnsToContents()
        self.structures_model.set_read_only_columns({StructuresData.COL_TXT_MOD})
        self.structures_model.set_checkbox_columns({StructuresData.COL_TOG_MOD})

        for i in range(1, len(df)):  # start at 1 to skip UNASSIGNED_STRUCT
            use_mod = df.iloc[i, StructuresData.COL_TOG_MOD]
            if use_mod == 0:
                self.structures_model.read_only_cells.add((i, StructuresData.COL_EDT_MOD))

        self.widgets['table_view'].setEditTriggers(QAbstractItemView.AllEditTriggers)

    def _setup_ui_structures_buttons(self):
        """Adds toolbar below the structures table view to add, delete, change all textures."""
        self.widgets['tool_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['tool_horiz_layout'])
        # Add the "+", "-", and change all buttons.
        self.widgets['toolbar'] = QToolBar()
        button_list = [
            [self.add_icon, 'Add Structure', self.on_btn_add_structure],
            [self.delete_icon, 'Delete Structures', self.on_btn_delete_structure],
            [self.change_all_icon, 'Change all structure textures', self.on_btn_change_all]
        ]
        self.actions = setup_toolbar(self.widgets['toolbar'], button_list)
        change_all_btn = self.widgets['toolbar'].widgetForAction(self.actions[self.change_all_icon])
        texture_icon = DisplayOptionIconFactory.get_icon(self.change_all_textures, change_all_btn.size().width())
        change_all_btn.setIcon(texture_icon)
        self.widgets['tool_horiz_layout'].addWidget(self.widgets['toolbar'])

    def _setup_ui_bottom_button_box(self):
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])
        self.widgets['main_vert_layout'].addLayout(self.widgets['btn_horiz_layout'])

    def _get_new_structure_name(self):
        """Get a unique name for a new structure.

        Returns:
            (:obj:`str`): See description
        """
        struct_name = 'new structure'
        unique = False
        i = 1
        while not unique:
            unique = True
            for row in self.structures_model.data_frame['name']:
                if str(row) == struct_name:
                    unique = False
                    struct_name = f'new structure ({i})'
                    i += 1
                    break
        return struct_name

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def on_btn_change_all(self):
        """Called when the change all structures texture/style button is clicked."""
        dlg = PolygonDisplayOptionsDialog(self.change_all_textures, self)
        dlg.ui.lab_line_color.setVisible(False)
        dlg.ui.color_btn.setVisible(False)
        if dlg and dlg.exec():
            self.change_all_textures = dlg.get_options()
            new_opts = self.delegates['structure_display'].poly_opts_to_str(self.change_all_textures)
            texture = new_opts.split()[-1]
            for row in range(self.structures_model.rowCount()):
                data = self.structures_model.data(
                    self.structures_model.index(row, StructuresData.COL_COLOR), Qt.EditRole
                )
                lst = data.split()
                data = f'{lst[0]} {lst[1]} {lst[2]} {texture}'
                self.structures_model.setData(
                    self.structures_model.index(row, StructuresData.COL_COLOR), data, Qt.EditRole
                )

    def on_btn_add_structure(self):
        """Called when the add structure button is clicked."""
        struct_id = max(self.structures_model.data_frame['id']) + 1
        description = self._get_new_structure_name()
        default_data = self.structure_data.default_data_dict()
        default_data['id'] = struct_id
        default_data['name'] = description
        self.structures_model.set_default_values(default_data)
        row_idx = self.structures_model.rowCount()
        self.structures_model.insertRows(row_idx, 1)

        # make the edit field read only since by default the mod is off
        self.structures_model.read_only_cells.add((row_idx, StructuresData.COL_EDT_MOD))

        # Connect a slots to the structure list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        new_struct_index = self.widgets['table_view'].model().index(row_idx, StructuresData.COL_NAME)
        selection_model.setCurrentIndex(new_struct_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)
        self.widgets['table_view'].resizeColumnsToContents()
        self.widgets['table_view'].filter_model.headerDataChanged.emit(Qt.Horizontal, 0, 0)

    def on_btn_delete_structure(self):
        """Called when the delete structure button is clicked."""
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        rows = {index.row(): index for index in indices}

        for row in sorted(rows.keys(), reverse=True):
            index = rows[row]
            # Keep track of deleted structure ids so we can reassign polygons in XMS.
            delete_id_index = self.structures_model.index(index.row(), StructuresData.COL_ID)
            struct_id = int(float(self.structures_model.data(delete_id_index, Qt.DisplayRole)))
            self.deleted_structure_ids.add(struct_id)
            self.structures_model.removeRows(index.row(), 1)

        # select the row above the first currently selected row
        row = sorted(rows.keys())[0] - 1
        select_index = self.widgets['table_view'].model().index(row, StructuresData.COL_NAME)
        self.widgets['table_view'].selectionModel().setCurrentIndex(
            select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
        )

    def on_structure_changed(self):
        """Called when the structure row selection changes."""
        # Don't allow the user to delete all structures
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        enable_delete = len(indices) > 1
        self.widgets['toolbar'].widgetForAction(self.actions[self.delete_icon]).setEnabled(enable_delete)

    def on_toggle_changed(self, index):
        """Called when the structure type combo is changed.

        Args:
            index (:obj:`QModelIndex`): Model index of the toggle whose state changed
        """
        if not index.isValid():
            return
        row = index.row()
        col = index.column()

        checked = index.data(Qt.CheckStateRole)
        update_indices = []
        table_view = self.widgets['table_view']
        if col == StructuresData.COL_TOG_MOD:
            if checked:
                self.structures_model.read_only_cells.discard((row, StructuresData.COL_EDT_MOD))
            else:
                self.structures_model.read_only_cells.add((row, StructuresData.COL_EDT_MOD))
            update_indices.append(table_view.filter_model.index(row, StructuresData.COL_EDT_MOD))
            update_indices.append(table_view.filter_model.index(row, StructuresData.COL_TXT_MOD))
        for update_idx in update_indices:
            table_view.update(update_idx)

    def on_combo_changed(self, index):
        """Called when the use default checkbox is toggled for a row.

        Args:
            index (:obj:`QModelIndex`): Model index of the toggle whose state changed
        """
        if not index.isValid():
            return
        row = index.row()

        update_indices = []
        table_view = self.widgets['table_view']
        update_indices.append(table_view.filter_model.index(row, StructuresData.COL_EDT_MOD))
        update_indices.append(table_view.filter_model.index(row, StructuresData.COL_TXT_MOD))

        for update_idx in update_indices:
            table_view.update(update_idx)

    def _export_structures_to_file(self, filename):
        """Writes structures to a file.

        Args:
            filename (:obj:`str`): file name
        """
        struct_data = StructuresData(filename)
        struct_data.structures = self.structures_model.data_frame.to_xarray()
        if os.path.exists(filename) and os.path.isfile(filename):
            os.remove(filename)
        struct_data.commit()

    def accept(self):
        """Save structure properties and block accepted signal if structure names are no longer unique."""
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        struct_names = self.structures_model.data_frame['name'].to_list()
        set_names = set()
        for name in struct_names:
            num_names = len(set_names)
            set_names.add(name)
            if num_names == len(set_names):
                msg = f'Structure: "{name}" is not a unique name.\n' \
                      f'Structure names must be unique. Edit the names to ensure that they are unique.'
                message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
                return

        # If we came from the Assign Structure command, make sure we have a single structure row selected.
        if self.parent.selected_structure is not None:
            indices = self.widgets['table_view'].selectionModel().selectedIndexes()
            selected_rows = {index.row() for index in indices if index.isValid()}
            if len(selected_rows) != 1:
                msg = 'Please select a single structure row to update the assignment of the selected polygon(s).'
                message_box.message_with_ok(
                    parent=self, message=msg, app_name=app_name, icon='Critical', win_icon=self.windowIcon()
                )
                return
            self.parent.selected_structure = indices[0].row()

        # Save the structure list and global attributes.
        self.structure_data.structures = self.structures_model.data_frame.to_xarray()
        if self.parent is not None:
            self.parent.accept()
        else:  # pragma: no cover
            super().accept()

    def reject(self):
        """Call reject slot on parent if we have one."""
        if self.parent is not None:
            self.parent.reject()
        else:  # pragma: no cover
            super().reject()
