"""Read files used to store CMS-Wave simulations as part of old SMS projects."""

# 1. Standard Python modules
import datetime
import os
import uuid

# 2. Third party modules
import h5py
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.constraint.rectilinear_geometry import Numbering, Orientation
from xms.constraint.rectilinear_grid_builder import RectilinearGridBuilder
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import (
    Component, Coverage, datetime_to_julian, julian_to_datetime, Point, RectilinearGrid, Simulation, UGrid
)
from xms.datasets.dataset_reader import DatasetReader
from xms.guipy.time_format import datetime_to_string

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const
from xms.cmswave.data import simulation_data
from xms.cmswave.data.simulation_data import SimulationData


class XmsReader:
    """A class for reading CMS-Wave simulations from files that are part of old SMS projects."""
    def __init__(self):
        """Constructor."""
        self.query = None
        self.grid_name = None
        self.nest_pts = []
        self.monitor_pts = []
        self.structure_pts = []
        self.using_darcy = False
        self.query = Query()
        self.dset_paths_to_widget = {}
        self.in_filename = ''
        self.sim_uuid = ''
        self.data = None
        self.free_dsets = {}

        self.cell_zs = []  # Store the cell elevations for testing

        self._link_items = []

    def read(self):
        """Reads the file, builds grid, coverage, loads data and sends it all to SMS."""
        in_proj_filename = self.query.read_file

        # Get the SMS temp directory.
        comp_dir = os.path.join(self.query.xms_temp_directory, 'Components')
        os.makedirs(comp_dir, exist_ok=True)

        with h5py.File(in_proj_filename, 'r') as f1:
            grid_names = list(f1["Cart2DModule"].keys())

        # Key is simulation UUID
        sim_data = {}
        grids = {}

        # add each simulation we will need
        for grid_name in grid_names:
            sim = Simulation(model='CMS-Wave', name=grid_name, sim_uuid=str(uuid.uuid4()))
            comp_uuid = str(uuid.uuid4())
            sim_comp_dir = os.path.join(comp_dir, comp_uuid)
            os.makedirs(sim_comp_dir, exist_ok=True)
            sim_mainfile = os.path.join(sim_comp_dir, 'simulation_comp.nc')
            comp = Component(
                comp_uuid=comp_uuid, main_file=sim_mainfile, model_name='CMS-Wave', unique_name='Sim_Component'
            )
            self.data = SimulationData(sim_mainfile)  # This will create a default model control

            # Copy the H5 file so we can feed it to data_objects RectilinearGrid reader code.
            self.in_filename = os.path.join(
                self.query.xms_temp_directory, f'{grid_name}_{os.path.basename(in_proj_filename)}'
            )
            io_util.copyfile(in_proj_filename, self.in_filename)

            self.sim_uuid = sim.uuid
            self.free_dsets[self.sim_uuid] = []
            grid_path = "Cart2DModule/" + grid_name + "/"

            base_path = grid_path + 'PROPERTIES/Model Params/'
            dset_path = grid_path + 'Datasets'

            # Set the contents of the Parameters tab of the model control
            # set PLANE_DEFINITION (iview)
            # set the boundary conditions while we are at it
            # side 1 is a spectra and side 3 comes from iview
            self.data.info.attrs['side1'] = 'Specified spectrum'
            self.data.info.attrs['side3'] = 'Zero spectrum'
            with h5py.File(in_proj_filename, 'r+') as f:
                if f.__contains__(base_path + 'iPlaneType'):
                    model_type = int(f[base_path + 'iPlaneType'][0])
                    option = 'Full plane'
                    if model_type == 0:
                        option = 'Half plane'
                    elif model_type == 2:
                        option = 'Full plane with input reverse spectra'
                        self.data.info.attrs['side3'] = 'Specified spectrum'
                    self.data.info.attrs['plane'] = option

                # set CURRENT_TYPE (icur)
                current_option = const.TEXT_NONE
                if f.__contains__(base_path + 'iCurrents'):
                    current_type = int(f[base_path + "iCurrents"][0])
                    # if the current type is dataset, make sure it is there
                    if current_type == 1:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'CurrentsFunc'):
                            dset_name = f[base_path + 'CurrentsFunc'][0].decode('UTF-8')
                            current_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(current_dset_path):
                                current_option = const.TEXT_USE_DATASET
                self.data.info.attrs['current_interaction'] = current_option

                # set BOTTOM_FRICTION (ibf)
                option = const.TEXT_NONE
                self.using_darcy = False
                if f.__contains__(base_path + 'iBedFriction'):
                    friction_type = int(f[base_path + "iBedFriction"][0])
                    if friction_type == 3:
                        option = const.TEXT_DARCY_CONSTANT
                        self.using_darcy = True
                    elif friction_type == 4:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'BedFrictionFunc'):
                            dset_name = f[base_path + 'BedFrictionFunc'][0].decode('UTF-8')
                            friction_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(friction_dset_path):
                                option = const.TEXT_DARCY_DATASET
                                self.using_darcy = True
                    elif friction_type == 5:
                        option = const.TEXT_MANNINGS_CONSTANT
                    elif friction_type == 6:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'BedFrictionFunc'):
                            dset_name = f[base_path + 'BedFrictionFunc'][0].decode('UTF-8')
                            friction_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(friction_dset_path):
                                option = const.TEXT_MANNINGS_DATASET
                self.data.info.attrs['friction'] = option
                # set the fric_val for constant friction
                darcy_value = 0.003
                manning_value = 0.025
                if f.__contains__(base_path + "dBedFriction"):
                    if self.using_darcy:
                        darcy_value = float(f[base_path + "dBedFriction"][0])
                    else:
                        manning_value = float(f[base_path + "dBedFriction"][0])
                self.data.info.attrs['darcy'] = darcy_value
                self.data.info.attrs['manning'] = manning_value

                # set the water level type (iSurge) - no variable in the std file
                surge_option = const.TEXT_NONE
                if f.__contains__(base_path + 'iSurge'):
                    surge_type = int(f[base_path + "iSurge"][0])
                    # if the surge type is constant (specified for each case), make sure it is there
                    if surge_type == 1:
                        surge_option = const.TEXT_CONSTANT
                    # else if the surge type is dataset, make sure it is there
                    elif surge_type == 2:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'SurgeFunc'):
                            func_name = f[base_path + 'SurgeFunc'][0].decode('UTF-8')
                            surge_dset_path = dset_path + '/Input Datasets/' + func_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(surge_dset_path):
                                surge_option = const.TEXT_USE_DATASET
                self.data.info.attrs['surge'] = surge_option

                # set the ENABLE_WIND (iwnd) - (also sets PROPAGATION_TYPE (iprp))
                wind_option = const.TEXT_NONE
                source_option = 'Propagation only'
                if f.__contains__(base_path + 'Wind'):
                    wind_type = int(f[base_path + "Wind"][0])
                    # if the wind type is constant (specified for each case), make sure it is there
                    if wind_type == 1:
                        wind_option = const.TEXT_CONSTANT
                        source_option = 'Source terms and propagation'
                    # else if the wind type is dataset, make sure it is there
                    elif wind_type == 2:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'WindFieldFunc'):
                            dset_name = f[base_path + 'WindFieldFunc'][0].decode('UTF-8')
                            wind_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(wind_dset_path):
                                wind_option = const.TEXT_USE_DATASET
                                source_option = 'Source terms and propagation'
                self.data.info.attrs['wind'] = wind_option
                self.data.info.attrs['source terms'] = source_option

                # set MATRIX_SOLVER (isolv)
                option = const.TEXT_GAUSS
                if f.__contains__(base_path + 'MatrixSolver'):
                    solver_type = int(f[base_path + 'MatrixSolver'][0])
                    if solver_type == 1:
                        option = const.TEXT_ADI
                self.data.info.attrs['matrix_solver'] = option

                # set NUM_THREADS (iproc)
                num_threads = 1
                if f.__contains__(base_path + 'Processes'):
                    num_threads = int(f[base_path + 'Processes'][0])
                self.data.info.attrs['num_threads'] = num_threads

                # Set the contents of the Boundary control tab of the model control
                # set spectral input
                option = 'Spectra (+ Wind)'
                if f.__contains__(base_path + 'iSpectra'):
                    boundary_type = int(f[base_path + 'iSpectra'][0])
                    if boundary_type == 1:
                        option = const.TEXT_NONE
                self.data.info.attrs['boundary_source'] = option

                # set the spectral interpolation method
                option = 'IDW'
                if f.__contains__(base_path + 'iInterpType'):
                    interpolation_type = int(f[base_path + 'iInterpType'][0])
                    if interpolation_type == 1:
                        option = 'Average Spectra'
                self.data.info.attrs['interpolation'] = option

                # set the computational spectral parameters (num frequencies, delta, minimum)
                min_frequency = 0.04
                max_frequency = 0.33
                delta = 0.01
                if f.__contains__(base_path + 'Spectral Grid/MinFreq'):
                    min_frequency = float(f[base_path + 'Spectral Grid/MinFreq'][0])
                if f.__contains__(base_path + 'Spectral Grid/MaxFreq'):
                    max_frequency = float(f[base_path + 'Spectral Grid/MaxFreq'][0])
                if f.__contains__(base_path + 'Spectral Grid/DeltaFreq'):
                    delta = float(f[base_path + 'Spectral Grid/DeltaFreq'][0])
                num_frequencies = (max_frequency - min_frequency) / delta + 1
                min_frequency = np.around(min_frequency, 3)
                delta = np.around(delta, 3)
                num_frequencies = np.around(num_frequencies, 0)
                self.data.info.attrs['min_frequency'] = min_frequency
                self.data.info.attrs['delta_frequency'] = delta
                self.data.info.attrs['num_frequencies'] = num_frequencies

                # set 1st spectral coverage (side 1)
                if f.__contains__(base_path + 'SpectralCov1Guid'):
                    cov_guid = f[base_path + 'SpectralCov1Guid'][0].decode('UTF-8')
                    self.data.info.attrs['spectral_uuid'] = cov_guid
                # set 2nd spectral coverage (side 3)
                if f.__contains__(base_path + 'SpectralCov3Guid'):
                    cov_guid = f[base_path + 'SpectralCov3Guid'][0].decode('UTF-8')
                    self.data.info.attrs['spectral2_uuid'] = cov_guid

                # set time units
                time_units = 'hours'
                if f.__contains__(base_path + 'TimeUnits'):
                    time_units = f[base_path + 'TimeUnits'][0].decode('UTF-8')
                self.data.info.attrs['reftime_units'] = time_units

                # set reference time
                reftime = datetime.datetime.now()
                if f.__contains__(base_path + 'RefTime'):
                    julian_time = float(f[base_path + 'RefTime'][0])
                    reftime = julian_to_datetime(julian_time)
                self.data.info.attrs['reftime'] = datetime_to_string(reftime)
                julian_time = datetime_to_julian(reftime)

                # set cases
                if f.__contains__(base_path + 'SpectralCases'):
                    num_cases = int(f[base_path + 'SpectralCases'][0])
                    if num_cases != 0:
                        time_unit_multiplier = 1.0  # days
                        if time_units == 'hours':
                            time_unit_multiplier = 24.0
                        elif time_units == 'minutes':
                            time_unit_multiplier = 1440.0
                        # read the case times
                        case_times = f[base_path + 'TimeVec'][:]
                        # subtract the reference time from the case times to get offsets
                        case_times = case_times - julian_time
                        # scale the case_times by the unit multiplier
                        case_times = case_times * time_unit_multiplier
                        # round off to get rid of excess precision
                        case_times = np.around(case_times, 3)
                        # read the wind magnitude
                        wind_speed = f[base_path + 'WindSpdVec'][:]
                        wind_dir = f[base_path + 'WindDirVec'][:]
                        surge = f[base_path + 'TideVec'][:]
                        case_time_data = simulation_data.case_data_table(case_times, wind_dir, wind_speed, surge)
                        self.data.case_times = xr.Dataset(data_vars=case_time_data)

                # get the names of the datasets referenced by the simulation
                if f.__contains__(base_path + 'CurrentsFunc'):
                    path_to_dset = 'Input Datasets/' + f[base_path + 'CurrentsFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'current_uuid'
                if f.__contains__(base_path + 'BedFrictionFunc'):  # Darcy-Weisbach or Manning's N
                    path_to_dset = 'Input Datasets/' + f[base_path + 'BedFrictionFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'Friction'
                if f.__contains__(base_path + 'SurgeFunc'):
                    path_to_dset = 'Input Datasets/' + f[base_path + "SurgeFunc"][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'surge_uuid'
                if f.__contains__(base_path + 'WindFieldFunc'):
                    path_to_dset = 'Input Datasets/' + f[base_path + 'WindFieldFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'wind_uuid'
                if f.__contains__(base_path + 'ForwardReflectionFunc'):
                    path_to_dset = 'Input Datasets/' + f[base_path + 'ForwardReflectionFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'forward_reflection_uuid'
                if f.__contains__(base_path + "BackwardReflectionFunc"):
                    path_to_dset = 'Input Datasets/' + f[base_path + 'BackwardReflectionFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'backward_reflection_uuid'
                if f.__contains__(base_path + "MuddyBedFunc"):
                    path_to_dset = 'Input Datasets/' + f[base_path + 'MuddyBedFunc'][0].decode('UTF-8')
                    self.dset_paths_to_widget[path_to_dset] = 'muddy_bed_uuid'

                # Read all datasets of the grid, even if they aren't referenced by a simulation
                f[dset_path].visititems(self.visit_dsets_func)

                # Set the contents of the Output control tab of the model control
                # set the RAD_STRESS_OUTPUT (irs)
                radiation_stress_option = 0  # 0 is Off
                if f.__contains__(base_path + 'RadStresses'):
                    radiation_stress_option = int(f[base_path + 'RadStresses'][0])
                self.data.info.attrs['rad_stress'] = radiation_stress_option

                # Note: the Sea/swell output is part of wet/dry. It is set below

                # set BREAKING_OUTPUT (ibk)
                option = const.TEXT_NONE
                if f.__contains__(base_path + 'iBreaking'):
                    break_type = int(f[base_path + 'iBreaking'][0])
                    if break_type == 1:
                        option = const.TEXT_DISSIPATION_INDICES
                    elif break_type == 2:
                        option = const.TEXT_DISSIPATION_VALUES
                self.data.info.attrs['breaking_type'] = option

                # Set the contents of the Options tab of the model control
                # set wetting/drying (iwet) (also sets the sea/swell output option)
                wet_dry_option = 1
                sea_swell_option = 0
                if f.__contains__(base_path + 'WettingAndDrying'):
                    flag = int(f[base_path + 'WettingAndDrying'][0])  # -1 both on, 1 both off, 0 wet/dry on swell off
                    if flag == 1:
                        wet_dry_option = 0
                    if flag == -1:
                        sea_swell_option = 1
                self.data.info.attrs['wet_dry'] = wet_dry_option
                self.data.info.attrs['sea_swell'] = sea_swell_option

                # set infra-gravity
                option = 0
                if f.__contains__(base_path + 'InfraGravWave'):
                    option = int(f[base_path + 'InfraGravWave'][0])
                self.data.info.attrs['infragravity'] = option

                # set diffraction and the intensity
                option = 0
                if f.__contains__(base_path + 'Diffraction'):
                    option = int(f[base_path + 'Diffraction'][0])
                self.data.info.attrs['diffraction_option'] = option
                diff_coefficient = 4.0
                if f.__contains__(base_path + 'dDiffCoef'):
                    diff_coefficient = float(f[base_path + 'dDiffCoeff'][0])
                self.data.info.attrs['diffraction_intensity'] = diff_coefficient

                # set non-linear wave effect
                option = 0
                if f.__contains__(base_path + 'NonLinearWave'):
                    option = int(f[base_path + 'NonLinearWave'][0])
                self.data.info.attrs['nonlinear_wave'] = option

                # set Run-up
                option = 0
                if f.__contains__(base_path + 'RunUp'):
                    option = int(f[base_path + 'RunUp'][0])
                self.data.info.attrs['runup'] = option

                # set Roller (not in 13.1)
                option = 0
                if f.__contains__(base_path + 'Roller'):
                    option = int(f[base_path + 'Roller'][0])
                self.data.info.attrs['roller'] = option

                # set Fast-mode (included in iprp which came from wind) - not working in 13.1
                option = 0
                if f.__contains__(base_path + 'fastmode'):
                    option = int(f[base_path + 'fastmode'][0])
                self.data.info.attrs['fastmode'] = option

                # set forward reflection (iark, ark)
                reflect_option = const.TEXT_NONE
                if f.__contains__(base_path + 'iForReflect'):
                    reflect_type = int(f[base_path + "iForReflect"][0])
                    # if the surge type is constant (specified for each case), make sure it is there
                    if reflect_type == 1:
                        reflect_option = const.TEXT_CONSTANT
                    # else if the reflect_option type is dataset, make sure it is there
                    elif reflect_type == 2:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'ForwardReflectionFunc'):
                            dset_name = f[base_path + 'ForwardReflectionFunc'][0].decode('UTF-8')
                            reflect_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(reflect_dset_path):
                                reflect_option = const.TEXT_USE_DATASET
                self.data.info.attrs['forward_reflection'] = reflect_option
                coefficient = 0.0
                if f.__contains__(base_path + 'dForReflect'):
                    coefficient = float(f[base_path + 'dForReflect'][0])
                self.data.info.attrs['forward_reflection_const'] = coefficient

                # set backward reflection (iarkr, arkr)
                reflect_option = const.TEXT_NONE
                if f.__contains__(base_path + 'iBackReflect'):
                    reflect_type = int(f[base_path + "iBackReflect"][0])
                    # if the surge type is constant (specified for each case), make sure it is there
                    if reflect_type == 1:
                        reflect_option = const.TEXT_CONSTANT
                    # else if the reflect_option type is dataset, make sure it is there
                    elif reflect_type == 2:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'BackwardReflectionFunc'):
                            dset_name = f[base_path + 'BackwardReflectionFunc'][0].decode('UTF-8')
                            reflect_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(reflect_dset_path):
                                reflect_option = const.TEXT_USE_DATASET
                self.data.info.attrs['backward_reflection'] = reflect_option
                coefficient = 0.0
                if f.__contains__(base_path + 'dBackReflect'):
                    coefficient = float(f[base_path + 'dBackReflect'][0])
                self.data.info.attrs['backward_reflection_const'] = coefficient

                # set muddy bed (imud)
                mud_option = const.TEXT_NONE
                if f.__contains__(base_path + 'MuddyBed'):
                    mud_type = int(f[base_path + "MuddyBed"][0])
                    # else if the muddy type is dataset, make sure it is there
                    if mud_type == 1:
                        # 1st, the name of the dataset must be specified
                        if f.__contains__(base_path + 'MuddyBedFunc'):
                            dset_name = f[base_path + 'MuddyBedFunc'][0].decode('UTF-8')
                            mud_dset_path = dset_path + '/Input Datasets/' + dset_name
                            # 2nd, the dataset must be in the file
                            if f.__contains__(mud_dset_path):
                                mud_option = const.TEXT_USE_DATASET
                self.data.info.attrs['muddy_bed'] = mud_option

                # set wave breaking (iwvbrk, gamma_BJ78) iBreaking???
                option = 'Extended Goda'
                if f.__contains__(base_path + 'iBreakingFunction'):
                    breaking_code = int(f[base_path + 'iBreakingFunction'][0])
                    if breaking_code == 1:
                        option = 'Extended Miche'
                    elif breaking_code == 2:
                        option = 'Battjes and Janssen 1978'
                    elif breaking_code == 3:
                        option = 'Chawla and Kirby'
                    elif breaking_code == 4:
                        option = 'Battjes and Janssen 2007'
                self.data.info.attrs['wave_breaking_formula'] = option
                self.data.info.attrs['gamma_bj78'] = 0.6

                # date formula - not supported in 13.1 output
                self.data.info.attrs['date_format'] = '12 digits'  # '8 digits'

                # Build the grid
                grid = RectilinearGrid(self.in_filename, grid_path)
                i_sizes = grid.i_sizes  # this makes us a SerialImpl
                j_sizes = grid.j_sizes
                # look for the depth values under /Depth/Values (13.0?)
                if f.__contains__(dset_path + '/Depth/Values'):
                    depth_values = f[dset_path + '/Depth/Values']
                    # At this point, we need to parse the SMS version from the project file. It may be a 13.1 project
                    # with a Z dataset named "Depth" that is actually elevation because you can't change the name of
                    # a CGrid Z dataset in 13.1 (but sometimes you can). We did not handle the depth <--> elevation
                    # transition gracefully, so there are a lot of old dirty projects out there in the wild.
                    sms_version = float(XmEnv.xms_environ_project_version())
                    if sms_version != 13.1:
                        cell_zs = depth_values[0, :] * -1.0
                    else:  # If this is a 13.1 project, assume the "Depth" values are elevations.
                        cell_zs = depth_values[0, :]
                # look for the depth values under /Elevation/Values (13.1)
                elif f.__contains__(dset_path + '/Elevation/Values'):
                    elev_values = f[dset_path + '/Elevation/Values']
                    cell_zs = elev_values[0, :]
                # we have a problem, no depth values - what to do, what to do
                else:
                    elev_values = np.zeros(len(i_sizes) + 1 * len(j_sizes) + 1)
                    cell_zs = elev_values[0, :]
                self.cell_zs = cell_zs.tolist()

                builder = RectilinearGridBuilder()
                builder.angle = grid.angle
                builder.origin = (grid.origin.x, grid.origin.y)
                builder.numbering = Numbering.kji
                builder.orientation = (Orientation.x_increase, Orientation.y_increase)
                builder.is_2d_grid = True
                builder.is_3d_grid = False

                locations_x = [0.0]
                locations_y = [0.0]
                offset = 0.0
                for i in i_sizes:
                    offset = offset + i
                    locations_x.append(offset)
                offset = 0.0
                for j in j_sizes:
                    offset = offset + j
                    locations_y.append(offset)
                builder.locations_x = locations_x
                builder.locations_y = locations_y
                rect_grid = builder.build_grid()
                rect_grid.cell_elevations = self.cell_zs  # already in elevations
                cogrid_file = os.path.join(self.query.process_temp_directory, 'cmswave_domain.xmc')
                rect_grid.write_to_file(cogrid_file, True)
                ugrid = UGrid(cogrid_file, name=grid_name)
                grid_projection = grid.projection
                ugrid.projection = grid_projection
                ugrid.uuid = str(uuid.uuid4())
                self._link_items.append((self.sim_uuid, ugrid.uuid))
                grids[self.sim_uuid] = ugrid

                sim_data[self.sim_uuid] = self.data, sim, comp, grid_projection

                # read the nesting cells for this simulation
                if f.__contains__(base_path + 'NestCells'):
                    xmugrid = rect_grid.ugrid
                    self.grid_name = grid_name
                    # Read the I,J cell indices for the nesting cells
                    i_vals = f[base_path + 'NestCells/I'][:].tolist()
                    j_vals = f[base_path + 'NestCells/J'][:].tolist()
                    # Get nesting points  (we have the grid)
                    pt_id = 1
                    for i, j in zip(i_vals, j_vals):
                        cell_idx = rect_grid.get_cell_index_from_ij(i, j)
                        location = xmugrid.get_cell_centroid(cell_idx=cell_idx)
                        if location and location[0] is True:
                            point = Point(location[1][0], location[1][1], location[1][2])
                            point.id = pt_id
                            pt_id += 1
                            self.nest_pts.append(point)

                # read the observation cells for this simulation
                if f.__contains__(base_path + 'ObsCells'):
                    xmugrid = rect_grid.ugrid
                    self.grid_name = grid_name
                    # Read the I,J cell indices for the nesting cells
                    i_vals = f[base_path + 'ObsCells/I'][:].tolist()
                    j_vals = f[base_path + 'ObsCells/J'][:].tolist()
                    # Get observation points  (we have the grid)
                    pt_id = 1
                    for i, j in zip(i_vals, j_vals):
                        cell_idx = rect_grid.get_cell_index_from_ij(i, j)
                        location = xmugrid.get_cell_centroid(cell_idx=cell_idx)
                        if location and location[0] is True:
                            point = Point(location[1][0], location[1][1], location[1][2])
                            point.id = pt_id
                            pt_id += 1
                            self.monitor_pts.append(point)

                # read the structure cells not supported for migration yet
                # would need to build the polygons and structure list (materials) - Lisa?
                # information in:   base_path + 'Structures' in data arrays I, J, Type, Val

        for sim_uuid, sim_tuple in sim_data.items():
            # add the simulation
            cur_sim_data = sim_tuple[0]
            sim = sim_tuple[1]
            comp = sim_tuple[2]
            grid_projection = sim_tuple[3]
            self.query.add_simulation(sim, [comp])

            # add the simulation's CGrid
            if sim_uuid in grids:
                self.query.add_ugrid(grids[sim_uuid])

            # add the datasets
            for h5_filename, group_path in self.free_dsets[sim_uuid]:
                dset = DatasetReader(h5_filename=h5_filename, group_path=group_path)
                self.query.add_dataset(dset)

            # add optional takes
            if self.nest_pts:
                nest_cov = Coverage()
                nest_cov.name = self.grid_name + ' Nesting Points'
                nest_cov.set_points(self.nest_pts)
                nest_cov.projection = grid_projection
                nest_cov.uuid = str(uuid.uuid4())
                nest_cov.complete()
                self.query.add_coverage(nest_cov)
                cur_sim_data.info.attrs['nesting'] = 1
                cur_sim_data.info.attrs['nesting_uuid'] = nest_cov.uuid
            if self.monitor_pts:
                monitor_cov = Coverage()
                monitor_cov.name = self.grid_name + ' Observation Cells'
                monitor_cov.set_points(self.monitor_pts)
                monitor_cov.projection = grid_projection
                monitor_cov.uuid = str(uuid.uuid4())
                monitor_cov.complete()
                self.query.add_coverage(monitor_cov)
                cur_sim_data.info.attrs['observation'] = 1
                cur_sim_data.info.attrs['observation_uuid'] = monitor_cov.uuid

            # Structures coverage??? Not yet

            cur_sim_data.commit()

        for link_items in self._link_items:
            self.query.link_item(link_items[0], link_items[1])

        self.query.send()  # send the data to SMS

    def visit_dsets_func(self, dset, item):
        """Walks the h5 file to detect all the datasets in the file.

        Args:
            dset (:obj:`str`): Name of the group or the dataset (we iterate through both)
            item (:obj:`object`): H5 group or H5 dataset being visited
        """
        # if item folder (group), just return
        if isinstance(item, h5py.Dataset):
            return
        # otherwise the item is a dataset
        if dset == 'Depth' or dset == 'Elevation' or 'DatasetCompression' not in item.attrs:
            return

        properties_path = item.name + '/PROPERTIES'
        properties_group = item.require_group(properties_path)
        if 'GUID' not in properties_group:
            dset_uuid = str(uuid.uuid4())
            ascii_list = [dset_uuid.encode("ascii", "ignore")]
            dtype = f'S{len(dset_uuid) + 1}'
            properties_group.create_dataset('GUID', data=ascii_list, dtype=dtype)
        else:
            dset_uuid = properties_group['GUID'][0].decode()

        self.free_dsets[self.sim_uuid].append((self.in_filename, item.name))
        if dset in self.dset_paths_to_widget:  # one of our referenced datasets
            attr_name = self.dset_paths_to_widget[dset]
            if attr_name == "Friction":  # Darcy-Weisbach or Manning's N
                if self.using_darcy:
                    self.data.info.attrs['darcy_uuid'] = dset_uuid
                else:
                    self.data.info.attrs['manning_uuid'] = dset_uuid
            else:
                self.data.info.attrs[attr_name] = dset_uuid
