"""Runs XMS DMI component save events."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import argparse

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules


def save_event_main(module_name, class_name, old_file, new_path, save_type, old_version):
    """Callable entry point for component save events.

    Args:
        module_name (str): The component's module import path
        class_name (str): The component's class name
        old_file (str): Path to the old component's main file
        new_path (str): Path to the new component instance directory
        save_type (str): The save event type
        old_version (Union[int, None]): The old component definition version, Unused
    """
    query = Query(timeout=300000)

    mod = __import__(module_name, fromlist=[class_name])
    klass = getattr(mod, class_name)
    class_instance = klass(old_file)

    new_main_file, messages, action_requests = class_instance.save_to_location(new_path, save_type)

    arg_list = []
    if action_requests:
        # Unwrap the pure Python ActionRequests.
        arg_list.append({"actions": [action._instance for action in action_requests]})
    if messages:
        arg_list.append({"messages": messages})
    if new_main_file:
        arg_list.append({"main_file": new_main_file})
    if arg_list:
        query._impl._instance.Set(arg_list)
        query.send(True)


if __name__ == "__main__":
    arguments = argparse.ArgumentParser(description="Component method runner.")
    arguments.add_argument(dest='script', type=str, help='script to run')
    arguments.add_argument(dest='module_name', type=str, help='module of the method to run')
    arguments.add_argument(dest='class_name', type=str, help='class of the method to run')
    arguments.add_argument(dest='old_file', type=str, help='old main file for the component')
    arguments.add_argument(dest='new_path', type=str, help='new file path for the component')
    arguments.add_argument(dest='save_type', type=str, help='one of: DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK')
    arguments.add_argument(
        dest='old_version',
        type=int,
        default=None,
        nargs='?',
        help='The old version of the component. Only happens during save_type=\'MIGRATE\''
    )
    parsed_args = arguments.parse_args()
    save_event_main(
        parsed_args.module_name,
        parsed_args.class_name,
        parsed_args.old_file,
        parsed_args.new_path,
        parsed_args.save_type,
        parsed_args.old_version,
    )
