"""Runs XMS DMI component UI event loop."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import gc
import os
import sqlite3
import sys
import traceback

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.components.runners.pool_process import PoolProcess

prog = None
query = None
conn = None
temp_dir = None
do_nothing_count = -1


def make_abs_path(main_path):
    """Makes a relative path absolute.

    Args:
        main_path (str): Relative path.

    Returns:
        (str): Absolute path.
    """
    if os.path.isabs(main_path):
        return main_path
    return os.path.normpath(os.path.join(temp_dir, main_path))


def get_merge_id_files(entity_path, uuid):
    """Gets the existing id files used for a merge.

    Args:
        entity_path (str): The folder to look in.
        uuid (str): The uuid in the file names.

    Returns:
        (tuple of str): A tuple of two file names, the first being an id file
                       of XMS ids, the second being an id file of component ids.
                       The tuple may be empty if the files do not exist.
    """
    merge_path = os.path.join(os.path.join(temp_dir, 'Merge'), entity_path)
    attids = os.path.join(merge_path, uuid + '.attids')
    compids = os.path.join(merge_path, uuid + '.compids')
    if os.path.isfile(attids) and os.path.isfile(compids):
        return attids, compids
    return ()


def command_function():
    """Gets a command from global query and sends it to SMS."""
    global do_nothing_count

    # Pre-import known modules as soon as XMS has written the import modules file (only happens once).
    PoolProcess.read_import_modules_file()

    gc.collect()
    result = query._impl._instance.Get("command", "uuids", "class_name", "module_name")
    if not result:
        raise Exception('Query::Get failed on command')
    command_result = result["command"]
    uuid_result = result["uuids"]
    if command_result and command_result[0] and uuid_result:
        do_nothing_count = 0
        arg_list = []
        try:
            command = command_result[0]
            module_result = result["module_name"]
            class_result = result["class_name"]

            select = 'SELECT MainFile, Locked, Uuid FROM Components WHERE Uuid IN ('
            first = True
            for uuid in result["uuids"][0]:
                if not first:
                    select += ', '
                select += "'" + uuid + "'"
                first = False
            select += ');'
            cursor = conn.execute(select)

            if command == "project_explorer_menu":
                # We will instantiate the component associated with the tree item that was right-clicked
                # on. There may be other components selected if this is a multi-select menu. It is up
                # to the component implementation to decide if multi-select menus are valid.
                main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                menu_items = {}
                mod_class_dict = {}
                # Group up the components by module and class name.
                for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                    if module_name not in mod_class_dict:
                        mod_class_dict[module_name] = {}
                    if class_name not in mod_class_dict[module_name]:
                        mod_class_dict[module_name][class_name] = []
                    mod_class_dict[module_name][class_name].append(comp_uuid)
                for module_name, class_info in mod_class_dict.items():
                    for class_name, uuids in class_info.items():
                        # Get all the menus for all instances of this type of component.
                        mod = __import__(module_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[uuids[0]][0])
                        main_file_list = [main_file_dict[comp_uuid] for comp_uuid in uuids]
                        menus = class_instance.get_project_explorer_menus(main_file_list)
                        if menus:
                            menu_items[uuids[0]] = [
                                menu_item if menu_item is None else menu_item._instance for menu_item in menus
                            ]
                if menu_items:
                    arg_list.append({"menus": menu_items})
            elif command == "display_menu":
                result = query._impl._instance.Get("selection", "component_coverage_ids")
                if not result:
                    raise Exception('Query::Get failed on selection')
                sel_result = result["selection"]
                comp_cov_result = result["component_coverage_ids"]
                if sel_result and sel_result[0] and comp_cov_result and comp_cov_result[0]:
                    selection = sel_result[0]
                    main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                    menu_items = {}
                    for module_name, class_name, comp_uuid, id_files in zip(
                        module_result[0], class_result[0], uuid_result[0], comp_cov_result[0]
                    ):
                        mod = __import__(module_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[comp_uuid][0])
                        lock_state = main_file_dict[comp_uuid][1] == 1
                        comp_id_files = {key: (value[0], value[1]) for key, value in id_files.items()}
                        menu_list = class_instance.get_display_menus(selection, lock_state, comp_id_files)
                        if menu_list:
                            # Unwrap the pure Python Menu/MenuItems.
                            menu_items[comp_uuid] = [
                                menu_item if menu_item is None else menu_item._instance for menu_item in menu_list
                            ]
                        # Clean up the id mapping files. The components did not ask for them.
                        for id_file_pair in comp_id_files.values():
                            try:
                                os.remove(id_file_pair[0])
                                os.remove(id_file_pair[1])
                            except Exception:
                                pass
                    if menu_items:
                        arg_list.append({"menus": menu_items})
            elif command == "edit":
                result = query._impl._instance.Get("edit_uuid")
                if not result:
                    raise Exception('Query::Get failed on edit_uuid')
                result_edit = result["edit_uuid"]
                if result_edit and result_edit[0]:
                    edit_uuid = result_edit[0]
                    main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                    all_messages = []
                    all_action_requests = []
                    for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                        mod = __import__(module_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[comp_uuid][0])
                        lock_state = main_file_dict[comp_uuid][1] == 1
                        messages, action_requests = class_instance.edit_event(edit_uuid, lock_state)
                        all_messages.extend(messages)
                        all_action_requests.extend(action_requests)
                    if all_action_requests:
                        arg_list.append({"actions": [action._instance for action in all_action_requests]})
                    if all_messages:
                        arg_list.append({"messages": all_messages})
            elif command == "delete":
                main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                all_messages = []
                all_action_requests = []
                for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                    mod = __import__(module_name, fromlist=[class_name])
                    klass = getattr(mod, class_name)
                    class_instance = klass(main_file_dict[comp_uuid][0])
                    lock_state = main_file_dict[comp_uuid][1] == 1
                    messages, action_requests = class_instance.delete_event(lock_state)
                    all_messages.extend(messages)
                    all_action_requests.extend(action_requests)
                if all_action_requests:
                    arg_list.append({"actions": [action._instance for action in all_action_requests]})
                if all_messages:
                    arg_list.append({"messages": all_messages})
            elif command == "create":
                main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                all_messages = []
                all_action_requests = []
                for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                    mod = __import__(module_name, fromlist=[class_name])
                    klass = getattr(mod, class_name)
                    class_instance = klass(main_file_dict[comp_uuid][0])
                    lock_state = main_file_dict[comp_uuid][1] == 1
                    messages, action_requests = class_instance.create_event(lock_state)
                    all_messages.extend(messages)
                    all_action_requests.extend(action_requests)
                if all_action_requests:
                    arg_list.append({"actions": [action._instance for action in all_action_requests]})
                if all_messages:
                    arg_list.append({"messages": all_messages})
            elif command == "import":
                main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                all_messages = []
                all_action_requests = []
                for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                    mod = __import__(module_name, fromlist=[class_name])
                    klass = getattr(mod, class_name)
                    class_instance = klass(main_file_dict[comp_uuid][0])
                    lock_state = main_file_dict[comp_uuid][1] == 1
                    messages, action_requests = class_instance.import_event(lock_state)
                    all_messages.extend(messages)
                    all_action_requests.extend(action_requests)
                if all_action_requests:
                    arg_list.append({"actions": [action._instance for action in all_action_requests]})
                if all_messages:
                    arg_list.append({"messages": all_messages})
            elif command == "link":  # know which single component it is
                result = query._impl._instance.Get("link_uuid", "link_parameter")
                if not result:
                    raise Exception('Query::Get failed on link')
                result_link = result["link_uuid"]
                result_param = result["link_parameter"]
                if result_link and result_param:
                    module_name = module_result[0][0]
                    class_name = class_result[0][0]
                    mod = __import__(module_name, fromlist=[class_name])
                    klass = getattr(mod, class_name)
                    component_inst = cursor.fetchone()
                    class_instance = klass(make_abs_path(component_inst[0]))
                    link_dict = {
                        link_uuid: [param for param in link_param]
                        for link_uuid, link_param in zip(result_link[0], result_param[0])
                    }
                    messages, action_requests = class_instance.link_event(link_dict, component_inst[1] == 1)
                    if action_requests:
                        # Unwrap the pure Python ActionRequests.
                        arg_list.append({"actions": [action._instance for action in action_requests]})
                    if messages:
                        arg_list.append({"messages": messages})
            elif command == "unlink":
                result = query._impl._instance.Get("unlink_uuid")
                if not result:
                    raise Exception('Query::Get failed on unlink')
                result_unlink = result["unlink_uuid"]
                if result_unlink and result_unlink[0]:
                    unlinks = [unlink for unlink in result_unlink[0]]
                    main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                    all_messages = []
                    all_action_requests = []
                    for module_name, class_name, item_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                        mod = __import__(module_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[item_uuid][0])
                        lock_state = main_file_dict[item_uuid][1] == 1
                        messages, action_requests = class_instance.unlink_event(unlinks, lock_state)
                        all_messages.extend(messages)
                        all_action_requests.extend(action_requests)
                    if all_action_requests:
                        arg_list.append({"actions": [action._instance for action in all_action_requests]})
                    if all_messages:
                        arg_list.append({"messages": all_messages})
            elif command == "renumber":
                result = query._impl._instance.Get("renumber")
                if not result:
                    raise Exception('Query::Get failed on renumber')
                result_renumber = result["renumber"]
                if result_renumber and result_renumber[0]:
                    renumbers = {
                        item_uuid:
                            {
                                entity_type: {
                                    old: new
                                    for old, new in mapping.items()
                                }
                                for entity_type, mapping in value.items()
                            }
                        for item_uuid, value in result_renumber[0].items()
                    }
                    main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                    all_messages = []
                    all_action_requests = []
                    for module_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                        mod = __import__(module_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[comp_uuid][0])
                        lock_state = main_file_dict[comp_uuid][1] == 1
                        messages, action_requests = class_instance.renumber_event(renumbers, lock_state)
                        all_messages.extend(messages)
                        all_action_requests.extend(action_requests)
                    if all_action_requests:
                        arg_list.append({"actions": [action._instance for action in all_action_requests]})
                    if all_messages:
                        arg_list.append({"messages": all_messages})
            elif command == "double_click":
                module_name = module_result[0][0]
                class_name = class_result[0][0]
                main_file_list = [(make_abs_path(component[0]), component[1]) for component in cursor]
                mod = __import__(module_name, fromlist=[class_name])
                klass = getattr(mod, class_name)
                class_instance = klass(main_file_list[0][0])
                lock_state = main_file_list[0][1] == 1
                messages, action_requests = class_instance.get_double_click_actions(lock_state)
                if action_requests:
                    # Unwrap the pure Python ActionRequests.
                    arg_list.append({"actions": [action._instance for action in action_requests]})
                if messages:
                    arg_list.append({"messages": messages})
            elif command == "double_click_selection":
                result = query._impl._instance.Get("selection", "component_coverage_ids")
                if not result:
                    raise Exception('Query::Get failed on selection')
                sel_result = result["selection"]
                comp_cov_result = result["component_coverage_ids"]
                if sel_result and sel_result[0] and comp_cov_result and comp_cov_result[0]:
                    selection = sel_result[0]
                    module_name = module_result[0][0]
                    class_name = class_result[0][0]
                    main_file_list = [(make_abs_path(component[0]), component[1]) for component in cursor]
                    mod = __import__(module_name, fromlist=[class_name])
                    klass = getattr(mod, class_name)
                    class_instance = klass(main_file_list[0][0])
                    lock_state = main_file_list[0][1] == 1
                    comp_id_files = {key: (value[0], value[1]) for key, value in comp_cov_result[0][0].items()}
                    messages, action_requests = class_instance.get_double_click_actions_for_selection(
                        selection, lock_state, comp_id_files
                    )
                    if action_requests:
                        # Unwrap the pure Python ActionRequests.
                        arg_list.append({"actions": [action._instance for action in action_requests]})
                    if messages:
                        arg_list.append({"messages": messages})
                    # Clean up the id mapping files. The components did not ask for them.
                    for id_file_pair in comp_id_files.values():
                        try:
                            os.remove(id_file_pair[0])
                            os.remove(id_file_pair[1])
                        except Exception:
                            pass
            elif command == "lock":
                pass
            elif command == "unlock":
                pass
            elif command == "merge":
                result = query._impl._instance.Get("merge_coverages")
                if not result:
                    raise Exception('Query::Get failed on merge')
                merge_result = result["merge_coverages"]
                all_messages = []
                all_action_requests = []
                if merge_result and merge_result[0]:
                    # get the old components that were a part of the coverages merged into the new one
                    merge_comp_uuids = merge_result[0][0]
                    merge_comp_classes = merge_result[0][1]
                    merge_comp_modules = merge_result[0][2]
                    merge_comp_main_files = merge_result[0][3]
                    # create a dictionary by class and modules with a list of
                    # (main file, {entity type : component id file, XMS attribute id file})
                    merge_dict = {}
                    for merge_uuid, class_name, mod_name, main_file in \
                            zip(merge_comp_uuids, merge_comp_classes, merge_comp_modules, merge_comp_main_files):
                        if mod_name not in merge_dict:
                            merge_dict[mod_name] = {}
                        if class_name not in merge_dict[mod_name]:
                            merge_dict[mod_name][class_name] = []
                        poly_files = get_merge_id_files('Polygon', merge_uuid)
                        arc_files = get_merge_id_files('Arc', merge_uuid)
                        point_files = get_merge_id_files('Point', merge_uuid)
                        id_dict = {'POLYGON': poly_files, 'ARC': arc_files, 'POINT': point_files}
                        merge_dict[mod_name][class_name].append((main_file, id_dict))

                    # go through the newly created components
                    main_file_dict = {component[2]: (make_abs_path(component[0]), component[1]) for component in cursor}
                    for mod_name, class_name, comp_uuid in zip(module_result[0], class_result[0], uuid_result[0]):
                        if mod_name not in merge_dict:
                            continue
                        elif class_name not in merge_dict[mod_name]:
                            continue
                        elif not merge_dict[mod_name][class_name]:
                            continue
                        mod = __import__(mod_name, fromlist=[class_name])
                        klass = getattr(mod, class_name)
                        class_instance = klass(main_file_dict[comp_uuid][0])
                        messages, action_requests = class_instance.handle_merge(merge_dict[mod_name][class_name])
                        all_messages.extend(messages)
                        all_action_requests.extend(action_requests)
                if all_action_requests:
                    # Unwrap the pure Python ActionRequests.
                    arg_list.append({"actions": [action._instance for action in all_action_requests]})
                if all_messages:
                    arg_list.append({"messages": all_messages})
        except Exception as ex:
            with open('debug_ui_loop.txt', 'a') as f:
                traceback.print_exception(type(ex), ex, ex.__traceback__, file=f)
        finally:
            if arg_list:
                query._impl._instance.Set(arg_list)
            query.send(True)
    else:
        do_nothing_count += 1
        # 16000 get operations leaks ~40MB, restart to cleanup memory
        if do_nothing_count == 16000:
            arg_list = [{"restart_ui": None}]
            query._impl._instance.Set(arg_list)
            query.send(True)
            sys.exit()


def main():
    """Start the XMS DMI component UI event loop."""
    global prog
    global query
    global conn
    global temp_dir

    query = Query(timeout=10000)

    try:
        prog = query.xms_agent.session.progress_loop
        temp_dir = os.path.dirname(prog.plot_db_file)
        conn = sqlite3.connect('file:' + prog.plot_db_file + '?mode=ro', uri=True)
    except Exception as ex:
        with open("debug_ui_db_connect.txt", "w") as f:
            traceback.print_exception(type(ex), ex, ex.__traceback__, file=f)
            sys.exit(1)

    prog.set_progress_function(command_function)
    query._impl._instance.SetContext(prog.progress_context._instance)
    try:
        prog.start_constant_loop(100)
    except SystemExit:
        sys.exit(0)
    except Exception as ex:
        with open('debug_ui_loop.txt', 'a') as f:
            traceback.print_exception(type(ex), ex, ex.__traceback__, file=f)
        sys.exit(1)


if __name__ == "__main__":
    main()
