"""Testing utility functions."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import filecmp
from pathlib import Path
import subprocess
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.executables.ncdump import paths as ncdump_paths
from xms.testing import file_comparison

# 4. Local modules


def sqlite_databases_equal(filename1, filename2):
    """Uses sqldiff.exe to compare two sqlite database files and returns True if they are equal.

    Args:
        filename1 (str): Path to first file.
        filename2 (str): Path to second file.

    Returns:
        True if equal.
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.sqlite_databases_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    return file_comparison.sqlite_databases_equal(filename1, filename2)


def compare_sqlite_files(dir1, dir2, database_files):
    """Check if database files exist in the directories and, if so, compare them.

    Args:
        dir1 (str): Path of first directory.
        dir2 (str): Path of second directory.
        database_files: List of base filenames to look for.

    Returns:
        An empty string if files not found or they match, otherwise an error message.
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.sqlite_databases_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    return file_comparison.sqlite_databases_equal(dir1, dir2)


def dir_trees_equal(dir1, dir2, sqlite_files=None) -> bool:
    """
    Compare two directories recursively.

    Files in each directory are assumed to be equal if their names and contents are equal.

    Args:
        dir1: First directory path
        dir2: Second directory path
        sqlite_files: List of database files that may be encountered. These must be compared differently.

    Returns:
        Whether the directories were equal.
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.are_dir_trees_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    return file_comparison.are_dir_trees_equal(dir1, dir2)


def are_dir_trees_equal(dir1, dir2, sqlite_files=None) -> str:
    """Compares two directories recursively.

    Files in each directory are assumed to be equal if their names and contents are equal.

    The return value may mention "funny" files. These are files that are in dir1 and dir2, but could not be compared
    for some reason. They might be different types (file vs. folder) or be inaccessible.

    Args:
        dir1: First directory path
        dir2: Second directory path
        sqlite_files: List of database files that may be encountered. These must be compared differently.

    Returns:
        An empty string if dirs are equal, or a message describing problems if they are unequal.
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.are_dir_trees_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    diffs = []
    file_comparison.are_dir_trees_equal(dir1, dir2, diffs=diffs)
    return '\n'.join(diffs)


def compare_h5_files(filename1: str | Path, filename2: str | Path) -> str:
    """
    Check if H5 files exist in the directories and, if so, compare them.

    Args:
        filename1: First file path.
        filename2: Second file path.

    Returns:
        An empty string if they match, otherwise an error message.
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.h5_files_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    diffs = []
    file_comparison.h5_files_equal(filename1, filename2, diffs=diffs)
    if diffs:
        return '\n'.join(diffs)
    return ''


def compare_file_list(comp_file_list):
    """Compares a list of lists of files and returns '' if they're all equal.

    Args:
        comp_file_list: List of lists of files.

    Returns:
        str: The diff string
    """
    message = ''
    for comp in comp_file_list:
        if not filecmp.cmp(comp[0], comp[1], shallow=False):
            message += f'FAIL - Mismatch:\n{comp[0]}\n{comp[1]}\n'
    return message


def netcdf_to_text(data_file: str | Path, out_file: str | Path, force_file_name: bool = False):
    """
    Dump a NetCDF (or any H5 file) to a text file.

    Args:
        data_file: Path to the NetCDF file
        out_file: Path to the output text file
        force_file_name: Forces the converter to behave as if the input file was named 'base.nc'.
            By default, the converter includes the input file's name in its output. This means that an output file's
            contents will depend on the name of the file, which is surprising. Passing True for this parameter makes the
            converter behave as if the input file's name was 'base.nc', which makes the output file's content
            independent of the file name, which is what most people expect to happen.

            This parameter only exists and defaults to False because a lot of tests expect the old behavior and haven't
            been updated yet. New tests should pass True, and old ones updated to use True too.
    """
    ncdump_exe = ncdump_paths['ncdump.exe']
    arguments = [ncdump_exe]
    if force_file_name:
        arguments.append('-n')
        arguments.append('base')
    else:
        warnings.warn(
            'Consider using force_file_name=True and updating tests to match.',
            category=DeprecationWarning,
            stacklevel=2
        )

    arguments.append(data_file)
    with open(out_file, 'w') as f:
        subprocess.run(arguments, stdout=f, check=True)


def xarray_base_to_text(data_file, out_file):
    """Dump an XarrayBase data class to a text file.

    Strips out the version number attribute because it was ticking me off

    Args:
        data_file (str): Path to the XarrayBase data file
        out_file (str): Path to the output text file

    """
    # Note: If you are calling this method, your package needs to be dependent on xmscomponents. I removed the
    #       dependency from this package to avoid circular dependencies.
    from xms.components.bases.xarray_base import XarrayBase
    data = XarrayBase(data_file)
    data.info.attrs.pop('VERSION')  # So you don't have to update baseline with every version bump
    data.commit()
    data.close()
    netcdf_to_text(data_file, out_file)
