"""Class for managing interprocess communication with XMS."""
# 1. Standard python modules
import os.path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util

# 4. Local modules
from xms.cstorm.data.sim_data import SimData
from xms.cstorm.file_io import util

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class XmsData:
    """Class for managing interprocess communication with XMS."""
    def __init__(self, query=None, at_sim=True):
        """Constructor.

        Args:
            query (:obj:`Query`): The XMS interprocess communication object
            at_sim (:obj:`bool`): True if the Context is at the simulation level, False if it is directly below the
            simulation.
        """
        self._query = query
        self._logger = util.get_logger()
        self._at_sim = at_sim
        self._adcirc_sim = None
        self._sim_uuid = None
        self._sim_name = None
        self._sim_data = None
        self._stwave_sims = None
        self._stwave_grids = None
        self._comp_folder = None

    @property
    def query(self):
        """Returns the xmsapi interprocess communication object.

        Notes:
            If constructed in a component runner script, Query should be passed in on construction. If constructed in
            an export or import script, creation of the Query can happen here. Only one Query/XmsData object should be
            constructed per script.
        """
        if self._query is None:
            self._query = Query()
        return self._query

    def set_sim(self, sim_uuid, sim_name):
        """Set the simulation that data will be retrieved for.

        Args:
            sim_uuid (:obj:`str`): The UUID of the simulation
            sim_name (:obj:`str`): The name of the simulation
        """
        self._sim_uuid = sim_uuid
        self._sim_name = sim_name

    @property
    def sim_uuid(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_uuid is None:
            self._sim_uuid = self.query.current_item_uuid() if self._at_sim else self.query.parent_item_uuid()
        return self._sim_uuid

    @property
    def sim_name(self):
        """Returns the name of simulation."""
        if self._sim_name is None:
            sim_item = self.query.current_item() if self._at_sim else self.query.parent_item()
            if sim_item and sim_item.name:
                self._sim_name = sim_item.name
            else:
                self._sim_name = 'Sim'
        return self._sim_name

    @property
    def sim_data(self):
        """Returns the simulation component's SimData."""
        if self._sim_data is None:
            self._logger.info('Retrieving simulation data from SMS...')
            do_comp = self.query.item_with_uuid(self.sim_uuid, model_name='CSTORM', unique_name='SimComponent')
            if not do_comp:
                raise RuntimeError('Unable to retrieve simulation data from SMS.')
            self._sim_data = SimData(do_comp.main_file)
        return self._sim_data

    @property
    def component_folder(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        return os.path.dirname(os.path.dirname(self.sim_data.main_file))

    def get_tree_item(self, item_uuid):
        """Get a project explorer item from UUID.

        Args:
            item_uuid (:obj:`str`): UUID of the item to retrieve

        Returns:
            (:obj:`TreeNode`): The tree item with specified UUID or None if not found
        """
        return tree_util.find_tree_node_by_uuid(self._query.project_tree, item_uuid)

    @property
    def adcirc_sim(self):
        """Returns the ADCIRC simulation linked to the CSTORM sim."""
        if self._adcirc_sim is not None:
            return self._adcirc_sim
        sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, self.sim_uuid)
        sim_items = tree_util.descendants_of_type(
            tree_root=sim_item, xms_types=['TI_DYN_SIM_PTR'], allow_pointers=True, model_name='ADCIRC'
        )
        pt = self._query.project_tree
        for adc_sim_ptr in sim_items:
            sim = tree_util.find_tree_node_by_uuid(pt, adc_sim_ptr.uuid)
            sim_comp = self._query.item_with_uuid(sim.uuid, model_name='ADCIRC', unique_name='Sim_Component')
            self._adcirc_sim = sim, sim_comp
        return self._adcirc_sim

    @property
    def stwave_sims(self):
        """Returns a list of the STWAVE simulations linked to the CSTORM sim."""
        if self._stwave_sims is not None:
            return self._stwave_sims
        sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, self.sim_uuid)
        sim_items = tree_util.descendants_of_type(
            tree_root=sim_item, xms_types=['TI_DYN_SIM_PTR'], allow_pointers=True, model_name='STWAVE'
        )
        sims = []
        for st_sim_ptr in sim_items:
            st_sim = tree_util.find_tree_node_by_uuid(self._query.project_tree, st_sim_ptr.uuid)
            sim_comp = self._query.item_with_uuid(st_sim.uuid, model_name='STWAVE', unique_name='Sim_Component')
            sims.append((st_sim, sim_comp))
        self._stwave_sims = sims
        return self._stwave_sims
