"""Python wrapping for xms._data_objects.parameters.Projection."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import Projection as CProjection


# These keywords are for constructing projections without a WKT.
XMS_COORDINATE_SYSTEMS = {
    'NONE',  # Same as 'LOCAL'
    'LOCAL',
    'GEOGRAPHIC',
    'STATEPLANE',
    'UTM',
}


XMS_PROJECTION_UNITS = {
    'ARC_DEGREES',
    'FEET (INTERNATIONAL)',
    'FEET (U.S. SURVEY)',
    'INCHES',
    'METERS',
    'CENTIMETERS',
}


XMS_DATUMS = {
    'LOCAL',
    'NAVD88',
    'NGVD29',
}


class Projection:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Projection objects."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Note:
            Can either construct with a WKT or build with the supported projection parameters and values. WKTs are more
            flexible and will take precedence over any of the other parameters.

        Args:
            **kwargs:
                wkt (str): The projection's well known text
                system (str): See coordinate_system.setter for more details
                zone (int): See coordinate_zone.setter for more details
                horizontal_units (str): See horizontal_units.setter for more details
                vertical_units (str): See vertical_units.setter for more details
                vertical_datum (str): See vertical_datum.setter for more details
                projection_name (str): The projection's user-friendly name. Note that this is only applicable for
                    projections created by XMS. Mostly here for testing putrposes.
                instance (CProjection): The C++ Projection object to wrap
        """
        self._instance = kwargs.get('instance')
        if not self._instance:
            self._instance = CProjection()

        if 'wkt' in kwargs:
            self._instance.SetWellKnownText(kwargs['wkt'])
        if 'system' in kwargs:
            self.coordinate_system = kwargs['system']  # Use property setter to validate
        if 'zone' in kwargs:
            self._instance.SetCoordinateZone(kwargs['zone'])
        if 'horizontal_units' in kwargs:
            self.horizontal_units = kwargs['horizontal_units']  # Use property setter to validate
        if 'vertical_units' in kwargs:
            self.vertical_units = kwargs['vertical_units']  # Use property setter to validate
        if 'vertical_datum' in kwargs:
            self.vertical_datum = kwargs['vertical_datum']  # Use property setter to validate
        if 'projection_name' in kwargs:
            self._instance.SetProjectionName(kwargs['projection_name'])

    @property
    def coordinate_system(self):
        """Returns the projection's coordinate system."""
        return self._instance.GetCoordinateSystem()

    @coordinate_system.setter
    def coordinate_system(self, coord_sys):
        """Sets the projection's coordinate system.

        Args:
            coord_sys (str): One of the XMS_COORDINATE_SYSTEMS strings
        """
        coord_sys_upper = coord_sys.upper()
        if coord_sys_upper and coord_sys_upper not in XMS_COORDINATE_SYSTEMS:  # Allow clearing with empty string
            raise ValueError(f'Unsupported coordinate system: {coord_sys}\nMust be one of: {XMS_COORDINATE_SYSTEMS}')
        self._instance.SetCoordinateSystem(coord_sys_upper)

    @property
    def coordinate_zone(self):
        """Returns the projection's coordinate zone if applicable, else -1."""
        return self._instance.GetCoordinateZone()

    @coordinate_zone.setter
    def coordinate_zone(self, coord_zone):
        """Sets the projection's coordinate zone.

        Only applicable if the coordinate system is 'STATEPLANE' or 'UTM'

        Args:
            coord_zone (int): The coordinate zone id
        """
        self._instance.SetCoordinateZone(coord_zone)

    @property
    def horizontal_units(self):
        """Returns the projection's horizontal units."""
        return self._instance.GetHorizontalUnits()

    @horizontal_units.setter
    def horizontal_units(self, horiz_units):
        """Sets the projection's horizontal units.

        Args:
            horiz_units (str): One of the XMS_PROJECTION_UNITS strings
        """
        horiz_units_upper = horiz_units.upper()
        if horiz_units_upper and horiz_units_upper not in XMS_PROJECTION_UNITS:  # Allow clearing with empty string
            raise ValueError(f'Unsupported horizontal units: {horiz_units}\nMust be one of: {XMS_PROJECTION_UNITS}')
        self._instance.SetHorizontalUnits(horiz_units_upper)

    @property
    def vertical_units(self):
        """Returns the projection's vertical units."""
        return self._instance.GetVerticalUnits()

    @vertical_units.setter
    def vertical_units(self, vert_units):
        """Sets the projection's vertical units.

        Args:
            vert_units (str): One of the XMS_PROJECTION_UNITS strings
        """
        vert_units_upper = vert_units.upper()
        if vert_units_upper and vert_units_upper not in XMS_PROJECTION_UNITS:  # Allow clearing with empty string
            raise ValueError(f'Unsupported vertical units: {vert_units}\nMust be one of: {XMS_PROJECTION_UNITS}')
        self._instance.SetVerticalUnits(vert_units_upper)

    @property
    def vertical_datum(self):
        """Returns the projection's vertical datum."""
        return self._instance.GetVerticalDatum()

    @vertical_datum.setter
    def vertical_datum(self, vert_datum):
        """Sets the projection's vertical datum.

        Args:
            vert_datum (str): One of the XMS_DATUMS strings
        """
        vert_datum_upper = vert_datum.upper()
        if vert_datum_upper and vert_datum_upper not in XMS_DATUMS:  # Allow clearing with empty string
            raise ValueError(f'Unsupported vertical datum: {vert_datum}\nMust be one of: {XMS_DATUMS}')
        self._instance.SetVerticalDatum(vert_datum_upper)

    @property
    def well_known_text(self):
        """Get the well-known text of the projection."""
        return self._instance.GetWellKnownText()

    @well_known_text.setter
    def well_known_text(self, wkt):
        """Set the well-known text of the projection.

        Note that this will not update other member variables such as the coordinate zone or zone id.
        When projection is retrieved from XMS, those members are set for convenience. When sending a
        projection to XMS, setting the WKT is the recommended method. All other attributes will be ignored
        when reading a projection into XMS when the WKT has been set.

        Args:
            wkt (str): The well-known text of the projection.
        """
        self._instance.SetWellKnownText(wkt)

    @property
    def projection_name(self):  # Read-only, text defined by XMS.
        """Returns the user-friendly projection name string for GUI display."""
        return self._instance.GetProjectionName()
