"""Python wrapping for xms._data_objects.parameters.Dataset."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import DsetDataMappingType, DsetActivityMappingType, Dataset as CDataset


DatasetDataBasisStrings = {
    'NODE': DsetDataMappingType.NODE_MAPPED,
    'CELL': DsetDataMappingType.CELL_MAPPED,
    'UNKNOWN': DsetDataMappingType.UNKNOWN_MAPPED,
}

DatasetDataActivityStrings = {
    'NONE': DsetActivityMappingType.ACTIVITY_NONE,
    'CELL': DsetActivityMappingType.ACTIVITY_CELL,
    'NODE': DsetActivityMappingType.ACTIVITY_NODE,
    'NULL': DsetActivityMappingType.ACTIVITY_NULL,
}

DatasetTimeUnits = [
    'Seconds',
    'Minutes',
    'Hours',
    'Days',
    'None',
]

XMDF_DATASET_GROUP = 'Datasets'


def dataset_enum_to_text(lookup_map, enum_value):
    """Convert C++ dataset enum to associated pure Python string constant.

    Raises ValueError if not found.

    Args:
        lookup_map (dict): Dictionary where keys are the pure Python string constants, and values are the C++ exposed
            enum values.
        enum_value (object): The C++ enum value to lookup

    Returns:
        str: The pure Python string constant associated with the specified C++ enum value.
    """
    try:
        return next(key for key, value in lookup_map.items() if value == enum_value)
    except StopIteration:
        raise ValueError(f'basis must be one of the following options: {lookup_map.values()}')


def dataset_text_to_enum(lookup_map, text_value):
    """Convert pure Python dataset string constant to associated C++ enum value.

    Raises ValueError if not found.

    Args:
        lookup_map (dict): Dictionary where keys are the pure Python string constants, and values are the C++ exposed
            enum values.
        text_value (str): The pure Python string constant to lookup

    Returns:
        object: The C++ enum value associated with the specified pure Python string constant.
    """
    basis_enum = lookup_map.get(text_value.upper())
    if basis_enum is None:
        raise ValueError(f'basis must be one of the following options: {lookup_map.keys()}')
    return basis_enum


class Dataset:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Dataset objects."""
    def __init__(self, *args, **kwargs):
        """Construct the wrapper.

        Args:
            *args:
                filename (str): The path to the H5 file containing the dataset data (in internal format)
                h5_group (str): Group path in the H5 file to the dataset data
                mapping_type (str, optional): Location of dataset values. One of the DatasetDataBasisStrings keys,
                    'UNKNOWN' by default.
                activity_type (str, optional): Location of activity array flags. One of the DatasetDataActivityStrings
                    keys, 'NONE' by default
            **kwargs:
                instance (CDataset):  - The C++ object to wrap
        """
        num_args = len(args)
        if num_args >= 2:
            data_basis = DsetDataMappingType.UNKNOWN_MAPPED if num_args < 3 else dataset_text_to_enum(
                DatasetDataBasisStrings, args[2])
            activity_basis = DsetActivityMappingType.ACTIVITY_NONE if num_args < 4 else dataset_text_to_enum(
                DatasetDataActivityStrings, args[3])
            self._instance = CDataset(args[0], args[1], data_basis, activity_basis)
        elif 'instance' in kwargs:
            self._instance = kwargs['instance']
        else:  # default constructed Dataset
            self._instance = CDataset()

    @property
    def num_values(self):
        """Returns the number of dataset values."""
        return self._instance.GetNumValues()

    @property
    def num_times(self):
        """Returns the number of dataset times."""
        return self._instance.GetNumTimes()

    @property
    def num_components(self):
        """Returns the number of dataset components."""
        return self._instance.GetNumComponents()

    @property
    def num_activity_values(self):
        """Returns the number of dataset activity values."""
        return self._instance.GetNumActivityValues()

    @property
    def name(self):
        """Returns the dataset's name."""
        return self._instance.GetName()

    @name.setter
    def name(self, dset_name):
        """Set the the dataset's name.

        Args:
            dset_name (str): The dataset geometry's name
        """
        self._instance.SetName(dset_name)

    @property
    def data_basis(self):
        """Returns the location of the dataset values (one of the DatasetDataBasisStrings keys)."""
        return dataset_enum_to_text(DatasetDataBasisStrings, self._instance.GetDataBasis())

    @data_basis.setter
    def data_basis(self, basis):
        """Set the location of the dataset values.

        Args:
            basis (str) : Location of the dataset values. One of the DatasetDataBasisStrings keys.
        """
        self._instance.SetDataBasis(dataset_text_to_enum(DatasetDataBasisStrings, basis))

    @property
    def activity_basis(self):
        """Returns the location of the dataset activity array flags (one of the DatasetDataActivityStrings keys)."""
        return dataset_enum_to_text(DatasetDataActivityStrings, self._instance.GetActivityBasis())

    @activity_basis.setter
    def activity_basis(self, basis):
        """Set the location of the dataset activity array flags.

        Args:
            basis (str) : Location of the dataset activity array flags. One of the DatasetDataActivityStrings keys.
        """
        self._instance.SetActivityBasis(dataset_text_to_enum(DatasetDataActivityStrings, basis))

    @property
    def dataset_uuid(self):
        """Returns the UUID of the dataset."""
        return self._instance.GetDatasetUUID()

    @dataset_uuid.setter
    def dataset_uuid(self, dset_uuid):
        """Set the UUID of the dataset.

        Args:
            dset_uuid (str): The dataset's UUID
        """
        self._instance.SetDatasetUUID(dset_uuid)

    @property
    def geom_uuid(self):
        """Returns the UUID of the dataset's geometry."""
        return self._instance.GetGeomUUID()

    @geom_uuid.setter
    def geom_uuid(self, uuid_str):
        """Set the UUID of the dataset's geometry.

        Args:
            uuid_str (str): The dataset geometry's UUID
        """
        self._instance.SetGeomUUID(uuid_str)

    @property
    def time_units(self):
        """Returns the time units of the dataset times in the H5 file (one of the DatasetTimeUnits constants)."""
        return self._instance.GetTimeUnits()

    @time_units.setter
    def time_units(self, units):
        """Set the time units of the dataset times in the H5 file..

        Args:
            units (str): The dataset time step time units. One of the constants defined in DatasetTimeUnits.

        """
        units = units.title()
        if units not in DatasetTimeUnits:
            raise ValueError(f'units must be one of: {DatasetTimeUnits}')
        self._instance.SetTimeUnits(units)

    @property
    def using_ref_time(self):
        """Returns whether the dataset is using ref time."""
        return self._instance.UsingRefTime()

    @property
    def ref_time(self):
        """Returns the dataset's reference time as a Julian date or 0.0 if undefined."""
        return self._instance.GetRefTime()

    @ref_time.setter
    def ref_time(self, julian_time):
        """Set the dataset's reference time.

        Args:
            julian_time (float): The dataset reference timestamp as a Julian value
        """
        self._instance.SetRefTime(julian_time)

    @property
    def using_null_value(self):
        """Returns whether the dataset is using a null value."""
        return self._instance.UsingNullValue()

    @property
    def null_value(self):
        """Returns the dataset's null value."""
        return self._instance.GetNullValue()

    @null_value.setter
    def null_value(self, value):
        """Set the dataset's null value.

        Args:
            value (float): The dataset null value
        """
        self._instance.SetNullValue(value)

    @property
    def ts_idx(self):
        """Returns the current time step index of the dataset."""
        return self._instance.GetTsIdx()

    @ts_idx.setter
    def ts_idx(self, idx):
        """Sets the current time step index of the dataset.

        Args:
            idx (int): The current time step index (0-based)
        """
        self._instance.SetTsIdx(idx)

    @property
    def ts_time(self):
        """Returns the current time step's time as an offset in days from the reference time."""
        return self._instance.GetTsTime()

    @property
    def activity(self):
        """Returns the activity array for the current time step."""
        return self._instance.GetActivity()

    @property
    def data(self):
        """Get the current time step's values.

        Returns:
            list: The current time step values. If scalar, 1-D list where len=num_values. If vector, list has shape
                (num_values, num_components).
        """
        return self._instance.GetData()

    @property
    def filename(self):
        """Returns the H5 filename of the dataset."""
        return self._instance.GetFilename()

    @filename.setter
    def filename(self, filename):
        """Set the dataset's filename.

        Args:
            filename (str): Path of the H5 file containing the dataset
        """
        self._instance.SetFilename(filename)

    @property
    def group_path(self):
        """Returns the H5 group path to the dataset."""
        return self._instance.GetPath()

    @group_path.setter
    def group_path(self, path):
        """Set the H5 group path to the dataset.

        Args:
            path (str): H5 group path to the dataset
        """
        self._instance.SetPath(path)

    def get_minimum(self, ts_idx=-1, all_times=False):
        """Get the minimum of a time step or the entire dataset.

        Args:
            ts_idx (int): Index of the time step to retrieve minimum for. Use current time step if specified index is
                less than zero. Ignored if all_times is True.
            all_times (bool): If True, retrieve minimum across all time steps.

        Returns:
            float: The minimum of the dataset or specified time step
        """
        return self._instance.GetMinimum(ts_idx, all_times)

    def get_maximum(self, ts_idx=-1, all_times=False):
        """Get the maximum of a time step or the entire dataset.

        Args:
            ts_idx (int): Index of the time step to retrieve maximum for. Use current time step if specified index is
                less than zero. Ignored if all_times is True.
            all_times (bool): If True, retrieve maximum across all time steps.

        Returns:
            float: The maximum of the dataset or specified time step
        """
        return self._instance.GetMaximum(ts_idx, all_times)

    def set_data(self, filename, values, activity=None, group_path=XMDF_DATASET_GROUP):
        """Writes dataset values to H5 file in XMDF format.

        Note that the ref_time, data_basis, activity_basis, time_units, name, geom_uuid, and dataset_uuid properties
        should be set before calling this method.

        Using this method for writing dataset files should be used with caution for performance reasons (all time steps
        must be loaded into memory). If possible, write an XMDF-compliant H5 file independently, and use the positional
        argument constructor to create a Dataset object to send back to XMS.

        Args:
            filename (str or xms.data_objects._data_objects.parameters.FileLocation): Path to the output H5 file as
                a string or a FileLocation object retrieved from an xms.api.dmi.Query.
            values (dict): The dataset values. Each key-value pair represents a time step. Keys are stringified time
                offsets from the ref_time property in the time_units property units. Values are a list of the time step
                values. If non-scalar, list should have shape of (num_values, num_components).
            activity (list): The dataset's activity array, if applicable. Shape of list should be
                (num_times, num_activity_values).
            group_path (str): Group path in the H5 file to write the dataset data.
        """
        activity = [] if activity is None else activity
        self._instance.SetData(filename, values, activity, group_path)
