"""Module for handling XMS interprocess communication required for running EWN Feature tool commands."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.adcirc.components.bc_component import BcComponent
from xms.adh.components.material_conceptual_component import MaterialConceptualComponent
from xms.constraint import GridType, read_grid_from_file

# 4. Local modules


class ToolRunnerInputQueries:
    """Class to retrieve input data for tool runners."""
    def __init__(self, query):
        """Constructor.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
        """
        self._query = query
        self._logger = logging.getLogger('xms.ewn')
        self.ugrid_wkt = ''
        self.is_geographic = False
        self.projection = None

    def get_insert_features_input(self, geom_uuid, coverage_uuids, coverage_type='EWN'):
        """Query XMS for input data required to run the insert EWN features command.

        Args:
            geom_uuid (:obj:`str`): UUID of the geometry to dump
            coverage_uuids (:obj:`list[str]`): UUIDs of the coverages to dump
            coverage_type (:obj:`str`): 'EWN' or 'Sediment'

        Returns:
            (:obj:`tuple`): Input data required for inserting EWN features -

                target geometry,

                input coverages,

                xms.grid.ugrid.UGrid,

                [(data_objects.parameters.Coverage, EwnCoverageComponent)],

                Boolean
        """
        grid = None
        if geom_uuid is not None:
            grid = self.get_ugrid(geom_uuid, return_co_grid=True)
        coverage_data = self.get_coverage_data(coverage_uuids, coverage_type)
        is_cartesian = grid.grid_type in [GridType.rectilinear_2d, GridType.rectilinear_3d]
        return grid, coverage_data, is_cartesian

    def get_generate_roughness_dataset_input(self, inputs):
        """Query XMS for input data required to run the generate roughness dataset command.

        Args:
            inputs (:obj:`dict`): User input data from dialog

        Returns:
            tuple: Input data required for generating a roughness dataset:

                target geometry,

                input coverages,

                initial values,

                xms.grid.ugrid.UGrid,

                [(data_objects.parameters.Coverage,EwnCoverageComponent)],

                [float]
        """
        cogrid = self.get_ugrid(inputs['target_geometry'], return_co_grid=True)
        coverage_data = self.get_coverage_data(inputs['ewn_coverages'])

        location_count = cogrid.ugrid.point_count
        if inputs['dataset_location'] == 'cells':
            location_count = cogrid.ugrid.cell_count
        initial_values, dset_location = self._get_initial_roughness(inputs, location_count)
        return cogrid, coverage_data, initial_values, dset_location

    def _get_initial_roughness(self, inputs, num_points):
        """Returns the initial roughness dataset values.

        Args:
            inputs (:obj:`dict`): User input data from dialog
            num_points (:obj:`int`): Number of points in the ugrid

        Returns:
            (:obj:`list[float]`): See description
        """
        # Get the initial roughness values.
        if 'constant_roughness' in inputs:  # Constant initial roughness value at all nodes
            return [inputs['constant_roughness']] * num_points, inputs['dataset_location']
        else:  # Use an existing dataset as the initial values
            dset_uuid = inputs['initial_roughness_uuid']
            dset = self._query.item_with_uuid(dset_uuid)
            ts_idx = max(inputs['initial_roughness_timestep'], 0)
            return dset.values[ts_idx], dset.location

    def get_ugrid(self, geom_uuid, return_co_grid=False):
        """Get a CoGrid dump of the geometry to insert the EWN features into.

        Args:
            geom_uuid (:obj:`str`): UUID of the geometry to dump

        Returns:
            (:obj:`xms.grid.ugrid.UGrid`): See description
        """
        self._logger.info('Retrieving geometry data...')
        do_ugrid = self._query.item_with_uuid(geom_uuid)
        self.projection = do_ugrid.projection
        if self.projection.coordinate_system == 'GEOGRAPHIC':
            self.is_geographic = True
        self.ugrid_wkt = self.projection.well_known_text
        co_grid = read_grid_from_file(do_ugrid.cogrid_file)
        if return_co_grid:
            return co_grid
        return co_grid.ugrid

    def get_lock_dataset(self, lock_dataset_uuid):
        """Get the lock dataset for a ugrid.

        Args:
            lock_dataset_uuid (:obj:`str`): The lock dataset uuid
        """
        self._logger.info('Retrieving lock dataset data...')
        lock_dataset = []
        do_lock_dataset = self._query.item_with_uuid(lock_dataset_uuid)
        if do_lock_dataset:
            lock_dataset = [x for x in do_lock_dataset.values[0]]
        return lock_dataset

    def get_coverage_data(self, coverage_uuids, coverage_type='EWN'):
        """Get coverage component data and polygon points.

        Args:
            coverage_uuids (:obj:`list[str]`): UUIDs of the coverages to dump
            coverage_type (:obj:`str`): 'EWN' or 'ADCIRC BC' or 'Channel'

        Returns:
            (:obj:`list[tuple]`):
                The coverage and component data:

                (:obj:`[ (data_objects.parameters.Coverage, EwnCoverageComponent) ]`)
        """
        from xms.ewn.components.ewn_coverage_component import EwnCoverageComponent
        coverage_data = []
        for idx, coverage_uuid in enumerate(coverage_uuids):
            self._logger.info(f'Retrieving coverage data ({idx + 1} of {len(coverage_uuids)})...')
            # Get the coverage's geometry
            cov_geom = self._query.item_with_uuid(coverage_uuid)
            # Get the coverage's component data
            comp = None
            inputs = {}
            if coverage_type == 'EWN':
                do_comp = self._query.item_with_uuid(
                    coverage_uuid, model_name='Engineering with Nature', unique_name='EwnCoverageComponent'
                )
                comp = EwnCoverageComponent(do_comp.main_file)
                inputs = {'component': comp, 'polygons': True, 'arcs': True}
            elif coverage_type == 'ADCIRC_BC':
                do_comp = self._query.item_with_uuid(coverage_uuid, model_name='ADCIRC', unique_name='Bc_Component')
                comp = BcComponent(do_comp.main_file)
                inputs = {'component': comp, 'arcs': True, 'points': True}
            elif coverage_type == 'ADH_MAT':
                do_comp = self._query.item_with_uuid(coverage_uuid, model_name='AdH', unique_name='MaterialConceptual')
                comp = MaterialConceptualComponent(do_comp.main_file)
                inputs = {'component': comp, 'polygons': True}
            else:
                self._logger.warning('Unknown coverage type.')
            self._query.load_component_ids(**inputs)
            coverage_data.append((cov_geom, comp))
        return coverage_data
