"""Class for running the generate roughness dataset command in a feedback dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.ewn.dmi.tool_runner_input_queries import ToolRunnerInputQueries
from xms.ewn.tools.ewn_dataset_generator import EwnDatasetGenerator
from xms.ewn.tools.runners import runner_util


class GenerateRoughnessDatasetRunner(QThread):
    """Class for running the generate roughness dataset command in a feedback dialog."""
    processing_finished = Signal()

    def __init__(self, query, inputs):
        """Constructor for class.

        Args:
            query (:obj:`xms.api.dmi.Query`): query for xms communication
            inputs (:obj:`dict`): User input data from dialog
        """
        super().__init__()
        self._inputs = inputs
        self._logger = logging.getLogger('xms.ewn')
        self._queries = ToolRunnerInputQueries(query)
        self._grid = None
        self._poly_data_list = []
        self._initial_values = []
        self._dset_locations = None
        self._cov_atts = {}
        self.tool = None

    def run(self):
        """Generates a roughness dataset from EWN polygons."""
        try:
            self._retrieve_xms_data()
            self.tool = EwnDatasetGenerator(
                self._poly_data_list, self._grid, self._inputs['target_geometry'], self._initial_values,
                self._dset_locations
            )
            self.tool.generate_roughness_dataset(self._cov_atts, self._inputs['output_dset_name'])
        except:  # noqa
            self._logger.exception('Error generating roughness data set.')
        finally:
            self.processing_finished.emit()

    def _retrieve_xms_data(self):
        """Get all input data needed for generating the dataset."""
        roughness_input = self._queries.get_generate_roughness_dataset_input(self._inputs)
        self._grid, coverage_data, self._initial_values, self._dset_locations = roughness_input
        for coverage in coverage_data:
            self._cov_atts[coverage[0].uuid] = coverage[1].data
            self._poly_data_list.extend(runner_util.get_ewn_polygon_input(coverage[0], coverage[1], True))
        self._ensure_ewn_feature_exists()

    def _ensure_ewn_feature_exists(self):
        """Log an error if there are no input EWN feature polygons."""
        if not self._poly_data_list:
            self._logger.error('At least one EWN feature polygon must be defined.')
            raise RuntimeError


def generate_roughness_dataset_with_feedback(inputs, query, parent):
    """Run the Generate Roughness Data Set command with a feedback dialog.

    Args:
        inputs (:obj:`dict`): User input data from dialog
        query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object
        parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container

    Returns:
        (:obj:`tuple (bool, EwnDatasetGenerator)`):

            False if an error level message was logged during the operation,
            the generator object for the operation (contains output data)
    """
    worker = GenerateRoughnessDatasetRunner(query, inputs)
    error_str = 'Error(s) encountered generating roughness data set. Review log output for more details.'
    warning_str = 'Warning(s) encountered generating roughness data set. Review log output for more details.'
    display_text = {
        'title': 'Generate Roughness Data Set',
        'working_prompt': 'Generating roughness data set. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully generated roughness data set.',
        'note': '',
        'auto_load': 'Close this dialog when operation complete.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker.tool
