"""Class for running the insert ewn feature command in a feedback dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import uuid

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
import xms.ewn.data.ewn_cov_data_consts as consts
from xms.ewn.dmi.tool_runner_input_queries import ToolRunnerInputQueries
from xms.ewn.tools.insert_ewn_features import InsertEwnFeatures
from xms.ewn.tools.runners import runner_util


class InsertEwnFeatureRunner(QThread):
    """Class for running the insert ewn feature command in a feedback dialog."""
    processing_finished = Signal()

    def __init__(self, query, grid_uuid, lock_dataset_uuid, coverage_uuids, bias):
        """Constructor for class.

        Args:
            query (:obj:`xms.api.dmi.Query`): query for xms communication
            coverage_uuids (:obj:`list[str]`): UUIDs of the input coverages
            lock_dataset_uuid (:obj:`str`): UUID for the lock coverage
            grid_uuid (:obj:`str`): UUID of the input geometry
            bias (:obj:`float`): bias for xms.mesher
        """
        super().__init__()
        self._coverage_uuids = coverage_uuids
        self._grid_uuid = grid_uuid
        self._bias = bias
        self._is_geographic = False
        self._logger = logging.getLogger('xms.ewn')
        self._queries = ToolRunnerInputQueries(query)
        self._grid = None
        self._cogrid = None
        self._poly_data_list = []
        self._arc_data_list = []
        self.tool = None
        self.projection = None
        self._lock_dataset_uuid = lock_dataset_uuid
        self._lock_dataset = None
        self._is_cartesian = False

    def run(self):
        """Inserts ewn features into an existing mesh."""
        try:
            self._retrieve_xms_data()
            self.tool = InsertEwnFeatures(
                self._poly_data_list + self._arc_data_list,
                self._grid,
                self._is_geographic,
                bias=1 - self._bias,
                lock_dataset=self._lock_dataset,
                is_cartesian=self._is_cartesian,
            )
            self.tool.insert_features()

            if self._is_cartesian:
                new_elevations = list([x for x in self._cogrid.cell_elevations])
                for cell, elev, method in self.tool.cgrid_elevations:
                    if method == consts.ELEVATION_METHOD_Z_OFFSET:
                        new_elevations[cell] += elev
                    else:
                        new_elevations[cell] = elev
                self._cogrid.cell_elevations = new_elevations
                self._cogrid.uuid = str(uuid.uuid4())
                self.tool.out_ugrid = self._cogrid
        except:  # noqa
            self._logger.exception('Error inserting EWN features.')
        finally:
            self.processing_finished.emit()

    def _retrieve_xms_data(self):
        """Get all input data needed for the insertion operation."""
        self._cogrid, coverage_data, self._is_cartesian = self._queries.get_insert_features_input(
            self._grid_uuid, self._coverage_uuids
        )
        self._grid = self._cogrid.ugrid
        self._lock_dataset = self._queries.get_lock_dataset(self._lock_dataset_uuid)
        self._is_geographic = self._queries.is_geographic
        self.projection = self._queries.projection

        for coverage in coverage_data:
            self._poly_data_list.extend(runner_util.get_ewn_polygon_input(coverage[0], coverage[1], False))
            self._arc_data_list.extend(runner_util.get_ewn_arc_input(coverage[0], coverage[1]))
        self._ensure_ewn_feature_exists()

    def _ensure_ewn_feature_exists(self):
        """Log an error if there are no input EWN feature polygons."""
        if not self._poly_data_list and not self._arc_data_list:
            self._logger.error('At least one EWN feature must be defined.')
            raise RuntimeError


def insert_ewn_features_with_feedback(target_geometry, lock_dataset, ewn_coverages, bias, query, parent):
    """Run the Insert EWN Features command with a feedback dialog.

    Args:
        target_geometry (:obj:`str`): UUID of the target geometry
        ewn_coverages (:obj:`list[str]`): UUIDs of the input EWN Feature coverages
        bias (:obj:`float`): bias for xms.mesher
        query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object
        parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container

    Returns:
        (:obj:`tuple (bool, InsertEwnFeatureRunner)`):

            False if an error level message was logged during the operation,
            the runner object for the operation (contains output data)
    """
    worker = InsertEwnFeatureRunner(query, target_geometry, lock_dataset, ewn_coverages, bias)
    error_str = 'Error(s) encountered inserting EWN features. Review log output for more details.'
    warning_str = 'Warning(s) encountered inserting EWN features. Review log output for more details.'
    display_text = {
        'title': 'Insert EWN Features',
        'working_prompt': 'Applying EWN Features to mesh. Please wait...',
        'error_prompt': error_str,
        'warning_prompt': warning_str,
        'success_prompt': 'Successfully inserted EWN Features.',
        'note': '',
        'auto_load': 'Close this dialog automatically when insertion of features is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.ewn', worker, parent)
    feedback_dlg.exec()
    return LogEchoQSignalStream.logged_error, worker
