"""Module to perform xms.api Query operations for the GenCade simulation component."""

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.gencade.components.grid_component import GridComponent


class SimQueries:
    """Performs xms.api calls for the sim component."""

    def __init__(self, sim_comp, query):
        """Initialize the API helper.

        Args:
            sim_comp (:obj:`SimComponent`): The component instance to perform API call for
            query (:obj:`Query`): Object for communicating with XMS
        """
        self.sim_comp = sim_comp
        self.query = query

    def get_grid_mapping_data(self):
        """Query for the XMS data required for mapping a 1D GenCade grid to the simulation.

        Data dictionary contains:
        ::
            {
                grid_cov = data_objects.parameters.Coverage
                grid_comp = data_objects.parameters.Component
                mapped_grids = []
                sim_uuid = :obj:str
                display_projection = str
                error = bool
            }

        Returns:
            (:obj:`dict`): A data dictionary containing the tidal constituent mapping data from XMS
        """
        grid_data = {
            'grid_cov': None,
            'grid_comp': None,  # Python component
            'sim_uuid': '',
            'wkt': '',
            'error': False,
        }

        try:
            # Find the parent GenCade simulation and store off the current display projection.
            grid_data['sim_uuid'] = self.query.parent_item_uuid()
            grid_data['wkt'] = self.query.display_projection.well_known_text
            sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, grid_data['sim_uuid'])
            if not sim_item:
                logging.getLogger('xms.gencade').error('Unable to retrieve GenCade simulation.')
                grid_data['error'] = True
            else:
                # Find the taken 1D grid coverage (should be one and only one).
                grid_item = tree_util.descendants_of_type(tree_root=sim_item, xms_types=['TI_COVER_PTR'],
                                                          allow_pointers=True, only_first=True,
                                                          coverage_type='GenCade Grid', model_name='GenCade')
                if not grid_item:
                    logging.getLogger('xms.gencade').error('Unable to retrieve linked GenCade grid coverage.')
                    grid_data['error'] = True
                else:  # Get the coverage geometry
                    grid_data['grid_cov'] = self.query.item_with_uuid(grid_item.uuid)
                    # Get the coverage's hidden component.
                    do_comp = self.query.item_with_uuid(grid_item.uuid, unique_name='GridComponent',
                                                        model_name='GenCade')
                    grid_data['grid_comp'] = GridComponent(do_comp.main_file)
                    self.query.load_component_ids(grid_data['grid_comp'], points=True)

                # Find any old mapped 1D grids (should be 0 or 1)
                grid_data['mapped_grids'] = tree_util.descendants_of_type(tree_root=sim_item,
                                                                          xms_types=['TI_COMPONENT'],
                                                                          unique_name='MappedGridComponent',
                                                                          model_name='GenCade')
        except Exception:
            logging.getLogger('xms.gencade').error('Unable to apply 1D grid coverage to the GenCade simulation.')
            grid_data['error'] = True
        return grid_data
