"""Qt table model using a pandas.DataFrame for storage."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules


def parse_datetime(token, time_format=None):
    """Datetime parser for Gencade formatted datetime strings.

    Args:
        token (:obj:`str`): The datetime string to parse
        time_format (:obj:`str`): The strptime format to use if not standard Gencade format (GUI)

    Returns:
        (:obj:`Union[pd.Timestamp, None]`): The parsed datetime or None if unable to parse
    """
    # Custom absolute datetime format specified
    if time_format:
        try:
            return pd.to_datetime(token, format=time_format)
        except ValueError:
            return None


class DatetimePandasModel(QxPandasTableModel):
    """Class derived from QxPandasTableModel to specialize the header data and datetime formatting."""

    def __init__(self, data_frame, time_format, parent):
        """Initializes the class.

        Args:
            data_frame (:obj:`DataFrame`): The pandas DataFrame
            time_format (:obj:`str`): The absolute datetime format to use (should use strftime specifiers)
            parent (:obj:`BcSeriesEditor`): The parent dialog
        """
        super().__init__(data_frame, parent)
        self._time_format = time_format

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if role == Qt.DisplayRole:
            if orientation == Qt.Horizontal:
                # if section in [0, 1]:
                #     return self.parent().get_time_units_label()
                # else:
                #     return self.data_frame.columns[section]
                return self.data_frame.columns[section]
            else:  # 1-based row number for vertical header
                return section + 1
        return super().headerData(section, orientation, role)

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (:obj:`int`): The role.

        Returns:
            The data at index, or None.
        """
        if role in [Qt.DisplayRole, Qt.EditRole]:
            col_idx = index.column()
            val = self.data_frame.iat[index.row(), col_idx]
            if self._x_is_time and col_idx == 0:  # Format the time
                if type(val) is pd.Timestamp:  # This value is a datetime, format it based on current SMS preferences.
                    return pd.to_datetime(val).strftime(self._time_format)
                return str(val)  # Float offset
            val = str(val)
            if val in ['nan', 'None']:
                return ''  # Show missing data as blank cells in the spreadsheet
            return val
        return None

    def setData(self, index, value, role=Qt.EditRole):  # noqa: N802, C901
        """Adjust the data (set it to <value>) depending on index and role.

        Args:
            index (:obj:`QModelIndex`): The index.
            value: The value.
            role (:obj:`int`): The role.

        Returns:
            (:obj:`bool`): True if successful; otherwise False.
        """
        if not index.isValid():
            return False

        col_idx = index.column()
        row_idx = index.row()
        if role == Qt.EditRole:
            old_value = self.data_frame.iat[row_idx, col_idx]
            # Will try to parse as float first, so should work even if not time
            value = parse_datetime(value, self._time_format)
            is_null = pd.isnull(value)
            emit = False
            # Don't allow user to set/paste an invalid datetime
            if value != old_value and not is_null:
                self.data_frame.iat[row_idx, col_idx] = value
                emit = True
            if emit and not self.parent().ui.tbl_xy_data.pasting and not self.parent().in_setup:
                self.dataChanged.emit(index, index)
            return True
        return False
