"""MaterialComponent class."""

# 1. Standard Python modules
import os
from pathlib import Path
import random
from typing import Optional
import warnings

# 2. Third party modules
from PySide2.QtGui import QColor
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.components.display.display_options_io import write_display_options_to_json
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.gmi.components.coverage_component import CoverageComponent, DEFAULT_COLORS
from xms.gmi.data.generic_model import GenericModel, Section, UNASSIGNED_MATERIAL_ID
from xms.gmi.data.material_data import MaterialData
from xms.gmi.gui.material_group_set_dialog import MaterialGroupSetDialog


class MaterialComponent(CoverageComponent):
    """
    A hidden Dynamic Model Interface (DMI) component for a GMI material coverage.

    Specializes `CoverageComponent` for materials. Most functionality is
    defined in the base class.
    """
    def __init__(
        self,
        main_file: str | Path,
        generic_model: Optional[GenericModel] = None,
    ):
        """
        Initialize the MaterialComponent.

        Args:
            main_file: The main file associated with this component.
            generic_model: Parameter definitions.
        """
        super().__init__(main_file, generic_model)
        warnings.warn(
            'The GMI material component is non-functional. The one in Hydro-AS will eventually be migrated here.',
            category=DeprecationWarning,
            stacklevel=2
        )

        self.point_commands = []
        self.arc_commands = []
        self.polygon_commands = [('Assign Materials...', 'open_assign_material')]
        self.polygon_dlg_title = 'Material Properties'

        self.unassigned_material_group = UNASSIGNED_MATERIAL_ID

    def _get_data_with_model(self, model: GenericModel):
        """
        Get a data manager for this component.

        Derived classes can override this to provide their own data manager. Implementations will typically look
        something like

        ```
        def _get_data(self):
            return SomeDataManager(self.main_file, model)
        ```

        GMI data classes define this because they have to be ready to deal with arbitrary models, but derived models
        generally won't need this because they only have a single model. They will generally prefer to define
        `self._get_data` instead. That alternative is the same as this, except it has no `model` parameter.
        """
        return MaterialData(self.main_file, model)

    def _make_category_dict(self):
        """
        Initialize some internal data.

        This is needed during the constructor, and derived classes might want to change it, but passing it in
        seems inappropriate since the user doesn't care, so it's a method to allow overriding.
        """
        category_dict = super()._make_category_dict()
        category_dict.pop(TargetType.point, None)  # Material coverages don't have points or arcs.
        category_dict.pop(TargetType.arc, None)
        category_dict[TargetType.polygon]['section'] = self.data.generic_model.material_parameters

        return category_dict

    def _add_feature(self, target_type: TargetType, values: str, active_group) -> int:
        """
        Handle adding values for a feature.

        Called by self._assign_feature() to let derived classes change how this works.

        Args:
            target_type: Target the feature applies to.
            values: Values to assign.
            active_group: The currently active group.

        Returns:
            A component ID to associate with the features.
        """
        model = self.data.generic_model
        parameters = model.material_parameters
        parameters.restore_values(values)
        values = parameters.extract_group_activity()
        parameters.deactivate_groups()
        self.data.generic_model = model
        self._category_dict[target_type]['section'] = self.data.generic_model.material_parameters
        return self.data.add_feature(target_type, values, active_group)

    def _update_display_options(self, section: Section):
        """
        Update the display options.

        Adds new ones for newly created materials, and cleans up old ones from deleted materials.

        Args:
            section: Section containing the current materials.
        """
        cat_list = self._read_display_options()
        categories = cat_list[0]
        have_ids = {category.id for category in categories.categories}
        need_ids = {group_name for group_name in section.group_names}

        remove_ids = have_ids - need_ids
        categories.categories = [category for category in categories.categories if category.id not in remove_ids]

        add_ids = need_ids - have_ids
        for add_id in add_ids:
            categories.categories.append(CategoryDisplayOption())
            c = categories.categories[-1]
            c.options = self._category_dict[TargetType.polygon]['option_class']()
            c.description = section.group(add_id).label
            c.id = add_id
            prefix = self._category_dict[TargetType.polygon]['id_file_prefix']
            c.file = f'{prefix}cat{add_id}.ids'
            clr = random.choice(DEFAULT_COLORS)
            c.options.color = QColor(clr[0], clr[1], clr[2], 255)
        self._write_display_options(cat_list)

    def open_assign_material(
        self,
        query: Query,
        params: list[dict],
        parent: QWidget,
    ) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Run the Assign Material dialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        window_name = f'{self.module_name}.polygon_dlg'
        window_title = self.polygon_dlg_title
        target_type = TargetType.polygon

        self._query = query
        comp_id = self._unpack_xms_data(params[0], target_type)
        if not self.selected_att_ids:
            return [('INFO', 'No feature selected. Select one or more features to assign boundary conditions.')], []

        section = self._category_dict[target_type]['section']
        values = self.data.material_values
        section.restore_values(values)
        section.deactivate_groups()

        active_group = self.data.feature_type(TargetType.polygon, comp_id)
        if active_group in section.group_names:
            section.group(active_group).is_active = True

        dlg = MaterialGroupSetDialog(
            parent=parent,
            section=section,
            get_curve=self.data.get_curve,
            add_curve=self.data.add_curve,
            is_interior=False,
            dlg_name=window_name,
            window_title=window_title,
            multi_select_message=self._dlg_message,
            show_groups=self._show_groups,
            enable_unchecked_groups=False
        )
        if dlg.exec():
            active_group = dlg.section.active_group_name(self._inactive_group_name, self._multiple_active_group_name)
            comp_id = self._add_feature(target_type, '', active_group)
            dlg.section.deactivate_groups()

            self.data.material_values = dlg.section.extract_values()

            dlg.section.clear_values()
            model = self.data.generic_model
            model.material_parameters = dlg.section
            self.data.generic_model = model

            self._update_display_options(dlg.section)

            # Associate all selected features with the new component id.
            for feature_id in self.selected_att_ids:
                self.update_component_id(target_type, feature_id, comp_id)
            self.refresh_component_ids(comp_id, target_type)
            self.data.commit()
        return [], []

    def _ensure_display_option_files_exist(self):
        """
        Copy default point, arc, and polygon display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists.

        Should only be called by the unmapped coverage on creation.
        """
        target_type = TargetType.polygon
        value = self._category_dict[target_type]
        disp_opts_file = value['disp_opts_file']
        if not os.path.exists(disp_opts_file):
            # We always have the unassigned material, so this should always be initialized.
            categories = self._default_display_categories(target_type)
            categories.file = disp_opts_file
            categories.comp_uuid = self.uuid
            categories.categories[0].is_unassigned_category = True
            write_display_options_to_json(disp_opts_file, categories)

    def _section(self, target_type: TargetType) -> Section:
        """
        Get the section to be used by the feature dialog for a given target type.

        Args:
            target_type: Which target the section should be for.

        Returns:
            The section.
        """
        return self.data.generic_model.material_parameters
