"""Parameter widget classes."""

from __future__ import annotations

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
from typing import Optional

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QDateTime, QModelIndex, QObject, Qt, Signal
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QDateTimeEdit, QHBoxLayout, QLabel, QLineEdit, QMessageBox, QPushButton, QSpinBox,
    QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DatasetItem
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename, get_open_foldername, get_save_filename
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.table_with_tool_bar import TableWithToolBar

# 4. Local modules
from xms.gmi.data.generic_model import Curve, Group, Parameter, Type
from xms.gmi.gui.dataset_callback import DatasetCallback, DatasetRequest


def create_float_widget(parent_widget: ParameterWidget, value: float, on_text_changed, custom_dlg: QObject):
    """Common code for creating a float edit field widget.

    Args:
        parent_widget: The parent widget.
        value (float): The value.
        on_text_changed: Function to call when the value is edited.
        custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
    """
    if custom_dlg:
        widget_name = widget_name_from_parameter(QLineEdit, parent_widget)
        widget = custom_dlg.find_widget(QLineEdit, widget_name)
        if widget is None:
            raise RuntimeError(f'{widget_name} widget not found.')
    else:
        widget = QLineEdit()
    validator = QxDoubleValidator(parent=parent_widget)
    widget.setValidator(validator)
    widget.setText(str(value))
    validator.setBottom(parent_widget.parameter.low)
    validator.setTop(parent_widget.parameter.high)
    widget.textChanged.connect(on_text_changed)
    return widget


class ParameterWidget(QWidget):
    """Class representing a widget for a parameter."""
    parent_updated = Signal()

    def __init__(
        self,
        group_name: str,
        parameter_name: str,
        parameter: Parameter,
        dataset_callback: Optional[DatasetCallback],
        custom_dlg: QObject = None,
        skip_label=False,
    ):
        """
        Initialize the ParameterWidget.

        Args:
            group_name: Name of the group the widget is in.
            parameter_name: Name of the parameter.
            parameter: The parameter.
            dataset_callback: Callback for managing dataset parameters. Not used in this class.
            custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
            skip_label: Whether this is the boolean widget.
        """
        super().__init__()
        self.gmi_parent: Optional[ParameterWidget] = None  # parent conflicts with QWidget.parent

        self.group_name = group_name
        self.parameter_name = parameter_name
        self.parameter = parameter
        self.custom_dlg = custom_dlg
        self.widgets: dict[str, QWidget] = {}
        if not self.custom_dlg:
            self.setLayout(QVBoxLayout())
            self.layout().setContentsMargins(0, 0, 0, 0)

            if not skip_label and parameter.label:
                # Add an asterisk if the parameter is required
                label_text = f'{parameter.label}:' if parameter.required is False else f'* {parameter.label}:'
                self.layout().addWidget(QLabel(label_text))

        self.init_widget()

        if self.parameter.description:
            if self.custom_dlg:
                for _, widget in self.widgets.items():
                    if not widget.toolTip():
                        widget.setToolTip(f'<span>{self.parameter.description}</span>')
            else:
                # Make it rich text so it will get word wrapped (see https://stackoverflow.com/questions/4795757)
                self.setToolTip(f'<span>{self.parameter.description}</span>')

    def add_widget(self, name: str, widget: QWidget):
        """
        Add a child widget to the parameter widget.

        Once added, self.widgets[name] will contain the added widget.

        Args:
            name: Name of the widget.
            widget: The widget.
        """
        self.widgets[name] = widget
        if not self.custom_dlg:
            self.layout().addWidget(widget)

    def find_widget(self, widget_type) -> QObject:
        """Finds the widget of type object_class with object name that matches our convention."""
        widget_name = widget_name_from_parameter(widget_type, self)
        widget = self.custom_dlg.find_widget(widget_type, widget_name)
        if widget is None:
            raise RuntimeError(f'{widget_name} widget not found.')
        return widget

    def update_instantiation(self, new_value):
        """
        Update the instantiation of the parameter this widget is associated with.

        Also informs any interested parties of the update.

        Args:
            new_value: The instantiation's new value.
        """
        self.parameter.value = new_value
        # noinspection PyUnresolvedReferences
        self.parent_updated.emit()

    def add_parent(self, parent: ParameterWidget):
        """
        Associate this widget with another one, such that its availability depends on the value of the parent.

        Args:
            parent: The parent widget to associate with.
        """
        self.gmi_parent = parent
        # noinspection PyUnresolvedReferences
        parent.parent_updated.connect(self.on_parent_updated)
        self.on_parent_updated()

    @property
    def gmi_enabled(self) -> bool:
        """Check whether this widget should be enabled."""
        if not self.gmi_parent:
            return True
        if not self.gmi_parent.gmi_enabled:
            return False

        value = self.gmi_parent.parameter.value
        enabled = self.parameter.dependency_flags[value]
        return enabled

    def on_parent_updated(self):
        """Inform this widget that its parent's value has been updated."""
        if self.custom_dlg:
            for widget in self.widgets.values():
                widget.setEnabled(self.gmi_enabled)
        else:
            self.setVisible(self.gmi_enabled)
        # noinspection PyUnresolvedReferences
        self.parent_updated.emit()


class BooleanWidget(ParameterWidget):
    """Class representing a boolean parameter's widget."""
    def __init__(
        self,
        group_name: str,
        parameter_name: str,
        parameter: Parameter,
        dataset_callback: Optional[DatasetCallback],
        custom_dlg: QObject = None
    ):
        """
        Initialize the BooleanWidget.

        Args:
            group_name: Name of the group the widget is in.
            parameter_name: Name of the parameter.
            parameter: The parameter.
            dataset_callback: Ignored.
            custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
        """
        super().__init__(group_name, parameter_name, parameter, dataset_callback, custom_dlg, skip_label=True)

    def init_widget(self):
        """Perform initialization specific to a boolean widget."""
        if self.custom_dlg:
            widget = self.find_widget(QCheckBox)
        else:
            widget = QCheckBox(self.parameter.label)
        widget.setChecked(self.parameter.value)
        # noinspection PyUnresolvedReferences
        widget.stateChanged.connect(self.handle_update)
        self.add_widget('checkbox', widget)

    def handle_update(self, new_value):
        """
        Slot for when the widget's value changes.

        Args:
            new_value: The widget's new value.
        """
        self.update_instantiation(new_value == Qt.Checked)


class CheckboxCurveWidget(ParameterWidget):
    """Class representing a curve with a checkbox."""
    def init_widget(self):
        """Perform initialization specific to this widget."""
        if self.custom_dlg:
            checkbox = self.find_widget(QCheckBox)
        else:
            checkbox = QCheckBox()
        checkbox.setText(self.parameter.checkbox_label)
        checkbox.setToolTip(self.parameter.checkbox_description)
        checkbox.setCheckState(Qt.Checked if self.parameter.value[0] else Qt.Unchecked)
        # noinspection PyUnresolvedReferences
        checkbox.stateChanged.connect(self._on_check_changed)
        self.add_widget('selector', checkbox)

        if self.custom_dlg:
            edit_button = self.find_widget(QPushButton)
        else:
            edit_button = QPushButton("Edit XY series...")
        # noinspection PyUnresolvedReferences
        edit_button.clicked.connect(self.on_edit_curve)
        self.add_widget('curve_edit', edit_button)

        value = self.parameter.value[1] if Curve.CONSTANT in self.parameter.mode else self.parameter.default[1]
        edit_field = create_float_widget(self, value, on_text_changed=self._on_text_changed, custom_dlg=self.custom_dlg)
        self.add_widget('float_edit', edit_field)

        # noinspection PyUnresolvedReferences
        checkbox.stateChanged.emit(checkbox.checkState())

    def _on_check_changed(self, new_state: int):
        """
        Slot for when the checkbox state changes.

        Args:
            new_state: The new check state of the checkbox.
        """
        new_state = (new_state == Qt.Checked)
        self.parameter.mode = new_state
        new_mode = self.parameter.mode

        if Curve.CURVE in new_mode:
            self.widgets['curve_edit'].setEnabled(True) if self.custom_dlg else self.widgets['curve_edit'].show()
            self.widgets['float_edit'].setEnabled(False) if self.custom_dlg else self.widgets['float_edit'].hide()
        else:
            self.widgets['curve_edit'].setEnabled(False) if self.custom_dlg else self.widgets['curve_edit'].hide()
            self.widgets['float_edit'].setEnabled(True) if self.custom_dlg else self.widgets['float_edit'].show()
        self.update_instantiation(self.parameter.value)

    def _on_text_changed(self, new_value: str):
        """
        Slot for when the float field's text changes.

        Args:
            new_value: New value of the float field.
        """
        check_box = self.widgets['selector']
        mode_index = 1 if check_box.checkState() == Qt.Checked else 0

        with contextlib.suppress(ValueError):
            self.update_instantiation((mode_index, float(new_value)))

    def on_edit_curve(self):
        """Slot for when the float/curve's curve edit button is pushed."""
        check_box = self.widgets['selector']
        mode_index = 1 if check_box.checkState() == Qt.Checked else 0
        current_mode = self.parameter.modes[mode_index]
        use_dates = Curve.DATES in current_mode
        stair_step = Curve.STAIRS in current_mode

        xy_id = self.parameter.value[1] if self.parameter.value is not None else -1
        new_id = check_box.window().on_btn_click_xy(xy_id, self.parameter.axis_titles, use_dates, stair_step)
        self.update_instantiation((mode_index, new_id))


class DatasetWidget(ParameterWidget):
    """Class representing a dataset picker."""
    def __init__(
        self,
        group_name: str,
        parameter_name: str,
        parameter: Parameter,
        dataset_callback: Optional[DatasetCallback],
        custom_dlg: QObject = None,
        skip_label=False,
    ):
        """
        Initialize the DatasetWidget.

        Args:
            group_name: Name of the group the widget is in.
            parameter_name: Name of the parameter.
            parameter: The parameter.
            dataset_callback: Callback for managing dataset parameters.
                Only needed if group contains dataset parameters.
            custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
            skip_label: Whether this is the boolean widget.
        """
        # super().__init__() calls self.init_widget, which expects this to be initialized.
        self._dataset_callback = dataset_callback
        super().__init__(group_name, parameter_name, parameter, dataset_callback, custom_dlg, skip_label)

    def init_widget(self):
        """Perform initialization specific to this widget."""
        container = QWidget()
        container.setLayout(QHBoxLayout())
        container.layout().setContentsMargins(0, 0, 0, 0)

        if self.custom_dlg:
            button = self.find_widget(QPushButton)
        else:
            button = QPushButton('Select...')
        # noinspection PyUnresolvedReferences
        button.clicked.connect(self._on_pick_dataset)
        self.widgets['button'] = button
        if not self.custom_dlg:
            container.layout().addWidget(button)

        if self.custom_dlg:
            label = self.find_widget(QLabel)
        else:
            label = QLabel()
        display_text = self._dataset_callback(DatasetRequest.GetLabel, self.parameter) or '(none selected)'
        label.setText(display_text)
        self.widgets['label'] = label
        if not self.custom_dlg:
            container.layout().addWidget(label)

        self.add_widget('container_widget', container)

    def _on_pick_dataset(self):
        """Handle when the button is clicked to pick a dataset."""
        tree = self._dataset_callback(DatasetRequest.GetTree, self.parameter)
        if tree is None or not tree.children:
            msg = QMessageBox(QMessageBox.Information, 'Error', 'No selectable datasets found.', QMessageBox.Ok, self)
            msg.exec()
            return

        dataset_selector = TreeItemSelectorDlg(
            title=f'Select dataset for {self.parameter.label}',
            target_type=DatasetItem,
            pe_tree=tree,
            previous_selection=self.parameter.value,
            show_root=False,
            parent=self
        )
        if dataset_selector.exec() and (dataset_uuid := dataset_selector.get_selected_item_uuid()):
            self.parameter.value = dataset_uuid
            label = self._dataset_callback(DatasetRequest.GetLabel, self.parameter)
            self.widgets['label'].setText(label)


class SeriesWidget(ParameterWidget):
    """Class representing a series parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a curve widget."""
        if self.custom_dlg:
            widget = self.find_widget(QPushButton)
        else:
            widget = QPushButton("Edit XY Series...")
        # noinspection PyUnresolvedReferences
        widget.clicked.connect(self.on_edit_curve)
        self.add_widget('curve_edit', widget)

    def on_edit_curve(self):
        """Slot for when the curve edit button is pushed."""
        xy_id = self.parameter.value
        window = self.widgets['curve_edit'].window()
        new_id = window.on_btn_click_xy(xy_id, self.parameter.axis_titles, self.parameter.use_dates, False)
        self.update_instantiation(new_id)


class CurveWidget(SeriesWidget):
    """Class representing a curve parameter's widget."""
    def on_edit_curve(self):
        """Slot for when the curve edit button is pushed."""
        x, y = self.parameter.value
        axis_titles = self.parameter.axis_titles

        pd_dict = {
            axis_titles[0]: x,
            axis_titles[1]: y,
        }
        df = pd.DataFrame(pd_dict)
        if self.parameter.use_dates:
            df[axis_titles[0]] = df[axis_titles[0]].astype(dtype='datetime64[ns]')
        df.index += 1

        dlg = XySeriesEditor(df, '', parent=self)
        if not dlg.exec():
            return

        df = dlg.model.data_frame

        if self.parameter.use_dates:
            df[axis_titles[0]] = df[axis_titles[0]].astype(dtype=str)

        if len(df) < 1:
            x = [0.0]
            y = [0.0]
        else:
            x = df[axis_titles[0]].to_list()
            y = df[axis_titles[1]].to_list()

        self.update_instantiation((x, y))


class DateTimeWidget(ParameterWidget):
    """Class representing a datetime parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a date/time widget."""
        q_date_time = QDateTime.fromString(self.parameter.value, Qt.ISODate)
        if self.custom_dlg:
            widget = self.find_widget(QDateTimeEdit)
            widget.setDateTime(q_date_time)
        else:
            widget = QDateTimeEdit(q_date_time)
        widget.setCalendarPopup(self.parameter.calendar_popup)
        widget.dateTimeChanged.connect(self.on_date_time_changed)
        self.add_widget('date_time_edit', widget)

    def on_date_time_changed(self, q_date_time: QDateTime):
        """Slot called when the date or time is changed."""
        date_time = q_date_time.toPython()
        self.update_instantiation(date_time.isoformat())


class FloatWidget(ParameterWidget):
    """Class representing a float parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a float widget."""
        widget = create_float_widget(
            self, value=self.parameter.value, on_text_changed=self.on_text_changed, custom_dlg=self.custom_dlg
        )
        self.add_widget('float_edit', widget)

    def on_text_changed(self, new_value):
        """Text changed signal.

        Args:
            new_value (str): new value in the edit field
        """
        with contextlib.suppress(ValueError):
            new_value = float(new_value)
            self.update_instantiation(new_value)


class FloatOrCurveWidget(ParameterWidget):
    """Class representing a float/curve parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a float/curve widget."""
        if self.parameter.value[0] == 'FLOAT':
            mode = 'Float'
            float_value = self.parameter.value[1]
        elif self.parameter.value[0] == 'CURVE':
            mode = 'Curve'
            float_value = 0.0
        else:  # pragma: nocover
            raise AssertionError("Unknown mode")  # New mode added without adding support here

        if self.custom_dlg:
            selector = self.find_widget(QComboBox)
        else:
            selector = QComboBox()
        selector.addItems(['Float', 'Curve'])
        selector.setCurrentText(mode)
        selector.currentTextChanged.connect(self.mode_changed)
        self.add_widget('selector', selector)

        float_edit = create_float_widget(
            self, value=float_value, on_text_changed=self.on_edit_float, custom_dlg=self.custom_dlg
        )
        self.add_widget('float_edit', float_edit)

        if self.custom_dlg:
            edit_button = self.find_widget(QPushButton)
        else:
            edit_button = QPushButton("Edit XY series...")
        # noinspection PyUnresolvedReferences
        edit_button.clicked.connect(self.on_edit_curve)
        self.add_widget('curve_edit', edit_button)

        self.mode_changed(mode)

    def mode_changed(self, new_mode):
        """
        Slot for when the widget's mode is changed.

        Args:
            new_mode: The new mode. Must be either 'Float' or 'Curve'.
        """
        if new_mode == 'Float':
            self.widgets['curve_edit'].setEnabled(False) if self.custom_dlg else self.widgets['curve_edit'].hide()
            self.widgets['float_edit'].setEnabled(True) if self.custom_dlg else self.widgets['float_edit'].show()
        elif new_mode == 'Curve':
            self.widgets['curve_edit'].setEnabled(True) if self.custom_dlg else self.widgets['curve_edit'].show()
            self.widgets['float_edit'].setEnabled(False) if self.custom_dlg else self.widgets['float_edit'].hide()
        else:  # pragma: nocover
            raise AssertionError("Unknown mode")  # New mode added without adding support

    def on_edit_curve(self):
        """Slot for when the float/curve's curve edit button is pushed."""
        xy_id = self.parameter.value[1] if self.parameter.value is not None else -1
        window = self.widgets['curve_edit'].window()
        new_id = window.on_btn_click_xy(xy_id, self.parameter.axis_titles, self.parameter.use_dates, False)
        self.update_instantiation(['CURVE', new_id])

    def on_edit_float(self, new_value):
        """Slot for when the float/curve's float value is edited."""
        with contextlib.suppress(ValueError):
            new_value = float(new_value)
            self.update_instantiation(['FLOAT', new_value])


class FloatCurveWidget(FloatOrCurveWidget):
    """Class representing a float/curve parameter's widget."""
    def mode_changed(self, new_mode):
        """
        Slot for when the widget's mode is changed.

        Args:
            new_mode: The new mode. Must be either 'Float' or 'Curve'.
        """
        new_value = list(self.parameter.value)

        if new_mode == 'Float':
            self.widgets['curve_edit'].setEnabled(False) if self.custom_dlg else self.widgets['curve_edit'].hide()
            self.widgets['float_edit'].setEnabled(True) if self.custom_dlg else self.widgets['float_edit'].show()
            new_value[0] = 'FLOAT'
        elif new_mode == 'Curve':
            self.widgets['curve_edit'].setEnabled(True) if self.custom_dlg else self.widgets['curve_edit'].show()
            self.widgets['float_edit'].setEnabled(False) if self.custom_dlg else self.widgets['float_edit'].hide()
            new_value[0] = 'CURVE'
        else:  # pragma: nocover
            raise AssertionError("Unknown mode")  # New mode added without adding support

        self.update_instantiation(tuple(new_value))

    def on_edit_curve(self):
        """Slot for when the float/curve's curve edit button is pushed."""
        _mode, _float_value, x, y = self.parameter.value
        axis_titles = self.parameter.axis_titles

        pd_dict = {
            axis_titles[0]: x,
            axis_titles[1]: y,
        }
        df = pd.DataFrame(pd_dict)
        df.index += 1

        dlg = XySeriesEditor(df, '', parent=self)
        if not dlg.exec():
            return

        df = dlg.model.data_frame
        if len(df) < 1:
            x = [0.0]
            y = [0.0]
        else:
            x = df[axis_titles[0]].to_list()
            y = df[axis_titles[1]].to_list()

        float_value = self.parameter.value[1]
        self.update_instantiation(('CURVE', float_value, x, y))

    def on_edit_float(self, new_value):
        """Slot for when the float/curve's float value is edited."""
        value = list(self.parameter.value)
        value[0] = 'FLOAT'
        with contextlib.suppress(ValueError):
            value[1] = float(new_value)
        self.update_instantiation(tuple(value))


class IntegerWidget(ParameterWidget):
    """Class representing an integer parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to an integer widget."""
        if self.custom_dlg:
            widget = self.find_widget(QSpinBox)
        else:
            widget = QSpinBox()
        # widget.setButtonSymbols(QSpinBox.NoButtons)
        widget.setMinimum(self.parameter.low)
        widget.setMaximum(self.parameter.high)
        widget.setValue(self.parameter.value)
        widget.valueChanged.connect(self.update_instantiation)
        self.add_widget('integer_edit', widget)


class OptionWidget(ParameterWidget):
    """Class representing an option parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to an option widget."""
        if self.custom_dlg:
            widget = self.find_widget(QComboBox)
        else:
            widget = QComboBox()
        widget.addItems(self.parameter.options)
        widget.setCurrentText(self.parameter.value)
        widget.currentTextChanged.connect(self.update_instantiation)
        self.add_widget('option_edit', widget)


class PathWidget(ParameterWidget):
    """
    Class representing a path parameter's widget.

    This is a base class. See DirectoryWidget, InputFileWidget, and OutputFileWidget for implementations.
    """
    def init_widget(self):
        """Perform initialization specific to a file widget."""
        container = QWidget()
        container.setLayout(QHBoxLayout())
        container.layout().setContentsMargins(0, 0, 0, 0)

        if self.custom_dlg:
            path_widget = self.find_widget(QLineEdit)
        else:
            path_widget = QLineEdit()
        path_widget.setText(self.parameter.value)
        self.widgets['path_widget'] = path_widget
        if not self.custom_dlg:
            container.layout().addWidget(path_widget)

        if self.custom_dlg:
            browse_widget = self.find_widget(QPushButton)
        else:
            browse_widget = QPushButton()
        browse_widget.setText("Browse...")
        # noinspection PyUnresolvedReferences
        browse_widget.clicked.connect(self.on_browse)
        self.widgets['browse_widget'] = browse_widget
        if not self.custom_dlg:
            container.layout().addWidget(browse_widget)

        self.add_widget('container_widget', container)

    def on_browse(self):
        """Slot to handle clicking the Browse... button."""
        path = self.get_path()
        if path:
            self.widgets['path_widget'].setText(path)
            self.update_instantiation(path)


class DirectoryWidget(PathWidget):
    """Class representing a directory parameter's widget."""
    def get_path(self):
        """Get a path."""
        return get_open_foldername(self, self.parameter.label)


class InputFileWidget(PathWidget):
    """Class representing an input file parameter's widget."""
    def get_path(self):
        """Get a path."""
        return get_open_filename(self, self.parameter.label, self.parameter.file_filter)


class OutputFileWidget(PathWidget):
    """Class representing an output file parameter's widget."""
    def get_path(self):
        """Get a path."""
        return get_save_filename(self, '', self.parameter.file_filter, self.parameter.label)


class TableWidget(ParameterWidget):
    """Class representing a table parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a table widget."""
        if self.parameter.display_table:
            if self.custom_dlg:
                widget = self.find_widget(TableWithToolBar)
            else:
                widget = TableWithToolBar()
            df = self.parameter.table_definition.to_pandas(rows=self.parameter.value)
            widget.setup(self.parameter.table_definition, df)
            widget.data_changed.connect(self.on_data_changed)
            self.add_widget('table_widget', widget)
        else:
            if self.custom_dlg:
                widget = self.find_widget(QPushButton)
            else:
                widget = QPushButton("Edit Table...")
            widget.clicked.connect(self.on_edit_table)
            self.add_widget('table_button', widget)

    def on_data_changed(self, top_left: QModelIndex, bottom_right: QModelIndex) -> None:
        """Called when the table is displayed in the dialog instead of with the button and the data changes."""
        table_widget = self.widgets['table_widget']
        df = table_widget.get_values()
        for column in df:
            if df[column].dtype == np.dtype('datetime64[ns]'):
                df[column] = df[column].astype(dtype=str)
        self.update_instantiation(df.values.tolist())

    def on_edit_table(self):
        """Slot for when the table edit button is pushed."""
        p = self.parameter  # for short
        window = self.widgets['table_button'].window()
        value = window.on_btn_click_table(p.label, p.table_definition, p.value)
        if value is not None:
            self.update_instantiation(value)


class TextWidget(ParameterWidget):
    """Class representing a text parameter's widget."""
    def init_widget(self):
        """Perform initialization specific to a text widget."""
        if self.custom_dlg:
            widget = self.find_widget(QLineEdit)
            widget.setText(self.parameter.value)
        else:
            widget = QLineEdit(self.parameter.value)
        widget.textChanged.connect(self.update_instantiation)
        self.add_widget('text_edit', widget)


widget_map = {
    Type.BOOLEAN: BooleanWidget,
    Type.CHECKBOX_CURVE: CheckboxCurveWidget,
    Type.CURVE: CurveWidget,
    Type.DATASET: DatasetWidget,
    Type.DATE_TIME: DateTimeWidget,
    Type.FLOAT: FloatWidget,
    Type.FLOAT_CURVE: FloatCurveWidget,
    Type.FLOAT_OR_CURVE: FloatOrCurveWidget,
    Type.INPUT_FILE: InputFileWidget,
    Type.INPUT_FOLDER: DirectoryWidget,
    Type.INTEGER: IntegerWidget,
    Type.OPTION: OptionWidget,
    Type.OUTPUT_FILE: OutputFileWidget,
    Type.SERIES: SeriesWidget,
    Type.TEXT: TextWidget,
    Type.TABLE: TableWidget,
}


def make_widgets(
    dataset_callback: Optional[DatasetCallback],
    container_widget: QWidget,
    group_name: str,
    group: Group,
    custom_dlg: QObject = None
):
    """
    Make parameter widgets for all the parameters in a group.

    Args:
        dataset_callback: Callback for managing dataset parameters. Only needed if group contains dataset parameters.
        container_widget: Widget to add all the parameter widgets to.
        group_name: Name of the group being added.
        group: Group of parameters to make widgets for.
        custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
    """
    widgets = {}

    for parameter_name in group.parameter_names:
        widget = make_widget(dataset_callback, group_name, group, parameter_name, custom_dlg)
        if not custom_dlg:
            container_widget.layout().addWidget(widget)

        widgets[parameter_name] = widget

    for parameter_name in widgets:
        widget = widgets[parameter_name]
        if widget.parameter.parent is not None:
            parent_name = widget.parameter.parent[-1]
            parent_widget = widgets[parent_name]
            widget.add_parent(parent_widget)

    if not custom_dlg:
        container_widget.layout().addStretch(1)

    return [(name, widgets[name]) for name in widgets]


def make_widget(
    dataset_callback: Optional[DatasetCallback], group_name: str, group: Group, name: str, custom_dlg: QObject = None
):
    """
    Make a parameter widget.

    Args:
        dataset_callback: Callback for managing dataset parameters. Only needed if group contains dataset parameters.
        group_name: Name of the group being added.
        group: Group the parameter is in.
        name: Name of the parameter.
        custom_dlg: If not using SectionDialog, the custom dialog (derived from CustomDialogBase)
    """
    parameter = group.parameter(name)
    try:
        widget_builder = widget_map[parameter.parameter_type]
    except KeyError:  # pragma: nocover
        raise AssertionError("Unknown type")  # A new type was added without adding support here.

    widget = widget_builder(group_name, name, parameter, dataset_callback, custom_dlg)
    return widget


def widget_name_from_parameter(widget_type: type, parameter_widget: ParameterWidget) -> str:
    """Returns the object name of the widget, given the widget type and ParameterWidget.

    Args:
        widget_type: QCheckBox, QPushButton etc.
        parameter_widget: Something derived from ParameterWidget.

    Returns:
        (str): See description.
    """
    return widget_name_from_strings(widget_type, parameter_widget.group_name, parameter_widget.parameter_name)


def widget_name_from_strings(widget_type: type, group_name: str, parameter_name: str) -> str:
    """Returns the object name of the widget, given the widget type, group name, and parameter name.

    Args:
        widget_type: QCheckBox, QPushButton etc.
        group_name: Name of the generic model group.
        parameter_name: Name of the parameter (i.e. Parameter.parameter_name).

    Returns:
        (str): See description.
    """
    widget_type_str_map = {
        QCheckBox: 'chk',
        QComboBox: 'cbx',
        QDateTimeEdit: 'dte',
        QLabel: 'txt',
        QLineEdit: 'edt',
        QPushButton: 'btn',
        QSpinBox: 'spn',
        TableWithToolBar: 'tbl',
    }
    widget_type_str = widget_type_str_map[widget_type]
    return f'{widget_type_str}_{group_name}_{parameter_name}'
