"""GsshaComponentBase class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem, XmsEnvironment as XmEnv
from xms.components.bases.component_base import ComponentBase
from xms.core.filesystem import filesystem
from xms.guipy import file_io_util
from xms.guipy.resources import resources_util

# 4. Local modules
from xms.gssha.components import dmi_util


class GsshaComponentBase(ComponentBase):
    """GSSHA component base class."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        self._tree_commands = []

    def get_project_explorer_menus(self, main_file_list: list[str]) -> list[MenuItem]:
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list or not self._tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for item in self._tree_commands:
            if item:
                menu_list.append(self._get_menu_item(item[0], item[1], item[2], main_file=main_file_list[0][0]))
            else:
                menu_list.append(None)  # None is interpreted as a spacer menu item
        return menu_list

    def get_double_click_actions(self, lock_state: bool):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state: True if the component is locked for editing.
                Do not change the files if locked.

        Returns:
            (tuple): tuple containing:

                messages (list[tuple(str)]): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        if self._tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            actions.append(self._get_menu_action(self._tree_commands[0][1]))

        return messages, actions

    def _get_menu_action(self, command_method, id_files=None, selection=None, main_file=None) -> ActionRequest:
        """Get an ActionRequest for a modal menu item command.

        Args:
            command_method (str): Name of the method to call
            id_files (tuple(str)): Paths to the XMS id and component id files, if applicable
            selection (list[int]): Feature ids of the selected items
            main_file (str): Path to the component main file.

        Returns:
            The component menu item ActionRequest
        """
        main_file = main_file if main_file is not None else self.main_file
        action = ActionRequest(
            main_file=main_file,
            modality='MODAL',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name=command_method
        )
        parameters = {}
        if id_files is not None:
            parameters['id_files'] = id_files
        if selection is not None:
            parameters['selection'] = selection
        if parameters:
            action.action_parameters = parameters
        return action

    def _get_menu_item(
        self,
        command_text: str,
        command_method: str,
        icon_file: str | None = None,
        id_files: tuple[str] | None = None,
        selection: list[int] | None = None,
        main_file: str | None = None
    ) -> MenuItem:
        """Get a menu item for a modal command.

        Args:
            command_text: The menu text
            command_method: Name of the method to call
            icon_file: Name of icon file.
            id_files: Paths to the XMS id and component id files, if applicable
            selection: Feature ids of the selected items
            main_file: Path to the component main file.

        Returns:
            The component menu item
        """
        action = self._get_menu_action(command_method, id_files=id_files, selection=selection, main_file=main_file)
        icon = resources_util.get_resource_path(f':/resources/icons/{icon_file}') if icon_file else None
        return MenuItem(text=command_text, action=action, icon_path=icon)

    def save_to_location(self, new_path: str, save_type: str):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        new_main_file = Path(new_path) / Path(self.main_file).name
        if filesystem.paths_are_equal(new_main_file, self.main_file):  # We are already in the new location
            return self.main_file, messages, action_requests

        if save_type == 'SAVE':  # We already updated ourselves, so copy our mainfile from temp to the new location.
            filesystem.copyfile(self.main_file, new_main_file)
        elif save_type == 'SAVE_AS':
            # SAVE_AS: main_file will already be copied to the new location by now
            # Fix the paths and rewrite the file
            cards = file_io_util.read_json_file(new_main_file)
            if cards and 'SOLUTION_FILE_FULL' in cards and 'SOLUTION_FILE_RELATIVE' in cards:
                full, relative = cards['SOLUTION_FILE_FULL'], cards['SOLUTION_FILE_RELATIVE']
                sol_file_path = dmi_util.find_solution_file(full, relative)
                if sol_file_path is not None:
                    xms_project_file_path = Path(XmEnv.xms_environ_project_path())
                    rel_path = filesystem.compute_relative_path(xms_project_file_path.parent, sol_file_path)
                    cards['SOLUTION_FILE_FULL'] = str(sol_file_path)
                    cards['SOLUTION_FILE_RELATIVE'] = rel_path
                    file_io_util.write_json_file(cards, new_main_file)
        return str(new_main_file), messages, action_requests
