"""SimComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import QProcess
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query, XmsEnvironment
from xms.api.tree import tree_util, TreeNode
from xms.core.filesystem import filesystem
from xms.gmi.data.generic_model import Parameter
from xms.gmi.gui.dataset_callback import DatasetRequest
from xms.gmi.gui.group_set_dialog import GroupSetDialog
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.gssha.components import dmi_util
from xms.gssha.components.gssha_component_base import GsshaComponentBase
from xms.gssha.data import sim_generic_model
from xms.gssha.data.gssha_sim_data import GsshaSimData
from xms.gssha.gui.mapping_tables_dialog import MappingTablesDialog
from xms.gssha.mapping.link_number_preview_creator import create_link_number_preview
from xms.gssha.misc.type_aliases import ActionRv, Messages


class SimComponent(GsshaComponentBase):
    """A Dynamic Model Interface (DMI) component for a GSSHA simulation."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = GsshaSimData(self.main_file)
        self._tree_commands = [
            ('Model Control...', 'open_model_control', ''),
            ('Mapping Tables...', '_open_mapping_tables', ''),
            ('Open Containing Folder', '_open_containing_folder', ''),
            ('Create Link Number Preview', '_create_link_number_preview', ''),
            None,
        ]  # [(menu_text, menu_method, icon_file)...]
        self.project_tree: 'TreeNode | None' = None  # Not internal because accessed in mapping_tables_dialog
        self._ugrid_node: 'TreeNode | None' = None
        self._query: 'Query | None' = None
        self._win_cont: 'QWidget | None' = None

    def create_event(self, lock_state: bool) -> ActionRv:
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not lock_state:
            # Initialize the simulation to default values
            generic_model = sim_generic_model.create(default_values=True)
            self.data.global_values = generic_model.global_parameters.extract_values()
            self.data.model_values = generic_model.model_parameters.extract_values()
            self.data.commit()
        return [], []

    def save_to_location(self, new_path: str, save_type: str) -> tuple[str, Messages, list[ActionRequest]]:
        """Save component files to a new location.

        Args:
            new_path: Path to the new save location.
            save_type: One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (Messages): List of messages
                - action_requests (list[ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))
        same_file = filesystem.paths_are_equal(new_main_file, self.main_file)
        if same_file:  # We are already in the new location
            return self.main_file, messages, action_requests

        # SAVE_AS: main_file will already be copied to the new location by now
        if save_type == 'SAVE':  # We already updated ourselves, so copy our mainfile from temp to the new location.
            filesystem.copyfile(self.main_file, new_main_file)
        return new_main_file, messages, action_requests

    def get_double_click_actions(self, lock_state: bool) -> ActionRv:
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state: True if the component is locked for editing. Do not change the files if locked.

        Returns:
            ActionRv.
        """
        messages = []
        actions = []

        if self._tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            actions.append(self._get_menu_action(self._tree_commands[0][1]))

        return messages, actions

    def open_model_control(self, query: Query, params: list[dict], win_cont: QWidget) -> ActionRv:
        """
        Open the Model Control dialog and save component data state on OK.

        Uses a generic model and the GroupSetDialog from xmsgmi. Not internal ("_open_model_control") because it is
        called from MappingTablesDialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            ActionRv.
        """
        # win_cont=None below means we don't have to have a linked UGrid
        self._prepare_dataset_callback(query, win_cont=None, copy_tree=False)
        generic_model = sim_generic_model.create(default_values=False)
        generic_model.global_parameters.restore_values(self.data.global_values)
        message = (
            'This GSSHA interface is in "beta". Feedback is appreciated. Please report any bugs to'
            ' <a href=\'mailto:support@aquaveo.com\'>support@aquaveo.com</a>.'
        )
        style_sheet = 'QLabel { background-color : rgb(200, 200, 150); color : rgb(0, 0, 0); }'

        dlg = GroupSetDialog(
            parent=win_cont,
            section=generic_model.global_parameters,
            get_curve=self.data.get_curve,
            add_curve=self.data.add_curve,
            is_interior=False,
            dlg_name='xms.gssha.components.sim_component',
            window_title='GSSHA Model Control',
            banner=(message, style_sheet),
            enable_unchecked_groups=True,
            hide_checkboxes=True,
            dataset_callback=self._dataset_callback,
        )

        if dlg.exec():
            # Update the attribute datasets
            self.data.global_values = dlg.section.extract_values()
            self.data.commit()
        return [], []

    def _open_mapping_tables(self, query: Query, params: list[dict], win_cont: QWidget) -> ActionRv:
        """Open the Model Control dialog and save component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            ActionRv
        """
        self._prepare_dataset_callback(query, win_cont, copy_tree=True)
        if not self._ugrid_node:
            return [], []

        dlg = self._create_mapping_tables_dialog(query, win_cont)
        if not dlg:
            return [], []

        if dlg.exec():
            # Update the values
            values = dlg.section.extract_values()
            self.data.model_values = values
            self.data.commit()
        return [], []

    def _create_mapping_tables_dialog(self, query: Query, win_cont: QWidget) -> MappingTablesDialog | None:
        """Create the Mapping Tables dialog and save component data state on OK.

        Args:
            query: Object for communicating with XMS
            win_cont: The window container.

        Returns:
            The dialog.
        """
        # Get the values
        generic_model = sim_generic_model.create(default_values=False)
        values = self.data.model_values
        section = generic_model.model_parameters
        section.restore_values(values)

        dlg = MappingTablesDialog(
            parent=win_cont,
            section=section,
            query=query,
            sim_component=self,
            ugrid_node=self._ugrid_node,
            dataset_callback=self._dataset_callback,
        )
        return dlg

    def _prepare_dataset_callback(self, query: Query, win_cont: 'QWidget | None', copy_tree: bool) -> None:
        """Make necessary preparations for calling _dataset_callback().

        Args:
            query: Object for communicating with GMS
            win_cont: The window container. If not None and there's an error, a message box will be displayed.
             Otherwise, the error will be logged.
            copy_tree: If True, a copy of the project_tree is put in self.project_tree. Otherwise, it's the original.
        """
        if self._query:
            return

        self._query = query
        if copy_tree:
            self.project_tree = query.copy_project_tree()
        else:
            self.project_tree = query.project_tree
        sim_node = tree_util.find_tree_node_by_uuid(self.project_tree, query.parent_item_uuid())
        self._ugrid_node = dmi_util.get_ugrid_node_or_warn(self.project_tree, win_cont, sim_node=sim_node)

    def _keep_node_condition(self, node: TreeNode):
        """Method used by tree_util.filter_project_explorer to filter the tree.

        Args:
            node (TreeNode): A node.

        Returns:
            (bool): True if the node should be kept.
        """
        # Keep cell datasets and their ancestors
        if node.item_typename == 'TI_SCALAR_DSET' and node.data_location == 'CELL':
            return True
        else:
            for child in node.children:
                return self._keep_node_condition(child)

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> 'str | TreeNode':
        """Handle a request for information when picking a dataset.

        This should match `xms.gmi.gui.dataset_callback.DatasetCallback`.

        Notes:
            - `self._query` will be initialized and available before this method is called.
            - If `request` is `DatasetRequest.GetTree`, the returned node will be a clean copy of the entire project
              tree. Derived classes can safely mutate it for filtering purposes.

        Args:
            request: The requested operation.
            parameter: The parameter the request is for. If `request` is `DatasetRequest.GetLabel`, then
                `parameter.value` will be initialized to the UUID of the dataset to get the label for.

        Returns:
            - If `request` is `DatasetRequest.GetLabel`, returns a label to identify the dataset with to the user.
            - If `request` is `DatasetRequest.GetTree`, returns a tree for picking a new dataset for the parameter.
        """
        if request == DatasetRequest.GetLabel:
            return tree_util.build_tree_path(self.project_tree, parameter.value)
        elif request == DatasetRequest.GetTree:
            tree_copy = tree_util.copy_tree(self._ugrid_node)
            tree_util.filter_project_explorer(tree_copy, self._keep_node_condition)
            return tree_copy
        raise AssertionError('Unknown request.')  # pragma: nocover

    def _open_containing_folder(self, query: Query, params: list[dict], win_cont: QWidget) -> ActionRv:
        """Opens the File Explorer to the .grok file directory, if it exists.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont: The window container.

        Returns:
            ActionRv
        """
        messages: list = []
        actions: list = []
        grok_file_path = dmi_util.get_gssha_file_path(query)
        if not XmsEnvironment.xms_environ_project_path():
            message = 'The project and simulation must first be saved.'
            message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        elif not grok_file_path.parent.is_dir():
            message = f'No simulation found at "{str(grok_file_path.parent)}". You must first save the simulation.'
            message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        else:
            # os.startfile(grok_file_path.parent) Causes Windows Explorer to appear behind GMS
            QProcess.startDetached('explorer', ['/select', ',', str(grok_file_path)])
        return messages, actions

    def _create_link_number_preview(self, query: Query, params: list[dict], win_cont: QWidget) -> ActionRv:
        """Creates a snapped bc component.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            ActionRv
        """
        create_link_number_preview(query=query, win_cont=win_cont)
        return [], []
