"""SimGenericModelCreator class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from datetime import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.gssha.data import mapping_tables
from xms.gssha.data.generic_model_base import GenericModelCreatorBase


class Overtype:
    """Overland flow options (OVERTYPE parameter)."""
    ADE = 'ADE'
    ADE_PC = 'ADE-PC'


class InfilType:
    """Infiltration options (infil_type parameter)."""
    NO_INFILTRATION = 'No infiltration'
    INF_REDIST = 'Green & Ampt w/ soil moisture redistribution (INF_REDIST)'
    INF_LAYERED_SOIL = 'Green & Ampt multi layer (INF_LAYERED_SOIL)'
    INF_RICHARDS = "Richard's infiltration (INF_RICHARDS)"


class RichardsCOption:
    """RICHARDS_C_OPTION parameter."""
    BROOKS = 'brooks'
    HAVERCAMP = 'havercamp'


class Routing:
    """Routing options (routing parameter)."""
    NO_ROUTING = 'No routing'
    DIFFUSIVE_WAVE = 'Diffusive wave (DIFFUSIVE_WAVE)'


class OutputUnits:
    """Output units options (output_units parameter)."""
    METRIC = 'Metric (cms) (METRIC)'
    QOUT_CFS = 'English (cfs) (QOUT_CFS)'


class PrecipitationType:
    """Precipitation options (precipitation_type)."""
    PRECIP_UNIF = 'Uniform (PRECIP_UNIF)'
    HYETOGRAPH = 'Hyetograph'


def create(default_values: bool = False) -> GenericModel:
    """Creates and returns the generic model.

    default_values should be False when importing, True when creating a new simulation.

    Args:
        default_values: If True, parameters with specific (non-zero etc.) default values will be set.

    Returns:
        (GenericModel): The generic model.
    """
    creator = SimGenericModelCreator()
    generic_model = creator.create()
    if default_values:
        _set_default_values(generic_model)
    return generic_model


class SimGenericModelCreator(GenericModelCreatorBase):
    """Creates the GenericModel for the simulation.

    By convention, parameter names are the same as the GSSHA card and are upper case. If there is no matching GSSHA
    card, the parameter names are lower case.
    """
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._gm = GenericModel()
        self._descriptions: dict[str, str] = _descriptions()

    def create(self) -> GenericModel:
        """Creates and returns the generic model.

        Returns:
            (GenericModel): The generic model.
        """
        self._add_global_parameters()
        self._add_model_parameters()
        return self._gm

    def _add_global_parameters(self) -> None:
        """Adds the global parameters, which are used for the Model Control dialog."""
        self._section = self._gm.global_parameters
        self._add_general()
        self._add_overland_flow()
        self._add_infiltration()
        self._add_channel_routing()
        self._add_output()
        self._add_precipitation()

    def _add_model_parameters(self) -> None:
        """Adds the model parameters, which are used for the Mapping Tables dialog."""
        self._section = self._gm.model_parameters
        self._add_group('mapping_tables', 'Mapping Tables')
        tables = mapping_tables.names()
        for name in tables:
            self._dataset(mapping_tables.dataset_param(name), '')  # 'Index map' datasets
            self._mapping_table(name)  # Mapping table

    def _add_general(self) -> None:
        """Adds the general group."""
        self._add_group('general', 'General')
        self._int('TOT_TIME', 'Total time (min) (TOT_TIME)', 0, low=1)
        self._int('TIMESTEP', 'Time step (sec) (TIMESTEP)', 0, low=1)
        self._float('OUTSLOPE', 'Outlet slope (OUTSLOPE)', 0.0)

    def _add_overland_flow(self) -> None:
        """Adds the overland flow group."""
        self._add_group('overland_flow', 'Overland Flow')
        self._dataset('ELEVATION', 'ELEVATION dataset')
        solver_opts = [Overtype.ADE, Overtype.ADE_PC]
        self._option('OVERTYPE', 'Solver (OVERTYPE)', Overtype.ADE, solver_opts)

        # The following are not standard GSSHA cards, but they allow us to serialize/restore these settings
        self._bool('ROUGH_EXP', 'ROUGH_EXP', False)  # For ROUGH_EXP mapping table. "Depth varying roughness" in WMS.
        self._bool('INTERCEPTION', 'INTERCEPTION', False)  # For INTERCEPTION mapping table
        # self._bool('INITIAL_DEPTH', 'INITIAL_DEPTH', False)  # continuous map - unsupported (even in WMS)
        self._bool('RETENTION', 'RETENTION', False)  # For RETENTION mapping table. WMS 'Retention depth' RETEN_DEPTH
        self._bool('AREA_REDUCTION', 'AREA_REDUCTION', False)  # For AREA_REDUCTION mapping table

    def _add_infiltration(self) -> None:
        """Adds the infiltration group."""
        self._add_group('infiltration', 'Infiltration')
        infil_opts = [
            InfilType.NO_INFILTRATION,
            InfilType.INF_REDIST,
            InfilType.INF_LAYERED_SOIL,
            InfilType.INF_RICHARDS,
        ]

        t = self._option('infil_type', 'Infiltration type', InfilType.NO_INFILTRATION, infil_opts)
        b = self._float('RICHARDS_WEIGHT', "Weight (RICHARDS_WEIGHT)", 0.0)
        richards_flags = {
            InfilType.NO_INFILTRATION: False,
            InfilType.INF_REDIST: False,
            InfilType.INF_LAYERED_SOIL: False,
            InfilType.INF_RICHARDS: True
        }
        b.add_dependency(t, richards_flags)

        b = self._float('RICHARDS_DTHETA_MAX', "DTHETA Max (RICHARDS_DTHETA_MAX)", 0.0)
        b.add_dependency(t, richards_flags)

        c_opts = [RichardsCOption.BROOKS, RichardsCOption.HAVERCAMP]
        b = self._option('RICHARDS_C_OPTION', "C option (RICHARDS_C_OPTION)", c_opts[0], c_opts)
        b.add_dependency(t, richards_flags)

        k_opts = ['geometric', 'arithmetic']
        b = self._option('RICHARDS_K_OPTION', "K option (RICHARDS_K_OPTION)", k_opts[0], k_opts)
        b.add_dependency(t, richards_flags)

        upper_opts = ['green_ampt', 'average']
        b = self._option('RICHARDS_UPPER_OPTION', "Upper option (RICHARDS_UPPER_OPTION)", upper_opts[0], upper_opts)
        b.add_dependency(t, richards_flags)

        b = self._int('RICHARDS_ITER_MAX', 'Max iteration (RICHARDS_ITER_MAX)', 0)
        b.add_dependency(t, richards_flags)

        b = self._int('MAX_NUMBER_CELLS', 'Max num. cells (MAX_NUMBER_CELLS)', 0)
        b.add_dependency(t, richards_flags)

        a = self._bool('SOIL_MOIST_DEPTH_CHK', 'Soil moisture depth (m) (SOIL_MOIST_DEPTH)', False)
        a.add_dependency(
            t, {
                InfilType.NO_INFILTRATION: False,
                InfilType.INF_REDIST: True,
                InfilType.INF_LAYERED_SOIL: True,
                InfilType.INF_RICHARDS: True
            }
        )
        b = self._float('SOIL_MOIST_DEPTH', '', 0.0)
        b.add_dependency(a, {True: True, False: False})

        a = self._bool('TOP_LAYER_DEPTH_CHK', 'Top layer depth (m) (TOP_LAYER_DEPTH)', False)
        a.add_dependency(
            t, {
                InfilType.NO_INFILTRATION: False,
                InfilType.INF_REDIST: True,
                InfilType.INF_LAYERED_SOIL: False,
                InfilType.INF_RICHARDS: False
            }
        )
        b = self._float('TOP_LAYER_DEPTH', '', 0.0)
        b.add_dependency(a, {True: True, False: False})

    def _add_channel_routing(self) -> None:
        """Adds the channel routing group."""
        self._add_group('channel_routing', 'Channel routing')
        routing_opts = [Routing.NO_ROUTING, Routing.DIFFUSIVE_WAVE]
        r = self._option('routing', 'Channel routing computation scheme', Routing.NO_ROUTING, routing_opts)

        routing_flags = {Routing.NO_ROUTING: False, Routing.DIFFUSIVE_WAVE: True}
        a = self._bool('OVERBANK_FLOW', 'Allow overbank flow (OVERBANK_FLOW)', False)
        a.add_dependency(r, routing_flags)
        a = self._bool('OVERLAND_BACKWATER', 'Allow overland backwater (OVERLAND_BACKWATER)', False)
        a.add_dependency(r, routing_flags)

        a = self._bool('HEAD_BOUND', 'Outlet head BC (m) (HEAD_BOUND / BOUND_DEPTH)', False)
        a.add_dependency(r, routing_flags)
        b = self._float('BOUND_DEPTH', '', 0.0)
        b.add_dependency(a, {True: True, False: False})

        a = self._bool('write_chan_hotstart_chk', 'Create hot start file', False)
        a.add_dependency(r, routing_flags)
        b = self._text('WRITE_CHAN_HOTSTART', 'Hot start filename (WRITE_CHAN_HOTSTART)', 'untitled.sta')
        b.add_dependency(a, {True: True, False: False})

        a = self._bool('read_chan_hotstart_chk', 'Start from hot start file', False)
        a.add_dependency(r, routing_flags)
        b = self._text('READ_CHAN_HOTSTART', 'Hot start filename (READ_CHAN_HOTSTART)', 'untitled.sta')
        b.add_dependency(a, {True: True, False: False})

        # erosion
        a = self._bool('SOIL_EROSION', 'Soil erosion (SOIL_EROSION)', False)
        a.add_dependency(r, routing_flags)
        b = self._float('SED_POROSITY', 'Sediment porosity (SED_POROSITY)', 0.0)
        b.add_dependency(a, {True: True, False: False})
        b = self._float('WATER_TEMP', 'Water temperature (C) (WATER_TEMP)', 0.0)
        b.add_dependency(a, {True: True, False: False})
        b = self._float('SAND_SIZE', 'Sand size (mm) (SAND_SIZE)', 0.0)
        b.add_dependency(a, {True: True, False: False})

        # contaminant transport
        a = self._bool('COMPUTE_CONTAMINANT_TRANSPORT', 'Compute contaminant transport', False)
        a.add_dependency(r, routing_flags)
        b = self._float('CHAN_DECAY_COEF', 'Decay coefficient (per day) (CHAN_DECAY_COEF)', 0.0)
        b.add_dependency(a, {True: True, False: False})
        b = self._float('CHAN_DISP_COEF', 'Dispersion coefficient (m^2/s) (CHAN_DISP_COEF)', 0.0)
        b.add_dependency(a, {True: True, False: False})
        b = self._float('INIT_CHAN_CONC', 'Initial concentration (mg/l) (INIT_CHAN_CONC)', 0.0)
        b.add_dependency(a, {True: True, False: False})

    def _add_output(self) -> None:
        """Adds the output group(s)."""
        self._add_output_general()
        self._add_output_gridded()
        self._add_output_link_node()

    def _add_output_general(self) -> None:
        """Adds the general output group."""
        self._add_group('output_general', 'Output - General')

        # Write frequency
        self._int('MAP_FREQ', 'Dataset write frequency (min) (MAP_FREQ)', 1, low=1)

        # Hydrograph
        self._int('HYD_FREQ', 'Hydrograph write frequency (min) (HYD_FREQ)', 1, low=1)
        output_units_opts = [OutputUnits.METRIC, OutputUnits.QOUT_CFS]
        self._option('output_units', 'Output units', OutputUnits.METRIC, output_units_opts)

        # Other
        self._bool('QUIET', 'Suppress screen printing (QUIET)', False)
        self._bool('SUPER_QUIET', 'Suppress long term simulation printing (SUPER_QUIET)', False)
        self._bool('STRICT_JULIAN_DATE', 'Strict Julian dates (STRICT_JULIAN_DATE)', False)
        self._bool('ALL_CONTAM_OUTPUT', 'Output contaminants (ALL_CONTAM_OUTPUT)', False)

    def _add_output_gridded(self) -> None:
        """Adds the gridded datasets output group."""
        self._add_group('output_gridded', 'Output - Gridded Datasets')
        # Gridded datasets
        self._bool('DIS_RAIN', 'Distributed rainfall intensity (DIS_RAIN)', False)
        self._bool('DEPTH', 'Surface depth (DEPTH)', False)
        self._bool('INF_DEPTH', 'Cumulative infiltration depth (INF_DEPTH)', False)
        self._bool('RATE_OF_INFIL', 'Infiltration rate (RATE_OF_INFIL)', False)
        self._bool('SURF_MOIST', 'Surface soil moisture (SURF_MOIST)', False)
        self._bool('FLOOD_GRID', 'Flood (max) depth (FLOOD_GRID)', False)

    def _add_output_link_node(self) -> None:
        """Adds the link/node datasets output group."""
        self._add_group('output_link_node', 'Output - Link / Node Datasets')

        # Link / Node data sets
        self._bool('CHAN_DEPTH', 'Channel depth (CHAN_DEPTH)', False)
        self._bool('CHAN_DISCHARGE', 'Channel flow (CHAN_DISCHARGE)', False)
        self._bool('CHAN_VELOCITY', 'Channel velocity (avg) (CHAN_VELOCITY)', False)
        self._bool('CHAN_SED_FLUX', 'Channel sediment flux (CHAN_SED_FLUX)', False)
        self._bool('FLOOD_STREAM', 'Flood (max) depth (FLOOD_STREAM)', False)
        self._bool('CHAN_STAGE', 'Water surface elev (CHAN_STAGE)', False)
        self._bool('PIPE_FLOW', 'Pipe flow (PIPE_FLOW)', False)
        self._bool('PIPE_HEAD', 'Pipe node depths (PIPE_HEAD)', False)
        self._bool('PIPE_TILE', 'Pipe tile in/out flow (PIPE_TILE)', False)
        self._bool('SUPERLINK_JUNC_FLOW', 'Superlink junction in/out flow (SUPERLINK_JUNC_FLOW)', False)
        self._bool('SUPERLINK_NODE_FLOW', 'Superlink node in/out flow (SUPERLINK_NODE_FLOW)', False)

    def _add_precipitation(self):
        """Adds precipitation."""
        self._add_group('precipitation', 'Precipitation')
        precip_opts = [PrecipitationType.PRECIP_UNIF, PrecipitationType.HYETOGRAPH]
        o = self._option('precipitation_type', 'Precipitation type', PrecipitationType.PRECIP_UNIF, precip_opts)
        start_date_time = datetime(year=2000, month=1, day=1)
        self._date_time('start_date_time', 'Start date/time (START_DATE, START_TIME)', start_date_time)

        # Uniform
        a = self._float('RAIN_INTENSITY', 'Intensity (mm/hr) (RAIN_INTENSITY)', 0.0)
        a.add_dependency(o, {PrecipitationType.PRECIP_UNIF: True, PrecipitationType.HYETOGRAPH: False})
        a = self._int('RAIN_DURATION', 'Duration (min) (RAIN_DURATION)', 0)
        a.add_dependency(o, {PrecipitationType.PRECIP_UNIF: True, PrecipitationType.HYETOGRAPH: False})

        # Hyetograph
        a = self._xy_series('hyetograph_xy', 'Hyetograph', ['Unit Time (min)', 'Distribution'])
        a.add_dependency(o, {PrecipitationType.PRECIP_UNIF: False, PrecipitationType.HYETOGRAPH: True})
        a = self._float('hyetograph_avg_depth', 'Average depth (mm)', 0.0)
        a.add_dependency(o, {PrecipitationType.PRECIP_UNIF: False, PrecipitationType.HYETOGRAPH: True})


def _descriptions() -> dict[str, str]:
    """Returns a dict of descriptions."""
    # yapf: disable
    return {
        # global_parameters section

        # - general group
        'general': 'provides the framework for how the simulation is to be set up and run',
        'TOT_TIME': 'simulation length (non-long-term) in minutes',
        'TIMESTEP': 'overland and stream time step in seconds',
        'OUTSLOPE': 'outlet slope, for overland only',

        # - overland_flow group
        'overland_flow': 'overland flow options',
        'ELEVATION': 'surface elevation grid file',
        'OVERTYPE': 'overland flow solver',
        'ROUGH_EXP': 'ROUGH_EXP mapping table (depth varying roughness exponent)',
        'INTERCEPTION': 'INTERCEPTION mapping table (water abstracted from rainfall by vegetation)',
        'RETENTION': 'RETENTION mapping table (retention depth used in the overland flow model)',
        'AREA_REDUCTION': 'AREA_REDUCTION mapping table (impervious area fraction)',

        # - channel_routing group
        'channel_routing': 'channel routing options',
        'routing': '',
        'MIN_FLOW': '',
        'OVERBANK_FLOW': 'allow streams to flow back to overland',
        'OVERLAND_BACKWATER': '',
        'HEAD_BOUND': 'Use a head boundary at the channel outlet',
        'BOUND_DEPTH': '(BOUND_DEPTH) Use a constant value (m) for the outlet channel depth (not stage.)',
        'write_chan_hotstart_chk': 'Create a hot start file',
        'WRITE_CHAN_HOTSTART': 'Create two channel hot-start files at the end of the simulation. Depth file has'
                               ' extension .dht, flow has extention .qht.',
        'read_chan_hotstart_chk': 'Read in a hot-start file, created by WRITE_CHAN_HOTSTART.',
        'READ_CHAN_HOTSTART': 'Read in a hot-start file, created by WRITE_CHAN_HOTSTART.',
        # -- erosion
        'SOIL_EROSION': '',
        'SED_POROSITY': '',
        'WATER_TEMP': 'The temperature of the water (constant)',
        'SAND_SIZE': 'Representative grain-size for sand-size particles.',
        # -- contaminant transport
        'COMPUTE_CONTAMINANT_TRANSPORT': '',
        'CHAN_DECAY_COEF': 'First order decay coefficient of constituents in channel',
        'CHAN_DISP_COEF': 'Channel dispersion coefficient',
        'INIT_CHAN_CONC': '',

        # - infiltration group
        'infiltration': 'infiltration options',
        'infil_type': 'infiltration type',
        'RICHARDS_WEIGHT': '',
        'RICHARDS_DTHETA_MAX': '',
        'RICHARDS_C_OPTION': '',
        'RICHARDS_K_OPTION': '',
        'RICHARDS_UPPER_OPTION': '',
        'RICHARDS_ITER_MAX': '',
        'MAX_NUMBER_CELLS': '',
        'SOIL_MOIST_DEPTH_CHK': '',
        'SOIL_MOIST_DEPTH': 'Full depth for soil moisture computations',
        'TOP_LAYER_DEPTH_CHK': '',
        'TOP_LAYER_DEPTH': 'Upper layer depth (included in soil_moist_depth) for just an upper layer soil moisture'
                           ' layer.',

        # - output_gridded group
        'output_gridded': 'gridded dataset output options.',

        # -- Gridded datasets
        'DIS_RAIN': 'rainfall rate as it is being applied',
        'DEPTH': 'overland flow depth maps, if map_type==0 then put a path else put a filename',
        'INF_DEPTH': 'infiltration depth',
        'RATE_OF_INFIL': '',
        'SURF_MOIST': 'soil moisture output',
        'FLOOD_GRID': 'outputs the maximum water depth on the overland grid',

        # - output_link_node group
        'output_link_node': 'link / node dataset output options.',

        # -- Link / Node data sets
        'CHAN_DEPTH': 'channel flow depth, link/node format',
        'CHAN_DISCHARGE': 'channel discharge, link/node format',
        'CHAN_VELOCITY': 'channel velocity, link/node format',
        'CHAN_SED_FLUX': '',
        'FLOOD_STREAM': 'outputs the maximum water depth on the stream network',
        'CHAN_STAGE': 'channel stage, link/node format',
        'PIPE_FLOW': '',
        'PIPE_HEAD': '',
        'PIPE_TILE': '',
        'SUPERLINK_JUNC_FLOW': 'File that contains the time series output data for the junctions listed in'
                               ' SUPERLINK_JUNC_LOCATION.',
        'SUPERLINK_NODE_FLOW': 'File that contains the time series output data for the nodes listed in'
                               ' SUPERLINK_NODE_LOCATION.',

        # - output_general group
        'output_general': 'general output options.',

        # -- Write frequency
        'MAP_FREQ': 'Gridded output frequency, in minutes',

        # -- Hydrograph
        'HYD_FREQ': 'hydrograph, link/node dataset output frequency, in minutes',
        'output_units': 'output units',

        # -- Other
        'QUIET': 'do not output to screen as model is running',
        'SUPER_QUIET': 'do not output to screen',
        'STRICT_JULIAN_DATE': 'force hydrograph output to be in strict julian date',
        'ALL_CONTAM_OUTPUT': 'Specifies that all contaminant information is output for every overland cell, every'
                             ' channel node, and every soil cell/layer every MAP_FREQ and HYD_FREQ, respectively.'
                             ' Maps of overland concentration and mass are written for every constituent using the'
                             ' constituent name in the mapping table with the extensions "conc" and "mass",'
                             ' respectively. Tables of concentration and mass for every channel node are written for'
                             ' every constituent using the constituent name in the mapping table with the extensions'
                             ' "chan.conc" and "chan.mass", respectively. Use of this option can result in an enormous'
                             ' amount of data being written.',

        # - precipitation group
        'precipitation': 'precipitation options',
        'precipitation_type': 'Type of precipitation',
        'start_date_time': 'Tab key advances to the next number. Up and down arrow keys increment and'
                           ' decrement.',
        # -- Uniform
        'RAIN_INTENSITY': 'Set the precipitation intensity for uniform precip. mm/hr.',
        'RAIN_DURATION': 'Set the precipitation duration for uniform precip. Minutes.',
        # -- Hyetograph
        'hyetograph_xy': 'Use dates/times with hyetograph distribution',
        'hyetograph_avg_depth': '',

        # model_parameters section

        # - mapping_tables group
        'mapping_tables': 'The Mapping Tables group',

        # -- 'Index map' datasets
        'roughness_index_map_dataset': 'Index map dataset to use',
        'rough_exp_index_map_dataset': 'Index map dataset to use',
        'interception_index_map_dataset': 'Index map dataset to use',
        'retention_index_map_dataset': 'Index map dataset to use',
        'green_ampt_infiltration_index_map_dataset': 'Index map dataset to use',
        'richards_eqn_infiltration_brooks_index_map_dataset': 'Index map dataset to use',
        'richards_eqn_infiltration_havercamp_index_map_dataset': 'Index map dataset to use',
        'green_ampt_initial_soil_moisture_index_map_dataset': 'Index map dataset to use',
        'area_reduction_index_map_dataset': 'Index map dataset to use',

        # -- Tables
        'roughness_table': '',
        'rough_exp_table': '',
        'interception_table': '',
        'retention_table': '',
        'green_ampt_infiltration_table': '',
        'richards_eqn_infiltration_brooks_table': '',
        'richards_eqn_infiltration_havercamp_table': '',
        'green_ampt_initial_soil_moisture_table': '',
        'area_reduction_table': '',
    }
    # yapf: enable


def _set_default_values(generic_model: GenericModel) -> None:
    """Sets certain parameters to their specific default values.

    Some parameters have sensible default values, but these should only be set when creating a new simulation. When
    importing, defaults should be 0.0 or False to help indicate that no value was read.

    Args:
        generic_model: The generic model.
    """
    # Global parameters
    section = generic_model.global_parameters

    group = generic_model.global_parameters.group('general')
    group.parameter('TOT_TIME').value = 1500
    group.parameter('TIMESTEP').value = 10
    group.parameter('OUTSLOPE').value = .001

    group = section.group('infiltration')
    group.parameter('RICHARDS_WEIGHT').value = 0.8
    group.parameter('RICHARDS_DTHETA_MAX').value = 0.025
    group.parameter('RICHARDS_ITER_MAX').value = 1

    group = section.group('output_general')
    group.parameter('MAP_FREQ').value = 15
    group.parameter('HYD_FREQ').value = 5
    group.parameter('output_units').value = OutputUnits.QOUT_CFS

    group = section.group('output_gridded')
    group.parameter('DEPTH').value = True

    group = section.group('precipitation')
    group.parameter('RAIN_INTENSITY').value = 40.0
    group.parameter('RAIN_DURATION').value = 120
