"""Classes for defining line display options in XMS."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from enum import IntEnum

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.point_symbol import PointOptions


class LineStyle(IntEnum):
    """An enumeration matching the defines in XMS.

    Values not supported in XMS are commented out for the time being.
    """
    SOLIDLINE = 0
    DASHEDLINE = 1
    DOTTEDLINE = 2  # only used by plots
    DASH_DOT_LINE = 3  # only used by plots
    DASH_DOT_DOT_LINE = 4  # only used by plots
    # DOT_DASH_DOT_LONGDASH = 20


class LinePointStyle(IntEnum):
    """Line style enum."""
    no_points = 0
    end_points = 1
    vertices = 2
    all_points = 3


@dataclass
class LineOptions:
    """A class to hold the display options."""
    color: QColor = field(default_factory=lambda: QColor('black'))  # To instantiate distinct black QColor objects
    width: int = 1
    style: LineStyle = LineStyle.SOLIDLINE
    point_style: LinePointStyle = LinePointStyle.no_points
    end_point: PointOptions = field(default_factory=PointOptions)
    vertex: PointOptions = field(default_factory=PointOptions)

    def from_dict(self, opt_dict):
        """Populate from a dict.

        Args:
            opt_dict (dict): dict containing the line option attributes
        """
        red = 0
        green = 0
        blue = 0
        alpha = 0
        if 'red' in opt_dict:
            red = opt_dict['red']
        if 'green' in opt_dict:
            green = opt_dict['green']
        if 'blue' in opt_dict:
            blue = opt_dict['blue']
        if 'alpha' in opt_dict:
            alpha = opt_dict['alpha']
        self.color = QColor(red, green, blue, alpha)
        if 'width' in opt_dict:
            self.width = opt_dict['width']
        if 'style' in opt_dict:
            self.style = LineStyle(int(opt_dict['style']))
        if 'point_style' in opt_dict:
            self.point_style = LinePointStyle(int(opt_dict['point_style']))
        if 'end_point' in opt_dict:
            self.end_point.from_dict(opt_dict['end_point'])
        if 'vertex' in opt_dict:
            self.vertex.from_dict(opt_dict['vertex'])

    def __iter__(self):
        """Used for conversion to dict."""
        yield 'red', self.color.red()  # Decompose QColor to R,G,B,A components
        yield 'green', self.color.green()
        yield 'blue', self.color.blue()
        yield 'alpha', self.color.alpha()
        yield 'width', self.width
        yield 'style', int(self.style)  # Cast enum to int for sending to C++
        yield 'point_style', int(self.point_style)  # Cast enum to int for sending to C++
        yield 'end_point', dict(self.end_point)
        yield 'vertex', dict(self.vertex)
