"""Polygon texture display options."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from enum import IntEnum

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules

# 4. Local modules


# in order XMS is expecting
class PolygonTexture(IntEnum):
    """An enumeration matching the stipples in XMS."""
    solid_pattern = 0
    diagonal_cross_pattern = 1
    cross_pattern = 2
    diagonal_right_pattern = 3
    diagonal_left_pattern = 4
    vertical_pattern = 5
    igneous2_pattern = 6
    conglomerate_pattern = 7
    shale_pattern = 8
    limestone_pattern = 9
    granite_pattern = 10
    quartzite_pattern = 11
    sandstone_pattern = 12
    tuff_pattern = 13
    polymorphic_pattern = 14
    sand_shale_pattern = 15
    dolomite_pattern = 16
    lava_pattern = 17
    igneous_pattern = 18
    clay_pattern = 19
    clay_invert_pattern = 20
    stripes_pattern = 21
    rays_pattern = 22
    null_pattern = 23


@dataclass
class PolygonOptions:
    """A class to hold the display options."""
    color: QColor = field(default_factory=lambda: QColor('black'))  # To instantiate distinct black QColor objects
    texture: PolygonTexture = PolygonTexture.diagonal_cross_pattern

    def from_dict(self, opt_dict):
        """Populate options from a dict.

        Args:
            opt_dict (dict): dict of the polygon display option attributes
        """
        red = 0
        green = 0
        blue = 0
        alpha = 0
        if 'red' in opt_dict:
            red = opt_dict['red']
        if 'green' in opt_dict:
            green = opt_dict['green']
        if 'blue' in opt_dict:
            blue = opt_dict['blue']
        if 'alpha' in opt_dict:
            alpha = opt_dict['alpha']
        self.color = QColor(red, green, blue, alpha)
        if 'texture' in opt_dict:
            self.texture = PolygonTexture(int(opt_dict['texture']))

    def __iter__(self):
        """Used for conversion to dict."""
        yield 'red', self.color.red()  # Decompose QColor to R,G,B,A components
        yield 'green', self.color.green()
        yield 'blue', self.color.blue()
        yield 'alpha', self.color.alpha()
        yield 'texture', int(self.texture)  # Cast enum to int for sending to C++

    def __hash__(self):
        """Hash the object.

        Returns:
             The hash value of the object.
        """
        return hash((self.color.red(), self.color.green(), self.color.blue(), self.color.alpha(), self.texture))
