"""Qt delegate for a dataset tree item selector button."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.guipy.dialogs import treeitem_selector_datasets
from xms.guipy.dialogs.dataset_selector import DatasetSelector


class DatasetButtonDelegate(QStyledItemDelegate):
    """Qt delegate for a dataset tree item selector button."""
    def __init__(
        self,
        pe_tree,
        filter_method,
        icon_method,
        parent=None,
        query=None,
        allow_all_time_steps=False,
        allow_change_selection=True,
        show_full_dataset_path=False
    ):
        """Initializes the class.

        Args:
            pe_tree (TreeNode): The project explorer tree.
            parent (QObject): The parent object.
            filter_method: Method for DatasetSelector.select_dataset to fitler datasets shown.
            icon_method: Method for TreeItemSelectorDlg to call to get the geom icon.
            query (xms.api.dmi.Query): Object for communicating with GMS. If not None, the
                dialog that appears will allow for selecting time steps (query is needed for that).
            allow_all_time_steps (bool): If true (and query is not None), an 'All time steps'
                toggle is displayed.
            allow_change_selection (bool): Whether to allow changing the selected dataset. If this is True,
                only the time step can be selected for the selected dataset.
            show_full_dataset_path (bool): If this argument is set to "True", the entire path to the dataset
                is shown on the dataset button instead of only the dataset name.
        """
        super().__init__(parent)
        self.parent_dlg = parent
        self.uuid_to_tree_name = {}
        self.pe_tree = pe_tree
        self.dialog_title = ''
        self.filter_method = filter_method
        self.icon_method = icon_method
        self.query = query
        self.allow_all_time_steps = allow_all_time_steps
        self.allow_change_selection = allow_change_selection
        self.show_full_dataset_path = show_full_dataset_path

        self._fill_uuid_to_tree_name(self.pe_tree)

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            dset_data = index.data()
            dset_uuid, _ = treeitem_selector_datasets.uuid_and_time_step_index_from_string(dset_data)
            if dset_uuid and dset_uuid in self.uuid_to_tree_name:
                ds_name = ''
                if self.show_full_dataset_path:
                    ds_name = tree_util.build_tree_path(self.pe_tree, dset_uuid)
                    ds_split = ds_name.split("/", 3)
                    if len(ds_split) > 3:
                        ds_name = ds_split[3]
                else:
                    ds_name = self.uuid_to_tree_name[dset_uuid]
                btn_text = ds_name
                pos = dset_data.rfind(':')  # Look for time step string text.
                if pos > 0:
                    btn_text += dset_data[pos:]
                btn.setText(btn_text)
            else:
                btn.setText('(none selected)')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the XY series editor button is clicked.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                dset_uuid = index.data()
                # See if there's a time step index at the end of the uuid (e.g. 'uuid_str:ts_idx')
                dset_uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(dset_uuid)

                # dataset selector dialog
                dlg_title = self.dialog_title if self.dialog_title else 'Select Dataset'
                temp_dict = {'temp': dset_uuid}
                if DatasetSelector.select_dataset(
                    self.parent_dlg,
                    None,
                    dlg_title,
                    self.pe_tree,
                    self.filter_method,
                    temp_dict,
                    'temp',
                    self.icon_method,
                    query=self.query,
                    selected_time_step=ts_idx,
                    allow_all_time_steps=self.allow_all_time_steps,
                    allow_change_selection=self.allow_change_selection
                ):
                    model.setData(index, temp_dict['temp'])  # Only update the model if user clicked OK
                return True
        return super().editorEvent(event, model, option, index)

    def _fill_uuid_to_tree_name(self, node):
        """Fills in a dictionary of UUIDs to names in the project explorer tree.

        Args:
            node (TreeNode): The current node in the project explorer tree.
        """
        if not node:
            return
        if node.uuid:
            self.uuid_to_tree_name[node.uuid] = node.name
        for child in node.children:
            self._fill_uuid_to_tree_name(child)
