"""Dialog for viewing and editing point display options in XMS."""

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QSize, Qt, Slot
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.dialogs.point_display_options_ui import Ui_PointDisplayOptionsDialog

# 4. Local modules


class PointDisplayOptionsDialog(QDialog):
    """A dialog to let the user set point display options."""
    def __init__(self, options, parent=None, change_color=True):
        """Initializes the dialog.

        Args:
            options (PointOptions): The color, size, and symbol of the point to draw.
            parent (Something derived from QWidget): The parent window.
            change_color (bool): If True, the user can change the point color
        """
        super().__init__(parent)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"

        self.ui = Ui_PointDisplayOptionsDialog()
        self.ui.setupUi(self)
        self.ui.symbols.set_symbol(options.symbol)
        self.ui.symbols.draw_all_symbols()
        if change_color:
            self.ui.color_btn.set_color(options.color)
            self.ui.color_btn.color_changed.connect(self.update_preview)
        else:  # Hide color selector widgeta if disabled.
            self.ui.color_btn.hide()
            self.ui.lab_line_color.hide()
        self.ui.spin_size.setValue(options.size)
        self.ui.spin_size.valueChanged.connect(self.update_preview)
        self.ui.symbols.symbol_changed.connect(self.update_preview)
        self.setMinimumSize(self.size())
        self.update_preview()

    def get_options(self):
        """Returns the current display options.

        Returns:
            options (PointOptions): The current display options.
        """
        options = PointOptions()
        options.color = self.get_color()
        options.size = self.get_size()
        options.symbol = self.get_symbol()
        return options

    def get_color(self):
        """Returns the current color.

        Returns:
            color (QColor): The current color.
        """
        return self.ui.color_btn.color

    def get_size(self):
        """Returns the current size.

        Returns:
            size (int): The current size.
        """
        return self.ui.spin_size.value()

    def get_symbol(self):
        """Returns the current symbol.

        Returns:
            symbol (PointSymbol): The current symbol.
        """
        return self.ui.symbols.get_symbol()

    @Slot()
    def update_preview(self):
        """Called when the color, size, or symbol changes."""
        btn_size = self.ui.btn_preview.size()
        icon = self.ui.symbols.get_current_icon(self.get_color(), self.get_size(), btn_size.width())
        icon_size = QSize(self.get_size(), self.get_size())
        self.ui.btn_preview.setIconSize(icon_size)
        self.ui.btn_preview.setIcon(icon)

    @Slot()
    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    @Slot()
    def accept(self):
        """Called when the OK button is clicked."""
        QDialog.accept(self)

    @Slot()
    def reject(self):
        """Called when the Cancel button is clicked."""
        QDialog.reject(self)
