"""Module for the `QxDoubleValidator` class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules
from PySide2.QtCore import QObject
from PySide2.QtGui import QDoubleValidator, QValidator  # noqa: AQU103  We need this to implement its replacement.

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103  We're implementing the replacement.
from xms.guipy.validators.qx_locale import QxLocale


class QxDoubleValidator(QDoubleValidator):
    """A specialization of `QDoubleValidator` that honors QxLocale."""

    MAX_WIDTH: int = 15

    def __init__(
        self,
        bottom: float = -1.79769e+308,
        top: float = 1.79769e+308,
        decimals: int = -1,
        pad: bool = False,
        parent: Optional[QObject] = None
    ):
        """
        Initialize the class.

        Args:
            bottom: The minimum value to allow.
            top: The maximum value to allow.
            decimals: How many digits to allow after the decimal point. -1 means any number of digits is accepted.
            pad: Whether to pad the value with trailing zeroes so it has exactly `decimals` digits after the decimal.
            parent: The parent widget.
        """
        # Blind replacement of QDoubleValidator with QxDoubleValidator might result in the parent being passed as
        # the first parameter instead of the last by mistake. This should help with transition.
        assert not isinstance(bottom, QObject)

        self._initialized = False  # To avoid bug in base class setRange() (see setRange() below)
        super().__init__(bottom=bottom, top=top, decimals=decimals, parent=parent)
        self.pad = pad
        self.setLocale(QxLocale)
        self._initialized = True  # To avoid bug in base class setRange() (see setRange() below)
        self.setRange(bottom, top, decimals)  # To avoid bug in base class setRange() (see setRange() below)

    def validate(self, value: str, pos: int) -> tuple[QValidator.State, str, int]:
        """
        Check whether the current value is valid.

        Args:
            value: The current value of the edit field.
            pos: Position where the field is being edited.

        Returns:
            Tuple of `(state, value, pos)` where `state` is whether the field is valid, `value` is the new value to
            assign to the field, and `pos` is the new edit position.
        """
        state, value, pos = super().validate(value, pos)
        if state == QValidator.Acceptable and value != self.fixup(value):
            # If the edit field loses focus while the value is Intermediate, Qt will call self.fixup to fix it.
            return QValidator.Intermediate, value, pos
        return state, value, pos

    def fixup(self, value: str) -> str:
        """
        Make a value look nice.

        The parent class already guarantees validity. This just makes it look the way we want.

        Args:
            value: The value to be formatted.

        Returns:
            A formatted version of value.
        """
        # Make sure the value is within the range
        parsed, _valid = QxLocale.toDouble(value)  # Parent enforces valid double, so it's always valid
        if parsed < self.bottom():
            parsed = self.bottom()
        if parsed > self.top():
            parsed = self.top()

        reformatted = NumberCorrector.format_double(parsed, self.decimals(), version=3)

        if self.pad:
            decimal_location = reformatted.find('.')
            present_decimals = len(reformatted) - decimal_location - 1
            missing_decimals = self.decimals() - present_decimals
            reformatted += '0' * missing_decimals

        # The above uses . as a decimal point, which will be wrong if we ever localize.
        return reformatted.replace('.', QxLocale.decimalPoint())

    def setRange(self, bottom, top, decimals=None):  # noqa: N802
        """
        Set the range of the validator.

        We define this because the base class has a bug: if decimals isn't passed in, it will be set to 0 when it is
        supposed to remain unchanged.

        Args:
            bottom: The minimum value to allow.
            top: The maximum value to allow.
            decimals: How many digits to allow after the decimal point.
        """
        if not self._initialized:  # To avoid infinite recursion
            return

        if decimals is None:
            decimals = self.decimals()
        super().setRange(bottom, top, decimals)
