"""BinaryFileReader class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path
import struct

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.misc import util


class BinaryFileReader:
    """Reads the binary file and returns the time stamp and the values."""

    BLOCK_MARKER_SIZE = 4

    def __init__(
        self,
        filepath: Path,
        value_size: int,
        data_count: int,
        components_count: int = 1,
        nodata: float = util.nodata
    ) -> None:
        """Initializes the class.

        Args:
            filepath (Path): File path of 0.head_pm.XXXX file.
            value_size (int): Size in bytes of each value (typically 4 or 8).
            data_count (int): Number of points or cells. Pass -1 if unknown and all the values will be read.
            components_count (int): Number of components (1 for scalar, 3 for 3D vector etc).
            nodata (float): Value to assign to any values that are nan or inf.
        """
        self._filepath = filepath
        self._value_size = value_size
        self._data_count = data_count
        self._components_count = components_count
        self._nodata = nodata
        self._log = logging.getLogger('xms.hgs')

    def read(self):
        """Reads the file and returns a tuple containing the time stamp, the values, and the activity.

        Returns:
            (tuple[float, np.array, np.array]): See description.
        """
        with self._filepath.open('rb') as fp:
            time_stamp = self._read_time_stamp(fp)
            values = self._read_values(fp)
            return time_stamp, values

    def _read_time_stamp(self, fp) -> float:
        """Reads and returns the timestamp as a float."""
        self._read_block_marker(fp)
        fmt = '<80s'
        chunk = fp.read(80)
        time_stamp = struct.unpack(fmt, chunk)[0]
        time_stamp = time_stamp.decode('utf-8').strip()
        self._read_block_marker(fp)
        try:
            time_float = float(time_stamp) if time_stamp is not None and time_stamp.strip() else 0.0
        except ValueError:
            return 0.0
        return time_float

    def _read_block_marker(self, fp) -> None:
        """Reads a block marker."""
        fp.read(self.BLOCK_MARKER_SIZE)

    def _read_values(self, fp) -> 'np.ndarray | None':
        """Reads the values."""
        try:
            if self._components_count == 1:
                values = self._read_for_1_component(fp)
            else:
                values = self._read_for_multiple_components(fp)
        except Exception as exc:
            self._log.error(str(exc))
            return None

        values[np.isnan(values)] = self._nodata  # Replace nan with nodata value (GMS can't handle nan)
        values[np.isinf(values)] = self._nodata  # Replace inf with nodata value (GMS can't handle inf)
        if self._data_count == -1.0:
            return values[0:-1]
        return values

    def _read_for_multiple_components(self, fp) -> np.ndarray:
        """Reads and returns the values when there are multiple components (vectors)."""
        real_char = {4: 'f', 8: 'd'}.get(self._value_size)
        if not real_char:
            raise RuntimeError(f'Unsupported data type "{str(self._value_size)}".')

        # '<' below means little endian, and don't add bytes to account for alignment on byte boundaries
        fmt = f'<i{self._components_count}{real_char}i'  # e.g. '<i3fi': [int4][real4][real4][real4][int4]
        struct_len = struct.calcsize(fmt)
        struct_unpack = struct.Struct(fmt).unpack_from
        values = np.zeros(shape=(self._data_count, self._components_count))
        for i in range(self._data_count):
            data = fp.read(struct_len)
            if not data:
                break
            s = struct_unpack(data)
            values[i] = list(s[1:self._components_count + 1])  # drop first and last marker and copy to array
        return values

    def _read_for_1_component(self, fp) -> np.ndarray:
        """Reads and returns the values when there is one component (scalars)."""
        dtype = {4: np.float32, 8: np.float64}.get(self._value_size)
        if not dtype:
            raise RuntimeError(f'Unsupported data type "{str(self._value_size)}".')

        self._read_block_marker(fp)
        values = np.fromfile(fp, dtype=dtype, count=self._data_count)
        return values
