"""MpropsWriter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.file_io.file_data import FileData


class PropsWriterBase:
    """Base class for writing the .mprops, .oprops, and .etprops files."""
    def __init__(self, materials_data: dict, material_lookup: dict, grok_filepath: Path) -> None:
        """Initializes the class.

        Args:
            materials_data (dict): Materials data dict.
            material_lookup (dict): Dict of materials data arranged in a more useful way.
            grok_filepath (Path): Path to the .grok file.
        """
        super().__init__()
        self._materials_data = materials_data
        self._material_lookup = material_lookup
        self._grok_filepath = grok_filepath

        self._file_data = FileData()
        self._domain = ''  # Set this in derived classes
        self._extensions = {'Porous media': '.mprops', 'Surface flow': '.oprops', 'ET': '.etprops'}

    def write(self) -> Path:
        """Writes the .*props file and returns the path to the file."""
        props_filepath = self._grok_filepath.with_suffix(self._extensions[self._domain])
        with open(props_filepath, 'w') as self._file_data.file:
            for i, (_zone, material_name, uuid) in enumerate(self._material_lookup[self._domain]):
                self._file_data.data = self._materials_data[uuid]  # Dict for just this material
                self._write_material(i, material_name, uuid)
        return props_filepath

    def _write_material(self, i: int, material_name: str, uuid: str) -> None:
        """Writes the material."""
        raise NotImplementedError()  # pragma no cover - shouldn't ever get here
