"""Module for ReadTemplateRunner."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['ReadTemplateRunner']

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread

# 4. Local modules
from xms.hydroas.file_io.errors import GmiError
from xms.hydroas.file_io.gmi_reader import GmiReader


class ReadTemplateRunner(FeedbackThread):
    """Read template thread."""
    def __init__(self, file_path: str | Path):
        """
        Construct the worker.
        """
        super().__init__(create_query=False)
        self.display_text |= {
            'title': 'Reading .2dm Geometry File',
            'working_prompt': 'Reading .2dm geometry file. Please wait...',
            'error_prompt': 'Error(s) encountered reading .2dm file. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered reading .2dm file. Review log output for more details.',
            'success_prompt': 'Successfully read .2dm geometry file.',
            'auto_load': 'Close dialog automatically if no errors.',
        }
        self.file_path: str | Path = file_path
        self.model: Optional[GenericModel] = None
        self.model_values: str = ''

    def _run(self):
        """Run the thread."""
        self._log.info('Reading template file...')
        reader = GmiReader(self.file_path)
        try:
            reader.read()
        except GmiError as exc:
            raise ExpectedError(str(exc))

        section = reader.model.model_parameters
        section.restore_values(reader.model_instantiation)

        try:
            name = section.group('model').parameter('model_name').value.lower()
        except (AttributeError, KeyError):
            raise ExpectedError('File was not a valid HydroAS template: No GM card found.')

        if not (name.startswith('hydroas') or name.startswith('hydro-as') or name.startswith('hydro_as')):
            self._log.info(f'Name was {name!r}')
            raise ExpectedError('File was not a valid HydroAS template: GM card did not contain HydroAS.')

        self.model = reader.model
        self.model_values = reader.model_instantiation
        self._log.info('Done.')
