"""Class to read a .2dm file."""

# 1. Standard Python modules
from logging import Logger
from pathlib import Path
from typing import Optional, Sequence
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UGrid2d
from xms.data_objects.parameters import Polygon, Projection
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.hydroas.file_io.extra_io import read_extra_file
from xms.hydroas.file_io.gmi_builder import GmiBuilder
from xms.hydroas.file_io.gmi_fixer import fix
from xms.hydroas.file_io.gmi_parser import parse
from xms.hydroas.file_io.gmi_validator import validate


class GmiReader:
    """Class for reading a .2dm file with a GMI definition at the end."""
    def __init__(self, file_name: Path | str, projection: Optional[Projection] = None, log: Optional[Logger] = None):
        """
        Initialize the reader.

        Args:
            file_name: Path to the file to read from.
            projection: The current display projection. If geometry is read in, and does not have its own projection,
                it will be assigned the display projection.
            log: Where to log progress to.
        """
        self._file_name: Path = Path(file_name)
        self.projection: Projection = projection or Projection()
        self._log = log or Logger('xms.gmi')

        self.mesh_name: str = ''  #: Name of the mesh. Value of MESHNAME card if present, or derives from file name.
        self.ugrid: Optional[UGrid2d] = None  #: A UGrid representing the mesh from the file.
        self.ugrid_uuid: Optional[str] = None  #: UUID of self.ugrid.

        #: Represents the GMI template contained in the file. Only definitions are included here. Values are stored
        #: elsewhere, e.g. self.point_values and self.global_instantiation. If the file only contains a mesh, then
        #: self.model.is_default() will be True.
        self.model: GenericModel = GenericModel()
        #: Values for parameters in the model section.
        self.model_instantiation: str = ''
        #: Values for parameters in the global section.
        self.global_instantiation: str = ''
        #: Values for parameters in the material section. Group activity is not included in these values. See
        #: self.material_values for that.
        self.material_instantiation: str = ''

        #: Points with assigned values. Each point is a node index in the UGrid.
        self.points: list[int] = []
        #: Feature IDs for points. Parallel to self.points.
        self.point_ids: list[int] = []
        #: Values associated with each point. Parallel to self.points. Can be restored into self.model.
        self.point_values: list[str] = []
        #: Active group for each point. Parallel to self.points. Uses GMI_MULTIPLE_ASSIGNED_TYPE if multiple assigned,
        #: or GMI_NONE_ASSIGNED_TYPE if none assigned.
        self.point_types: list[int] = []

        #: Arcs (and their locations) with assigned values. Each arc is a list of node indices in the UGrid.
        self.arcs: list[list[int]] = []
        #: Feature IDs for each arc.
        self.arc_ids: list[int] = []
        #: Values associated with each arc. Parallel to self.arcs. Can be restored into self.model.
        self.arc_values: list[str] = []
        #: Active group for each arc. Parallel to self.arcs. Uses GMI_MULTIPLE_ASSIGNED_TYPE if multiple assigned,
        #: or GMI_NONE_ASSIGNED_TYPE if none assigned.
        self.arc_types: list[int] = []

        #: Polygons (and their locations) with assigned values. Since .2dm files can't express arbitrary polygons, these
        #: will each be equivalent to a cell (tri or quad) in the mesh. These will not be renumbered; IDs on the
        #: polygons will match their cell IDs in the .2dm file. There may be duplicate polygons if the .2dm file
        #: contained duplicate cells and assigned values to both, but this shouldn't happen in well-formed files. If a
        #: point in self.points is also on a polygon, then the point in the polygon will have the same ID as the one in
        #: self.points. Likewise, points that are shared by two or more polygons will have the same ID in both polygons.
        #: This can happen if two adjacent cells have values assigned. Polygons are not cleaned, so if two cells
        #: overlap in the .2dm file, their polygons will also overlap. This shouldn't happen with well-formed files.
        self.polygons: list[Polygon] = []
        #: Values associated with each polygon. Parallel to self.polygons. Can be restored into self.model.
        self.polygon_values: list[str] = []
        #: Active group for each polygon. Parallel to self.polygons. Uses GMI_MULTIPLE_ASSIGNED_TYPE if multiple
        #: assigned, or GMI_NONE_ASSIGNED_TYPE if none assigned.
        self.polygon_types: list[int] = []

        #: Material ID for each cell. Each element is the ID of the material of the cell at the same index in the UGrid.
        self.material_cells: Sequence[int] = []
        #: Active group for each material.
        self.material_types: list[str] = []

        #: Mapping from material ID to material name. A card like `MAT 1 "Material name"` will result in an entry in
        #: here mapping 1 -> 'Material name'.
        self.material_names: dict[int, str] = {}

    def read(self) -> None:
        """Read the file."""
        self.mesh_name = Path(self._file_name).stem

        with open(self._file_name) as f:
            cards, curves = parse(f, log=self._log)

        self._log.info('Patching cards...')
        fix(cards)
        self._log.info('Validating cards...')
        validate(cards, curves)
        builder = GmiBuilder(cards, curves, default_name=self.mesh_name, log=self._log)
        builder.build()

        self._log.info('Finishing read...')
        self.model = builder.model

        self.ugrid = builder.ugrid
        self.mesh_name = builder.mesh_name
        self.material_names = builder.material_names

        self.model_instantiation = builder.model_instantiation or ''
        self.global_instantiation = builder.global_instantiation or ''
        self.material_instantiation = builder.material_instantiation or ''

        self.points = builder.points
        self.point_ids = builder.point_ids
        self.point_values = builder.point_values
        self.point_types = builder.point_types

        self.arcs = builder.arcs
        self.arc_ids = builder.arc_ids
        self.arc_values = builder.arc_values
        self.arc_types = builder.arc_types

        self.polygons = builder.polygons
        self.polygon_values = builder.polygon_values
        self.polygon_types = builder.polygon_types

        self.material_cells = builder.material_cells
        self.material_types = builder.material_types

        if self.ugrid:
            mesh_uuid, projection = read_extra_file(self.mesh_name, self._file_name)
            self.ugrid_uuid = mesh_uuid or str(uuid.uuid4())
            self.projection = projection or self.projection

        self._log.info('File read successfully')
