"""Mf6Exporter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.guipy.testing import testing_tools

# 4. Local modules
from xms.mf6.components import dis_grid_comparer
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.misc import log_util


def run_script() -> bool:
    """Runs the simulation exporter when called as a script.

    Returns:
        Whether the thread completed successfully.
    """
    query = _create_query()
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
    mfsim_nam = query.item_with_uuid(
        item_uuid=sim_node.uuid, model_name="MODFLOW 6", unique_name="Sim_Manager"
    ).main_file
    export_dir = os.getcwd()
    return _export(query, mfsim_nam, sim_node, export_dir, win_cont=None)


def run_export(query: Query, mfsim_nam: str, export_dir: str, win_cont: QWidget) -> bool:
    """Runs the simulation exporter when called from the component Export command.

    Args:
        query: Object for communicating with GMS
        mfsim_nam: mfsim.nam file path of existing simulation.
        export_dir: Path where simulation is to be exported.
        win_cont: The window container.

    Returns:
        Whether the thread completed successfully.
    """
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())
    return _export(query, mfsim_nam, sim_node, export_dir, win_cont=win_cont)


def _export(query: Query, mfsim_nam: str, sim_node: TreeNode, export_dir: str, win_cont: QWidget | None) -> bool:
    """Runs the simulation exporter.

    Args:
        query: Object for communicating with GMS
        mfsim_nam: mfsim.nam file path.
        sim_node: Simulation tree node.
        export_dir: Path where simulation is to be exported.
        win_cont: The window container.

    Returns:
        Whether the thread completed successfully.
    """
    _save_tree(sim_node)
    thread = ExportFeedbackThread(query, mfsim_nam, sim_node, export_dir)
    return process_feedback_dlg.run_feedback_dialog(thread, win_cont)


class ExportFeedbackThread(FeedbackThread):
    """Thread for exporting the simulation."""
    def __init__(self, query: Query, mfsim_nam: str, sim_node: TreeNode, export_dir: str):
        """Initializes the class."""
        super().__init__(query, is_export=True)
        self._query = query
        self._mfsim_nam = mfsim_nam
        self._sim_node = sim_node
        self._export_dir = export_dir

        self.display_text |= {
            'title': 'Export MODFLOW 6 simulation',
            'working_prompt': f'Exporting MODFLOW 6 simulation \"{mfsim_nam}\".',
            'error_prompt': 'Error(s) encountered while exporting.',
            'warning_prompt': 'Warning(s) encountered while exporting.',
            'success_prompt': f'Successfully exported \"{mfsim_nam}\".',
        }

    def _run(self) -> None:
        """Does the work."""
        exporter = Mf6Exporter(self._query, self._mfsim_nam, self._sim_node, self._export_dir)
        exporter.export()


class Mf6Exporter:
    """Exports a MODFLOW 6 simulation."""
    def __init__(self, query: Query, mfsim_nam: Path | str, sim_node: TreeNode, export_dir: Path | str):
        """Initializes the class.

        Args:
            query: Object for communicating with GMS
            mfsim_nam: mfsim.nam file path.
            sim_node: Simulation tree node.
            export_dir: Path where simulation is to be exported.
        """
        self._query = query
        self._mfsim_nam = mfsim_nam
        self._sim_node = sim_node
        self._export_dir = export_dir

        self._log = log_util.get_logger()
        self._mfsim_dir = os.path.dirname(self._mfsim_nam)

    def export(self):
        """Exports this MODFLOW 6 simulation."""
        mfsim = self._read()
        self._log_dis_ugrid_mismatches(mfsim)
        mfsim.filename = os.path.join(self._export_dir, 'mfsim.nam')
        self._write(mfsim)
        self._log.info('Simulation export complete.')

    def _log_dis_ugrid_mismatches(self, mfsim: MfsimData) -> None:
        """Checks that the linked UGrids match the DIS packages in each model, and logs errors if not.

        Args:
            mfsim: The simulation.
        """
        self._log.info('Checking that DIS package matches grid...')
        errors = dis_grid_comparer.get_sim_mismatches(mfsim)
        for error in errors:
            self._log.warning(error)

    def _read(self) -> MfsimData:
        """Reads the mfsim.nam file.

        Returns:
            The simulation.
        """
        self._log.info('Reading simulation...')

        reader = io_factory.reader_from_ftype('MFSIM6')
        mfsim = reader.read(self._mfsim_nam, self._sim_node, self._query)
        return mfsim

    def _write(self, mfsim: MfsimData) -> None:
        """Writes the simulation.

        Args:
            mfsim: The simulation class.
        """
        self._log.info('Writing simulation...')

        gms_options = MfsimData.read_or_init_gms_options(self._mfsim_dir)
        writer_options = WriterOptions(
            use_open_close=gms_options.get('USE_OPEN_CLOSE', False),
            use_input_dir=gms_options.get('USE_INPUT_DIR', False),
            use_output_dir=gms_options.get('USE_OUTPUT_DIR', False),
            use_periods_db=True,
            auto_file_naming=gms_options.get('AUTO_FILE_NAMING', False)
        )
        mfsim.write(writer_options)


def _create_query() -> Query:  # pragma no cover - this is always mocked when testing
    """Creates and returns the query, and makes testing easier because this can be mocked."""
    return Query(timeout=300000)


def _save_tree(sim_node: TreeNode) -> None:
    if not sim_node:
        return
    dbg_file = 'C:/temp/debug_mf6_save_tree.dbg'
    if not Path(dbg_file).is_file():
        return

    project_tree = tree_util.ancestor_of_type(sim_node, xms_types=['TI_PROJECT'])
    testing_tools.write_tree_to_file(project_tree, 'C:/temp/project_tree.json')
