"""TdisComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query

# 4. Local modules
from xms.mf6.components.package_component_base import PackageComponentBase
from xms.mf6.file_io import io_factory
from xms.mf6.gui import gui_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.tdis_dialog import TdisDialog


class TdisComponent(PackageComponentBase):
    """The TDIS package component class."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): File path to main file.
        """
        super().__init__(main_file.strip('\'"'))
        self.ftype = 'TDIS6'  # package ftype
        self.dialog = TdisDialog

    @staticmethod
    def sto_package_data(mfsim):
        """Gather the STO package data for the model.

        Args:
            mfsim (MfsimData): The Mfsim

        Returns:
            (list): All the STO packages in mfsim
        """
        sto_data_list = []
        for model in mfsim.models:
            for package in model.packages:
                if package.ftype == 'STO6':
                    package.mname = model.mname
                    sto_data_list.append(package)
        return sto_data_list

    def _open_dialog(self, query: Query, win_cont: QWidget | None) -> list[ActionRequest]:
        """Open the TDIS Package dialog.

        Args:
            query: Object for communicating with GMS
            win_cont: The window container.

        Returns:
            list[ActionRequest]: List of action requests.
        """
        try:
            locked = query.current_item().locked
            mfsim, model, package = self.read_sim(query)
            tdis_data = mfsim.tdis

            # Add STO package data
            sto_data_list = self.sto_package_data(mfsim)

            # Run the dialog
            help_id = gui_util.help_id_from_key(self.ftype)
            dlg_input = DialogInput(data=tdis_data, locked=locked, sto_data_list=sto_data_list, help_id=help_id)
            dialog = TdisDialog(dlg_input=dlg_input, parent=win_cont)
            dialog.setModal(True)
            if dialog.exec() == QDialog.Accepted:
                # Write the data
                writer = io_factory.writer_from_ftype('TDIS6', None)
                writer.write(dialog.get_data())

                # Write STO package data if we have it
                TdisDialog.update_sto_packages(dialog, '')

        except Exception as error:
            raise error
            # messages.append(error)

        return []
