"""DisData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.dis_data_base import DisDataBase
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import options_util, units_util


class DisData(DisDataBase):
    """A DIS package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'DIS6'
        self._grid_info = self._base._grid_info if self._base._grid_info is not None else GridInfo(dis=DisEnum.DIS)
        self.add_block('GRIDDATA', ['DELR', 'DELC', 'TOP', 'BOTM', 'IDOMAIN'])

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of an array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'DELR' | 'DELC' | 'TOP' | 'BOTM':
                units_spec = units_util.UNITS_LENGTH
            case 'IDOMAIN':
                units_spec = units_util.UNITS_UNITLESS
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'IDOMAIN'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() != 'IDOMAIN'

    @override
    def can_be_layered(self, array_name: str) -> bool:
        """Returns True if the array can have the LAYERED option.

        Args:
            array_name: The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in ['BOTM', 'IDOMAIN']

    @override
    def can_be_dataset(self, array_name: str) -> bool:
        """Returns True if the array can be converted to a 3d dataset.

        Args:
            array_name: The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in ['TOP', 'BOTM', 'IDOMAIN']

    @override
    def array_size_and_layers(self, array_name: str, layered: bool) -> tuple[int, int, tuple]:
        """Returns a tuple of nvalues, layers_to_read based on array_name.

        Args:
            array_name: The name of an array.
            layered: The 'LAYERED' word was found next to the array.

        Returns:
            (tuple): tuple containing:
                - nvalues (int): Number of values to read.
                - layers_to_read (int): Number of layers to read.
                - shape (tuple) : shape of array
        """
        if not self.grid_info():
            return 0, 0, 0
        if array_name.upper() == 'DELR':
            nvalues = self.grid_info().ncol  # <delr(ncol)> column spacing in the row direction
            layers_to_read = 1
            shape = (nvalues, 1)  # nvalues rows, 1 column
        elif array_name.upper() == 'DELC':
            nvalues = self.grid_info().nrow  # <delc(nrow)> row spacing in the column direction
            layers_to_read = 1
            shape = (nvalues, 1)  # nvalues rows, 1 column
        elif array_name.upper() == 'TOP':
            # Top is special in that there is always only one layer
            nvalues, shape = ArrayLayer.number_of_values_and_shape(True, self.grid_info())
            layers_to_read = 1
        else:
            nvalues, shape = ArrayLayer.number_of_values_and_shape(layered, self.grid_info())
            layers_to_read = self.grid_info().nlay if layered else 1

        return nvalues, layers_to_read, shape

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Structured Discretization (DIS) Package'

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {'TOP': None, 'BOTM': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        return {}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        options_block = super()._setup_options()
        options_block.definitions.append(options_util.export_array_ascii_def())
        return options_block

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'DELR':
            txt = 'Column spacing in the row direction.'
        elif tab_text == 'DELC':
            txt = 'Row spacing in the column direction.'
        elif tab_text == 'TOP':
            txt = 'Top elevation for each cell in the top model layer.'
        elif tab_text == 'BOTM':
            txt = 'Bottom elevation for each cell.'
        elif tab_text == 'IDOMAIN':
            txt = (
                'Optional array that characterizes the existence status of a cell. If the IDOMAIN array is not'
                ' specified, then all model cells exist within the solution. If the IDOMAIN value for a cell is 0, the'
                ' cell does not exist in the simulation. Input and output values will be read and written for the cell,'
                ' but internal to the program, the cell is excluded from the solution. If the IDOMAIN value for a cell'
                ' is 1 or greater, the cell exists in the simulation. If the IDOMAIN value for a cell is -1, the cell'
                ' does not exist in the simulation. Furthermore, the first existing cell above will be connected to the'
                ' first existing cell below. This type of cell is referred to as a “vertical pass through” cell.'
            )
        return txt
