"""StoData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import options_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField


class StoData(GriddataBase):
    """Data class to hold the info from the STO package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'STO6'
        self.add_block('GRIDDATA', ['ICONVERT', 'SS', 'SY'])
        self.stress_periods = {}  # Dict of stress period number -> 1 (steady-state) or 0 (transient)
        self.mname = ''  # mname of GWF model from mfsim.nam file. Displayed in TDIS dialog.

    def get_units(self, array_name: str) -> str:
        """Returns a units string.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        units_str = ''
        match array_name:
            case 'ICONVERT':
                units_str = units_util.UNITS_UNITLESS
            case 'SS':
                if not self.options_block.get('STORAGECOEFFICIENT'):
                    units_str = units_util.UNITS_SS  # specific storage
            case 'SY':
                units_str = units_util.UNITS_UNITLESS
            case _:
                return ''  # This is an error
        return units_str

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'ICONVERT'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return True  # They're all required

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Storage (STO) Package'

    def set_stress_periods_from_list(self, stress_period_list):
        """Sets the self.stress_periods dict from a list of stress periods.

        The lists is assumed to be comprehensive (i.e. all stress periods are provided).
        The list should contain a 1 for steady state and a 0 for transient.

        Args:
            stress_period_list (list of int): List of 1s (ss) and 0s (tr) size of nper.
        """
        self.stress_periods.clear()
        for index, item in enumerate(stress_period_list):
            self.stress_periods[index + 1] = item

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {'SS': None, 'SY': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        return {}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                Checkbox(
                    'STORAGECOEFFICIENT', brief='SS array is read as storage coefficient rather than specific storage'
                ),
                Checkbox(
                    'SS_CONFINED_ONLY',
                    brief='Specific storage is only calculated when a cell is under confined conditions'
                ),
                CheckboxField('TVS6 FILEIN', brief='Time-varying storage (TVS) file', type_='str'),
                options_util.export_array_ascii_def(),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'ICONVERT':
            txt = (
                'Flag for each cell that specifies whether or not a cell is convertible for the storage calculation. 0'
                ' indicates confined storage is used. >0 indicates confined storage is used when head is above cell top'
                ' and a mixed formulation of unconfined and confined storage is used when head is below cell top.'
            )
        elif tab_text == 'SS':
            txt = (
                'Specific storage (or the storage coefficient if STORAGECOEFFICIENT is specified as an option).'
                ' Specific storage values must be greater than or equal to 0. If the CSUB Package is included in the'
                ' GWF model, specific storage must be zero for every cell.'
            )
        elif tab_text == 'SY':
            txt = (
                'Specific yield. Specific yield values must be greater than or equal to 0. Specific yield does not have'
                ' to be specified if there are no convertible cells (ICONVERT=0 in every cell).'
            )
        return txt
