"""LakReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shlex
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.misc import log_util


class LakReader(PackageReader):
    """Reads a LAK package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='LAK6')
        self._block_methods = {
            'PACKAGEDATA': '_handle_list_block',
            'CONNECTIONDATA': '_handle_list_block',
            'TABLES': '_handle_list_block',
            'OUTLETS': '_handle_list_block'
        }

    def _on_end_file(self):
        """Called after closing the file."""
        if 'TABLES' not in self._data.list_blocks:
            return
        logger = log_util.get_logger()
        fname = self._data.list_blocks['TABLES']
        with open(fname, 'r') as f:
            lines = f.readlines()
        with open(fname, 'w') as f:
            for line in lines:
                words = shlex.split(line, posix="win" not in sys.platform)
                if len(words) < 4:
                    logger.warning(f'Invalid line in LAK table block: {line}')
                    continue
                tab_file = fs.resolve_relative_path(os.path.dirname(self._data.filename), words[3])
                f.write(f'{words[0]} {words[1]} {words[2]} "{tab_file}"\n')
