"""SetupSwiPackageDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box, xy_series_editor
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.gwf.swi_set_up_dialog_ui import Ui_SetupSwiPackageDialog
from xms.mf6.tools.swi_tools import SwiSetUpInputs


def run_dialog(main_file: Path | str, inputs: SwiSetUpInputs, parent: QWidget) -> SwiSetUpInputs | None:
    """Opens a dialog to get the sea level time series and the conductances from the user.

    Return:
        The dataframe, or None if there was a problem.
    """
    dialog = SwiSetUpDialog(main_file, inputs, parent)
    if dialog.exec() == QDialog.Accepted:
        return dialog.inputs
    else:
        return None


class SwiSetUpDialog(XmsDlg):
    """Dialog used to set up the SWI package."""
    def __init__(self, main_file: Path | str, inputs: SwiSetUpInputs, parent: QWidget | None = None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            main_file: Package main file.
            inputs: The inputs.
            parent: The parent window.
        """
        super().__init__(parent=parent, dlg_name='xms.mf6.gui.gwf.swi_set_up_dialog')
        self.ui = Ui_SetupSwiPackageDialog()
        self.ui.setupUi(self)

        # Init private variables
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('SwiSetUpDialog'))
        self._main_file: Path | str = main_file
        self._inputs = inputs

        # Set widgets
        gui_util.set_read_only_and_grey(self.ui.edt_time_series_file, True)
        self.ui.edt_time_series_file.setText(self._inputs.sea_level_csv)
        self.ui.chk_create_ghb.setChecked(self._inputs.create_ghb)
        self.ui.spn_ghb_cond.setValue(self._inputs.ghb_cond)
        self.ui.chk_create_drn.setChecked(self._inputs.create_drn)
        self.ui.spn_drn_cond.setValue(self._inputs.drn_cond)
        self.ui.chk_fix_rch.setChecked(self._inputs.fix_rch)
        self.ui.chk_create_saltwater_head_tva.setChecked(self._inputs.create_tva_file)

        # Signals
        self.ui.btn_browse.clicked.connect(self._on_btn_browse)
        self.ui.chk_create_ghb.clicked.connect(self._do_enabling)
        self.ui.chk_create_drn.clicked.connect(self._do_enabling)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

        self._do_enabling()

    @property
    def inputs(self) -> SwiSetUpInputs:
        """Return the inputs."""
        return self._inputs

    def _do_enabling(self) -> None:
        """Enables/disables stuff."""
        self.ui.lbl_ghb_cond.setEnabled(self.ui.chk_create_ghb.isChecked())
        self.ui.spn_ghb_cond.setEnabled(self.ui.chk_create_ghb.isChecked())
        self.ui.lbl_drn_cond.setEnabled(self.ui.chk_create_drn.isChecked())
        self.ui.spn_drn_cond.setEnabled(self.ui.chk_create_drn.isChecked())

    def _on_btn_browse(self) -> None:
        """Gets the sea level time series .csv file path from the user and reads it.

        Return:
            The dataframe, or None if there was a problem.
        """
        filepath = self._get_time_series_file()
        if not filepath:
            return

        self._read_csv_file_or_warn(filepath)

    def _get_time_series_file(self) -> Path | None:
        """Gets a file path from the user via the Open File dialog."""
        ts_file = self.ui.edt_time_series_file.text()
        filter_str = 'CSV (Comma delimited) Files (*.csv);;Text files (*.txt);;All Files (*.*)'
        ts_file = gui_util.run_open_file_dialog(self, 'Time Series File', ts_file, filter_str)
        if ts_file:
            self.ui.edt_time_series_file.setText(ts_file)
            return Path(ts_file)
        else:
            return None

    def _read_csv_file_or_warn(self, filepath: Path | str) -> pd.DataFrame | None:
        """Read the csv file and warn user if it couldn't be read.

        Args:
            filepath: The file path.

        Returns:
            True if OK, False if error.
        """
        try:
            return xy_series_editor.import_csv_file(filepath, [])
        except Exception:
            msg = f'Error reading time series file "{str(filepath)}".'
            message_box.message_with_ok(parent=self, message=msg)
            return None

    def accept(self):
        """OK button clicked."""
        # Read time series file to a dataframe. Warn and abort if there is trouble.
        filepath = self.ui.edt_time_series_file.text()
        df = self._read_csv_file_or_warn(filepath)
        if df is None:
            return

        # Set inputs
        self._inputs.sea_level_csv = filepath
        self._inputs.sea_level_df = df
        self._inputs.create_ghb = self.ui.chk_create_ghb.isChecked()
        self._inputs.ghb_cond = self.ui.spn_ghb_cond.value()
        self._inputs.create_drn = self.ui.chk_create_drn.isChecked()
        self._inputs.drn_cond = self.ui.spn_drn_cond.value()
        self._inputs.fix_rch = self.ui.chk_fix_rch.isChecked()
        self._inputs.create_tva_file = self.ui.chk_create_saltwater_head_tva.isChecked()
        super().accept()
