"""Base class for GwfDialog and GwtDialog classes."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase


class ModelDialogBase(PackageDialogBase):
    """Base class for GwfDialog and GwtDialog classes."""

    # Columns in the PACKAGES table
    PACKAGE_FTYPE_COL = 0
    PACKAGE_FNAME_COL = 1
    PACKAGE_PNAME_COL = 2

    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'PACKAGES']
        self.default_sections = ['PACKAGES']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'PACKAGES':
            self.setup_packages_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_packages_section(self):
        """Sets up the PACKAGES block."""
        name = 'PACKAGES'
        self.add_group_box_to_scroll_area(name)
        self._add_auto_name_option()
        self._add_packages_table()
        model = self._load_package_data_into_table()
        self._add_header_tool_tips(model)
        self._set_read_only_columns(model)

    def _add_packages_table(self):
        name = 'PACKAGES'
        self.uix[name]['table_packages'] = gui_util.new_table_view()
        self.uix[name]['layout'].addWidget(self.uix[name]['table_packages'])
        self.uix[name]['txt_pnames'] =\
            QLabel('("pname" is the package\'s Project Explorer item name and cannot be changed here.)')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_pnames'])

    def _add_auto_name_option(self):
        # GMS auto name option
        if self._auto_file_naming():
            label_str = (
                'File names are being set automatically by GMS. To disable this,'
                ' go to the Simulation Options dialog and, under "GMS Options", turn off the'
                ' "Automatically name all files..." check box.'
            )
            txt = self.uix['PACKAGES']['txt_auto_name'] = QLabel(label_str)
            gui_util.set_label_styles_for_warning_message(txt)
            self.uix['PACKAGES']['layout'].addWidget(txt)

    def _auto_file_naming(self):
        """Returns True if the Auto name checkbox is checked.

        Returns:
            (bool): See description.
        """
        return self.dlg_input.data.mfsim.gms_options.get('AUTO_FILE_NAMING', False)

    def _load_package_data_into_table(self):
        # Load the package data into the table using a dataframe
        ftypes = []
        fnames = []
        pnames = []
        for package_data in self.dlg_input.data.packages:
            ftypes.append(package_data.ftype)
            fnames.append(os.path.basename(package_data.fname))
            pnames.append(package_data.pname)
        if self._auto_file_naming():
            fnames = data_util.compute_auto_names(self.dlg_input.data.packages, self.dlg_input.data.mname)
        df_dict = {'FTYPE': ftypes, 'FNAME': fnames, 'PNAME': pnames}
        df = pd.DataFrame(df_dict)
        model = QxPandasTableModel(df)
        self.uix['PACKAGES']['table_packages'].setModel(model)
        self.uix['PACKAGES']['table_packages'].verticalHeader().hide()
        return model

    def _set_read_only_columns(self, model):
        columns = {self.PACKAGE_FTYPE_COL, self.PACKAGE_PNAME_COL}
        if self.dlg_input.locked or self._auto_file_naming():
            columns.update({self.PACKAGE_FNAME_COL})
        model.set_read_only_columns(columns)

    def _add_header_tool_tips(self, model):
        model.set_horizontal_header_tooltips(
            {
                0: 'File type ("DIS6", "WEL6" etc.)',
                1: 'Name of the file containing the package input',
                2:
                    'User-defined name for the package, restricted to 16 characters. No spaces are allowed.'
                    ' Same as the name of the item in the Project Explorer.'
            }
        )

    def save_packages(self):
        """Saves changes the user may have made to the packages."""
        name = 'PACKAGES'
        model = self.uix[name]['table_packages'].model()
        for row in range(model.rowCount()):
            filename = model.index(row, 1).data()
            # f = os.path.join(os.path.dirname(self.dlg_input.data.packages[row].filename), filename)
            self.dlg_input.data.packages[row].fname = filename

    def accept(self):
        """Called when user hits the OK button. Saves the widget state to self.dlg_input.data and closes the dialog."""
        if not self.dlg_input.locked:
            self.dlg_input.data.gui_edit_active = True
            self.save_comments()
            self.save_packages()
        super().accept()
