"""ObsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QAbstractItemView, QHBoxLayout, QLabel, QListWidget, QListWidgetItem, QSplitter, QToolBar, QVBoxLayout, QWidget
)
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.resources.svgs import ADD_SVG, DELETE_SVG, ROW_ADD_SVG, ROW_DELETE_SVG, ROW_INSERT_SVG
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget


class ObsDialog(PackageDialogBase):
    """A dialog showing the package data."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.actions = None
        self.ui_obs = {}  # Widgets in the OBSERVATIONS block
        self.current_filename = ''

        self.setup_ui()

    @override
    def clear_sections(self) -> None:
        """Delete all section widgets."""
        self.actions = None
        self.ui_obs = {}  # Widgets in the OBSERVATIONS block
        self.current_filename = ''
        super().clear_sections()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'OBSERVATIONS']
        self.default_sections = ['OBSERVATIONS']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'OBSERVATIONS':
            self.setup_observations_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_observations_section(self):
        """Sets up the observations section."""
        section = 'OBSERVATIONS'
        self.add_group_box_to_scroll_area(section)

        # Files stuff
        self.ui_obs['vlay_files'] = QVBoxLayout()
        self.ui_obs['txt_files'] = QLabel('Files:')
        self.ui_obs['vlay_files'].addWidget(self.ui_obs['txt_files'])
        self.ui_obs['lst_files'] = QListWidget()
        self.ui_obs['lst_files'].setDragDropMode(QAbstractItemView.InternalMove)
        self.ui_obs['vlay_files'].addWidget(self.ui_obs['lst_files'])
        self.ui_obs['hlay_file_toolbar'] = QHBoxLayout()
        w = self.ui_obs['toolbar'] = QToolBar()
        button_list = [
            [ADD_SVG, 'Add File', self.on_btn_add_file],
            [DELETE_SVG, 'Delete File', self.on_btn_delete_file],
        ]
        self.actions = widget_builder.setup_toolbar(w, button_list)
        self.ui_obs['hlay_file_toolbar'].addWidget(self.ui_obs['toolbar'])
        self.ui_obs['hlay_file_toolbar'].addStretch()
        self.ui_obs['vlay_files'].addLayout(self.ui_obs['hlay_file_toolbar'])

        # Observations stuff

        # Need to use widgets with the splitter, not layouts
        self.ui_obs['wgt_files'] = QWidget(self)
        self.ui_obs['wgt_files'].setLayout(self.ui_obs['vlay_files'])
        self.ui_obs['tbl_wgt'] = ListBlockTableWidget(self, self.dlg_input, 'Observations')

        # Splitter. The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        splitter = QSplitter(self)
        splitter.setOrientation(Qt.Vertical)
        splitter.addWidget(self.ui_obs['wgt_files'])
        splitter.addWidget(self.ui_obs['tbl_wgt'])
        splitter.setMinimumHeight(400)
        splitter.setSizes([100, 300])
        splitter.setChildrenCollapsible(False)
        # splitter.setStyleSheet('QSplitter::handle:horizontal { border: 1px outset darkgrey; }'
        #                        'QSplitter::handle:vertical { border: 1px outset darkgrey; }')
        splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        pos = self.uix[section]['layout'].indexOf(self.ui_obs['tbl_wgt'])
        self.uix[section]['layout'].insertWidget(pos, splitter)

        # Context menus
        if not self.dlg_input.locked:
            self.ui_obs['lst_files'].setContextMenuPolicy(Qt.CustomContextMenu)
            self.ui_obs['lst_files'].customContextMenuRequested.connect(self.on_right_click_files)

        # Load the data
        self.setup_file_list()
        self.loaded = True
        self.ui_obs['tbl_wgt'].setup_models()
        self.ui_obs['tbl_wgt'].load_block(self.current_filename)

    def setup_signals(self):
        """Sets up any needed signals."""
        super().setup_signals()
        self.ui_obs['lst_files'].currentItemChanged.connect(self.on_lst_changed)
        self.ui_obs['lst_files'].itemChanged.connect(self.on_lst_item_changed)

    def get_existing_filenames(self):
        """Returns the set of all filenames in the list.

        Returns:
            (set of str): Set of all filenames in the list.
        """
        names = set()
        for i in range(self.ui_obs['lst_files'].count()):
            names.add(self.ui_obs['lst_files'].item(i).text())
        return names

    def get_unique_name(self):
        """Returns a new filename different from all existing filenames.

        Returns:
            (str): A unique filename.
        """
        existing_names = self.get_existing_filenames()
        new_name = 'new file'
        i = 1
        while new_name in existing_names:
            new_name = f'new file {i}'
            i += 1

        return new_name

    def add_or_insert_file(self, row):
        """Inserts a new row in the list of files."""
        new_name = self.get_unique_name()
        item = QListWidgetItem(new_name)
        if self.dlg_input.locked:
            item.setFlags(item.flags() & ~Qt.ItemIsEditable)
        else:
            item.setFlags(item.flags() | Qt.ItemIsEditable)
        self.ui_obs['lst_files'].insertItem(row, item)

        self.ui_obs['tbl_wgt'].add_default_dataframe(new_name)
        if row < 0:
            row = 0
        self.ui_obs['lst_files'].setCurrentRow(row)
        self.do_enabling()

    def on_btn_insert_file(self):
        """Inserts a new row in the list of files."""
        self.add_or_insert_file(self.ui_obs['lst_files'].currentRow())

    def on_btn_add_file(self):
        """Inserts a new row in the list of files."""
        self.add_or_insert_file(self.ui_obs['lst_files'].count())

    def on_btn_delete_file(self):
        """Deletes a row in the list of files."""
        lst_files = self.ui_obs['lst_files']
        current_row = lst_files.currentRow()
        if current_row < 0:
            return

        filename = lst_files.item(current_row).text()
        lst_files.takeItem(current_row)
        self.ui_obs['tbl_wgt'].delete_model(filename)
        current_row = current_row - 1 if current_row >= lst_files.count() else current_row
        lst_files.setCurrentRow(current_row)
        self.do_enabling()

    def on_btn_up(self):
        """Moves the current row in the list of files up one."""
        lst_files = self.ui_obs['lst_files']
        current_row = lst_files.currentRow()
        if current_row == 0:
            return

        current_item = lst_files.takeItem(current_row)
        lst_files.insertItem(current_row - 1, current_item)
        lst_files.setCurrentRow(current_row - 1)

    def on_btn_down(self):
        """Moves the current row in the list of files down one."""
        lst_files = self.ui_obs['lst_files']
        current_row = lst_files.currentRow()
        if current_row == lst_files.count() - 1:
            return

        current_item = lst_files.takeItem(current_row)
        lst_files.insertItem(current_row + 1, current_item)
        lst_files.setCurrentRow(current_row + 1)

    def on_right_click_files(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(QPoint): The point clicked.
        """
        # row = self.ui.table_view.logicalIndexAt(point)
        menu_list = [
            [ROW_INSERT_SVG, 'Insert', self.on_btn_insert_file], [ROW_ADD_SVG, 'Add', self.on_btn_add_file],
            [ROW_DELETE_SVG, 'Delete', self.on_btn_delete_file]
        ]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui_obs['lst_files'].viewport().mapToGlobal(point))

    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        super().do_enabling()
        current_row = self.ui_obs['lst_files'].currentRow()
        w = self.ui_obs['toolbar'].widgetForAction(self.actions[':/resources/icons/add.svg'])
        w.setEnabled(not self.dlg_input.locked)
        w = self.ui_obs['toolbar'].widgetForAction(self.actions[':/resources/icons/delete.svg'])
        w.setEnabled(not self.dlg_input.locked and current_row >= 0)
        self.ui_obs['tbl_wgt'].setEnabled(current_row >= 0)
        self.ui_obs['tbl_wgt'].do_enabling()

    def setup_file_list(self):
        """Sets up the file list."""
        self.ui_obs['lst_files'].clear()
        for block in self.dlg_input.data.list_blocks.keys():
            item = QListWidgetItem(block, self.ui_obs['lst_files'])
            if self.dlg_input.locked:
                item.setFlags(item.flags() & ~Qt.ItemIsEditable)
            else:
                item.setFlags(item.flags() | Qt.ItemIsEditable)

        self.ui_obs['lst_files'].setCurrentRow(0)
        current_item = self.ui_obs['lst_files'].currentItem()
        if current_item:
            self.current_filename = current_item.text()

    def on_lst_changed(self, current_item, previous_item):
        """Called when the current item in the list is changed.

        Args:
            current_item: The current item.
            previous_item: The previous item.
        """
        if current_item:
            self.current_filename = current_item.text()

        self.ui_obs['tbl_wgt'].load_block(self.current_filename)
        self.do_enabling()

    def on_lst_item_changed(self, item):
        """Called when the current item has been renamed.

        Args:
            item: The item.
        """
        if not self.loaded or not item:
            return

        new_name = item.text()
        self.ui_obs['tbl_wgt'].set_model(self.ui_obs['tbl_wgt'].get_model(self.current_filename), new_name)
        self.ui_obs['tbl_wgt'].delete_model(self.current_filename)

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.ui_obs['tbl_wgt'].accept()

    @classmethod
    def run_dialog_on_file(cls, filepath, base_file_data, locked, parent):
        """Function called when Edit button in List dialog is clicked.

        See ListDialog.on_btn_edit

        Args:
            filepath (str): Filepath of obs file.
            base_file_data (BaseFileData): BaseFileData object.
            locked (bool): True if locked.
            parent: Parent widget
        """
        from xms.mf6.components import dialog_runner
        dialog_runner.run_dialog_with_file(filepath, base_file_data, locked, parent, 'OBS6', ObsDialog)
