"""PackageDialogBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import cProfile
import io
import os
import pstats
from pstats import SortKey

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QDialog,
    QGroupBox,
    QListWidgetItem,
    QPlainTextEdit,
    QSizePolicy,
    QSpacerItem,
    QVBoxLayout,
    QWidget,
)
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.components import arrays_to_datasets, file_exporter, file_importer, map_runner
from xms.mf6.data import data_util
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.gwf.array_package_data import ArrayPackageData
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util, list_dialog, map_from_coverage_dialog
from xms.mf6.gui.list_dialog import ListDialog
from xms.mf6.gui.map_from_coverage_dialog import MapOpt
from xms.mf6.gui.package_dialog_base_ui import Ui_package_dialog_base
from xms.mf6.misc import log_util, util
from xms.mf6.misc.settings import Settings


class PackageDialogBase(XmsDlg):
    """A dialog used as a base class for package dialogs."""

    IMPORTING = 123  # result code indicating the user is importing a package

    def __init__(self, dlg_input, parent):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.package_dialog_base')
        self.dlg_input = dlg_input
        self.sections = []  # All the sections to show in the list
        self.default_sections = []  # Sections to show by default
        self.ui: Ui_package_dialog_base | None = None
        self.uix = {}  # dictionary of widgets for sections
        self.loaded = False
        self.hiding_or_showing_in_progress = False
        self.blocks_requiring_spacer = ['COMMENTS', 'OPTIONS']
        self.options_gui = None
        self.help_getter = gui_util.help_getter(self.dlg_input.help_id) if self.dlg_input else None
        self._log = log_util.get_logger()
        self.sp_widget = None  # Pointer to either self.uix['PERIODS']['sp_list_widget'] or 'sp_array_widget'
        self._save_geom = True  # Used by CellPropertiesDialog to not save geometry of child dialogs
        self._setup_base_ui()

    # @abstractmethod
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        raise NotImplementedError()

    def setup_section(self, section_name: str) -> None:
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'COMMENTS':
            self.setup_comment_section()
        elif section_name == 'OPTIONS':
            self.setup_options_section()
        else:
            raise RuntimeError()

    def _setup_base_ui(self):
        """Set up the UI for the base dialog - things that only should be done once - not for the dynamic sections."""
        self.ui = Ui_package_dialog_base()
        self.ui.setupUi(self)

        if not self.dlg_input:
            return

        self.setWindowTitle(self.dlg_input.data.dialog_title())
        gui_util.handle_read_only_notice(self.dlg_input.locked, self.ui.txt_read_only)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

        # Signals
        self.ui.lst_sections.itemChanged.connect(self._on_section_list_item_changed)
        self.ui.btn_map_from_coverage.clicked.connect(self._on_btn_map_from_coverage)
        self.ui.btn_arrays_to_datasets.clicked.connect(self._on_btn_arrays_to_datasets)
        self.ui.btn_import.clicked.connect(self._on_btn_import)
        self.ui.btn_export.clicked.connect(self._on_btn_export)

        # Enable stuff
        locked = self.dlg_input.locked
        self.ui.btn_map_from_coverage.setVisible(self._show_map_from_coverage())
        self.ui.btn_map_from_coverage.setEnabled(not locked)
        show_arrays_to_dataset = isinstance(self.dlg_input.data, (GriddataBase, ArrayPackageData))
        self.ui.btn_arrays_to_datasets.setVisible(show_arrays_to_dataset)
        # Must disable Arrays to Datasets when locked because it changes stuff. Could probably fix that.
        self.ui.btn_arrays_to_datasets.setEnabled(show_arrays_to_dataset and not locked)
        self.ui.btn_activate_cells.setVisible(False)  # This is handled by DisDialogBase
        self.ui.btn_import.setVisible(self.dlg_input.data.can_import)
        self.ui.btn_import.setEnabled(self.dlg_input.data.can_import and not locked)
        self.ui.btn_export.setVisible(self.dlg_input.data.can_export)
        self.ui.btn_export.setEnabled(self.dlg_input.data.can_export)

    def _show_map_from_coverage(self) -> bool:
        """Return True if the Map From Coverage button should be visible."""
        data = self.dlg_input.data  # for short
        if hasattr(data,
                   'map_info') and (data.map_info('points') or data.map_info('arcs') or data.map_info('polygons')):
            return True
        return False

    def _setup_sections_and_profile(self) -> None:
        """Profiles _setup_sections() and writes report to C:/temp/profile.txt.

        See https://docs.python.org/3/library/profile.html
        """
        profiler = cProfile.Profile()
        profiler.runcall(self._setup_sections)
        s = io.StringIO()
        sortby = SortKey.CUMULATIVE
        ps = pstats.Stats(profiler, stream=s).sort_stats(sortby)
        ps.print_stats()
        with open('C:/temp/profile.txt', 'w') as file:
            file.write(s.getvalue())

    def _setup_sections(self) -> None:
        """Call setup function for each section."""
        for section in self.sections:
            self.setup_section(section)

    def setup_ui(self) -> None:
        """Set up everything dealing with sections.

        This is called by derived classes in their initializers.
        """
        self.loaded = False
        self.define_sections()
        self._setup_scroll_area()
        if util.file_hack('debug_mf6_profile_setup_sections.dbg'):
            self._setup_sections_and_profile()
        else:
            self._setup_sections()
        self.setup_section_list()
        self.set_section_visibility_and_enabling()
        self.setup_signals()
        self.do_enabling()
        self.loaded = True

    def setup_signals(self):
        """Sets up any needed signals."""
        pass

    def do_enabling(self):
        """Enables and disables widgets appropriately."""
        pass

    def _setup_scroll_area(self):
        """Sets up the scroll area on the right side of the dialog."""
        self.uix['scroll_area'] = {}
        scroll = self.uix['scroll_area']
        scroll['central'] = QWidget()
        scroll['layout'] = QVBoxLayout(scroll['central'])
        self.ui.scroll_area.setWidget(scroll['central'])
        self.ui.scroll_area.setWidgetResizable(True)

    def clear_sections(self) -> None:
        """Delete all section widgets."""
        self._clear_scroll_area()

    def _clear_scroll_area(self) -> None:
        """Delete everything in the scroll area."""
        self.uix = {}
        scroll_area_widget = self.ui.scroll_area.takeWidget()
        # self._clear_layout(scroll_area_widget.layout())
        scroll_area_widget.deleteLater()

    # def _clear_layout(self, layout: QLayout):
    #     if layout is None:
    #         return
    #     while layout.count():
    #         item = layout.takeAt(0)
    #         if item.widget():
    #             item.widget().deleteLater()
    #         elif item.layout():
    #             self._clear_layout(item.layout())  # Recurse
    #         del item

    def add_group_box_to_scroll_area(self, name):
        """Creates a group box with layout and 'name' as the text.

        Args:
            name (str): Name of the section shown in the group box.

        Returns:
            (QGroupBox): The group box.
        """
        self.uix[name] = {}
        self.uix[name]['group'] = QGroupBox(name)
        self.uix[name]['layout'] = QVBoxLayout()
        self.uix[name]['group'].setLayout(self.uix[name]['layout'])

        # Add to scroll area
        self.uix['scroll_area']['layout'].addWidget(self.uix[name]['group'])
        return self.uix[name]['group']

    def _on_section_list_item_changed(self, item):
        """Slot called when itemChanged signal sent.

        Args:
            item (QTreeWidgetItem): The item that was changed.
        """
        if item:
            name = item.text()
            visible = item.checkState() == Qt.Checked
            self.uix[name]['group'].setVisible(visible)
            if not self.hiding_or_showing_in_progress:
                self.add_or_remove_stretch_at_bottom()

    def setup_section_list(self):
        """Sets up the list of sections on the left side of the dialog."""
        # Preserve the list if it already matches the sections
        names = [self.ui.lst_sections.item(i).text() for i in range(self.ui.lst_sections.count())]
        if names == self.sections:
            return

        self.ui.lst_sections.clear()

        with gui_util.SignalBlocker(self.ui.lst_sections):  # So setCheckState() doesn't send signal
            states = Settings.get(self.dlg_input.data.filename, 'SECTION_STATES')
            for section_name in self.sections:
                item = QListWidgetItem(section_name)
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                if states and item.text() in states:
                    item.setCheckState(Qt.Checked if states[item.text()] else Qt.Unchecked)
                elif item.text() in self.default_sections:
                    item.setCheckState(Qt.Checked)
                else:
                    item.setCheckState(Qt.Unchecked)
                self.ui.lst_sections.addItem(item)
        self.ui.lst_sections.setMinimumWidth(self.ui.lst_sections.sizeHintForColumn(0))

    def set_section_visibility_and_enabling(self):
        """Hides or shows the sections."""
        self.hiding_or_showing_in_progress = True
        for row in range(self.ui.lst_sections.count()):
            item = self.ui.lst_sections.item(row)
            self._on_section_list_item_changed(item)
            # This is too blunt force. The widgets will enable/disable themselves
            # item_text = item.text()
            # self.enable_layout_items(self.uix[item_text]['layout'], not self.dlg_input.locked)
        self.hiding_or_showing_in_progress = False
        self.add_or_remove_stretch_at_bottom()

    def add_or_remove_stretch_at_bottom(self):
        """Adds or removes a QSpacerItem at the bottom of the scroll area to make things look nice.

        If Comments and Options are the only things shown, a spacer at the
        bottom keeps them together and that looks better. If a table is shown,
        that looks better full size we don't want a spacer. We find the last
        visible thing and see if it is one of the default sections and if so,
        don't add a spacer, otherwise we add one. This assumes that default
        things are tables.
        """
        scroll_area_layout = self.uix['scroll_area']['layout']
        spacer_index = -1
        need_spacer = True
        for index in range(scroll_area_layout.count() - 1, -1, -1):
            item = scroll_area_layout.itemAt(index)
            if item:
                if isinstance(item, QSpacerItem):
                    spacer_index = index
                else:
                    group_box = item.widget()
                    # if group_box and group_box.isVisible():
                    # For some reason the above doesn't work. We'll use the list for visibility.
                    if self.ui.lst_sections.item(index).checkState() == Qt.Checked:
                        need_spacer = group_box.title() in self.blocks_requiring_spacer
                        break

        if not need_spacer and spacer_index >= 0:
            scroll_area_layout.takeAt(spacer_index)
        elif need_spacer and spacer_index < 0:
            scroll_area_layout.addStretch(1)

    def setup_comment_section(self):
        """Sets up the comment section. In this base class because all comment sections are identical."""
        name = 'COMMENTS'
        self.add_group_box_to_scroll_area(name)

        # Add widgets
        self.uix[name]['edt_comments'] = QPlainTextEdit()
        self.uix[name]['edt_comments'].setWhatsThis(
            'Comments that will get written at the top of the file, '
            'preceded by a \'#\' symbol.'
        )
        self.uix[name]['layout'].addWidget(self.uix[name]['edt_comments'])
        widget_builder.set_textedit_height(self.uix[name]['edt_comments'], io_util.comment_lines)

        # set size policy for group box so it only expands horizontally
        self.uix[name]['group'].setSizePolicy(QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Maximum))

        for line in self.dlg_input.data.comments:
            self.uix['COMMENTS']['edt_comments'].appendPlainText(line)

        self.uix['COMMENTS']['edt_comments'].setEnabled(not self.dlg_input.locked)

    def setup_options_section(self):
        """Sets up the OPTIONS section."""
        name = 'OPTIONS'
        self.add_group_box_to_scroll_area(name)

        # Add widgets
        self.setup_options(self.uix[name]['layout'])

        # set size policy for group box so it only expands horizontally
        self.uix[name]['group'].setSizePolicy(QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Maximum))

        self.enable_layout_items(self.uix[name]['layout'], not self.dlg_input.locked)

    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout: The layout that the option widgets will be added to.
        """
        pass

    def save_comments(self):
        """Saves the comments."""
        text = self.uix['COMMENTS']['edt_comments'].toPlainText()
        self.dlg_input.data.comments = text.splitlines()

    def enable_layout_items(self, layout, enable: bool):
        """Makes the OPTIONS section read only.

        Note that this only works on widgets, not layouts. See options_util.create_hlayout_widget().
        """
        for index in range(layout.count()):
            layout_item = layout.itemAt(index)
            widget = layout_item.widget()
            if widget:
                widget.setEnabled(enable)

    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.

        Returns:
            (ListBlockTableWidget): The current ListBlockTableWidget object.
        """
        return None

    def save_list_blocks_to_temp(self):
        """If changes have been made, saves changes to temporary files."""
        for block, filename in self.dlg_input.data.list_blocks.items():
            table_widget = self.get_table_widget(block)
            if table_widget.data_changed:
                model = table_widget.get_model(block)
                temp_filename = self.dlg_input.data.dataframe_to_temp_file(block, model.data_frame)
                if io_util.is_temporary_file(filename):
                    os.remove(filename)
                self.dlg_input.data.list_blocks[block] = temp_filename

    def on_btn_tas6_filein(self):
        """Called when a FILEIN button is clicked."""
        list_dialog.run_list_dialog_for_filein('TAS6', self, self.options_gui.options_block)

    def on_btn_ts6_filein(self):
        """Called when a FILEIN button is clicked."""
        list_dialog.run_list_dialog_for_filein('TS6', self, self.options_gui.options_block)

    def on_btn_tva6_filein(self):
        """Called when a FILEIN button is clicked."""
        list_dialog.run_list_dialog_for_filein('TVA6', self, self.options_gui.options_block)

    def on_btn_obs6_filein(self):
        """Called when a FILEIN button is clicked."""
        list_dialog.run_list_dialog_for_filein('OBS6', self, self.options_gui.options_block)

    def _get_sp_widget(self):
        """Returns the PeriodListWidget.

        Returns:
            (PeriodListWidget): See description.
        """
        return self.sp_widget

    def _get_aux_widget(self):
        """Returns the widget that includes aux columns (PeriodListWidget or ListBlockTableWidget).
        """
        return self._get_sp_widget()  # This one is common so we'll make it the default

    def on_chk_auxiliary(self, checked):
        """Called when the AUXILIARY checkbox is clicked."""
        aux_block = self.dlg_input.data.block_with_aux()
        aux_widget = self._get_aux_widget()
        if aux_widget:
            aux_widget.change_aux_variables(block=aux_block, use_aux=(checked == Qt.Checked))

    def on_btn_auxiliary(self):
        """Opens AUX variable dialog and updates AUX in all stress periods."""
        options_block = self.options_gui.options_block
        the_list = options_block.get('AUXILIARY', [])
        dialog = ListDialog(
            dialog_title='Auxiliary Variables',
            the_list=the_list,
            locked=self.dlg_input.locked,
            parent=self,
            query=self.dlg_input.query
        )
        if dialog.exec() == QDialog.Accepted:
            # Check for duplicate aux variables
            if len(set(dialog.the_list)) != len(dialog.the_list):
                msg = 'Duplicate auxiliary names in list. Duplicates not allowed. Action aborted.'
                message_box.message_with_ok(parent=self, message=msg)
                return

            # Let aux widget check if the new aux list is valid
            aux_block = self.dlg_input.data.block_with_aux()
            aux_widget = self._get_aux_widget()
            error = aux_widget.check_aux_change(aux_block, dialog.the_list)
            if error:
                message_box.message_with_ok(parent=self, message=error, icon='Warning')
                return

            options_block.set('AUXILIARY', True, dialog.the_list)
            if aux_widget:
                aux_widget.change_aux_variables(block=aux_block, use_aux=True)

    def _on_btn_map_from_coverage(self) -> None:
        """Open coverage selector and map the coverage(s) to the package."""
        ok, dialog_outputs = map_from_coverage_dialog.run_from_package(self.dlg_input, self)
        if not ok:
            return

        try:
            if dialog_outputs.append_or_replace == MapOpt.APPEND:
                self.widgets_to_data()

            self.clear_sections()  # Do this before reading so database gets closed
            rv = map_runner.map_from_coverage(self, dialog_outputs)
            if not rv:
                msg = 'Errors mapping from coverage.'
                message_box.message_with_ok(parent=self, message=msg, icon='Warning')
                self.setup_ui()
                return
            self.dlg_input.restore_on_cancel = True
            self.setup_ui()
            msg = 'Mapping successful'
            message_box.message_with_ok(parent=self, message=msg, icon='Information')
        except Exception as ex:
            msg = f'Error: "{str(ex)}"'
            message_box.message_with_ok(parent=self, message=msg, icon='Warning')

    def _on_btn_arrays_to_datasets(self) -> None:
        """Make UGrid datasets from arrays."""
        self.widgets_to_data()
        self.dlg_input.restore_on_cancel = True
        if arrays_to_datasets.ask_and_create(self):
            msg = 'Datasets will be added when exiting the dialog.'
            message_box.message_with_ok(parent=self, message=msg, icon='Information')

    def _on_btn_import(self) -> None:
        """Open the Open File dialog and imports the file selected by the user."""
        filepath = self._run_open_file_dialog()
        if not filepath:
            return
        old_data = self.dlg_input.data
        try:
            self.clear_sections()  # Do this before reading so database gets closed
            new_data = file_importer.import_file(filepath, self.dlg_input.data, self)
            if new_data is None or new_data.readasarrays != old_data.readasarrays:
                msg = 'Could not import file.'
                if new_data and new_data.readasarrays != old_data.readasarrays:
                    msg = (
                        'Importing a file that has READASARRAYS into one that does not'
                        ' (or vice-versa) is not supported. Import aborted.'
                    )
                message_box.message_with_ok(parent=self, message=msg, icon='Warning')
                self.setup_ui()
                return
            self.dlg_input.data = new_data
            self.dlg_input.restore_on_cancel = True
            self.setup_ui()
            msg = 'File imported successfully'
            message_box.message_with_ok(parent=self, message=msg, icon='Information')
        except Exception as ex:
            msg = "Could not import file."
            message_box.message_with_ok(parent=self, message=msg, icon='Warning', details=str(ex))
            self.dlg_input.data = old_data
            self.dlg_input.restore_on_cancel = True
            self.setup_ui()

    def _on_btn_export(self) -> None:
        """Open the Save File dialog and exports the package to the file selected by the user."""
        filepath = self._run_save_file_dialog()
        if not filepath:
            return
        self.widgets_to_data()
        rv = file_exporter.export_file(self.dlg_input.data, filepath, self)
        self.dlg_input.restore_on_cancel = True
        msg = 'File exported successfully' if rv else 'Errors exporting file'
        icon = 'Information' if rv else 'Warning'
        message_box.message_with_ok(parent=self, message=msg, icon=icon)

    def _run_open_file_dialog(self) -> str:
        """Run the import dialog."""
        filepath = Settings.get(self.dlg_input.data.filename, 'LAST_IMPORT_FILE', '')
        file_filter = data_util.filter_from_ftype(self.dlg_input.data.ftype)
        filters = f'All Files (*.*);;{file_filter}'
        filepath = gui_util.run_open_file_dialog(self, 'Import', filepath, filters, file_filter)
        if filepath:
            Settings.set(self.dlg_input.data.filename, 'LAST_IMPORT_FILE', filepath)
        return filepath

    def _run_save_file_dialog(self) -> str:
        """Run the export dialog."""
        filepath = Settings.get(self.dlg_input.data.filename, 'LAST_EXPORT_FILE', '')
        file_filter = data_util.filter_from_ftype(self.dlg_input.data.ftype)
        filepath = gui_util.run_save_file_dialog(self, 'Export', filepath, file_filter)
        if filepath:
            Settings.set(self.dlg_input.data.filename, 'LAST_EXPORT_FILE', filepath)
        return filepath

    def _save_section_states(self) -> None:
        """Save which sections are turned on and off."""
        states = {}
        for row in range(self.ui.lst_sections.count()):
            item = self.ui.lst_sections.item(row)
            states[item.text()] = bool(item.checkState() == Qt.Checked)
        Settings.set(self.dlg_input.data.filename, 'SECTION_STATES', states)

    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        if not self.dlg_input.locked:
            self.save_comments()
            if self.options_gui:
                self.options_gui.save()

    @override
    def _save_geometry(self) -> None:
        """Saves current dialog size and position, if self._save_geom is True."""
        if self._save_geom:
            super()._save_geometry()

    def accept(self) -> None:
        """Called when OK button is clicked. Saves the table contents to self.dlg_input.data and closes the dialog."""
        self.widgets_to_data()
        self._save_section_states()
        super().accept()

    def reject(self) -> None:
        """Called when the user clicks Cancel."""
        super().reject()
