"""Mapping utilities."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import json
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class Mapping:
    """Provides information needed to use the transient data file for map to coverage."""
    def __init__(self, mf_name='', node2=''):
        """Initializes the class.

        Args:
            mf_name (str): Full name of the MODFLOW variable. Only needed with array or grid-based variables.
            node2 (str): If attribute is at the nodes, the name of the other node att. This only
                        needs to be specified for the first node att.
        """
        self.mf_name = mf_name
        self.node2 = node2


def read_table_definition_file(att_table_filename: Path | str):
    """Reads the att table definition (.def) filename.

    Args:
        att_table_filename: Filepath of att table file. Def file is that plus .def

    Returns:
        (dict) The json from the file as a dict.
    """
    att_table_filename = str(att_table_filename)
    table_def_filename = att_table_filename + '.def'
    if os.path.isfile(table_def_filename):
        with open(table_def_filename, 'r') as file:
            table_def = json.loads(file.read())
    else:
        table_def = None
    return table_def


def fix_field_names(table_def: dict, fields_dict: dict) -> dict:
    """Makes field names 10 characters and standard MODFLOW names.

    Args:
        table_def (dict): Table definition dict.
        fields_dict (dict): Dict of bad to good field names.

    Returns:
        A new dict with the fixed names.
    """
    new_table_def = copy.deepcopy(table_def)
    for column in new_table_def['columns']:
        if column['name'] in fields_dict:
            column['name'] = fields_dict[column['name']]
    return new_table_def


def standard_fields():
    """Initializes the dict that contains bad field names and their good counterparts.

    Shapefile field names must be 10 characters max, and we try to match standard MODFLOW names. By convention,
    '_TS' means XY Series, '_TN' means TIN, '_RA' means raster.
    """
    fields = {
        'Auto assign layer': 'AUTO_LAYER',
        'Bot. elev.': 'BOT',
        'Bot. elev. Raster': 'BOT_RA',
        'Bot. elev. TIN': 'BOT_TN',
        'Bot. elev. TS': 'BOT_TS',
        'Bot. scr.': 'BOT_SCRN',
        'Cond.': 'COND',
        'Cond. Raster': 'COND_RA',
        'Cond. TIN': 'COND_TN',
        'Cond. TS': 'COND_TS',
        'ET elev.': 'SURFACE',
        'ET elev. TS': 'SURFACE_TS',
        'ET ext. depth': 'DEPTH',
        'ET ext. depth TS': 'DEPTH_TS',
        'ETSW TS': 'ETSW_TS',
        'Flow rate': 'Q',
        'Flow rate TS': 'Q_TS',
        'From layer': 'FROM_LAYER',
        'FLOW TS': 'FLOW_TS',
        'Grid Top elev.': 'TOP',
        'Grid Bot. elev.': 'BOTM',
        'Head-Stage': 'STAGE',
        'Head-Stage TS': 'STAGE_TS',
        'Head-Stage TIN': 'STAGE_TN',
        'Head-Stage Raster': 'STAGE_RA',
        'Horizontal K': 'HK',
        'Horizontal anis.': 'HANI',
        'Hyd. char.': 'HYDCHR',
        'Max ET rate': 'RATE',
        'Max ET rate TS': 'RATE_TS',
        'Node1 Bot. elev.': 'ND1_BOT',
        'Node2 Bot. elev.': 'ND2_BOT',
        'Node1 Bot. elev. Raster': 'ND1_BOT_RA',
        'Node2 Bot. elev. Raster': 'ND2_BOT_RA',
        'Node1 Bot. elev. TIN': 'ND1_BOT_TN',
        'Node2 Bot. elev. TIN': 'ND2_BOT_TN',
        'Node1 Bot. elev. TS': 'ND1_BOT_TS',
        'Node2 Bot. elev. TS': 'ND2_BOT_TS',
        'Node1 Head-Stage': 'ND1_HED',
        'Node2 Head-Stage': 'ND2_HED',
        'Node1 Head-Stage TS': 'ND1_HED_TS',
        'Node2 Head-Stage TS': 'ND2_HED_TS',
        'Node1 Head-Stage TIN': 'ND1_HED_TN',
        'Node2 Head-Stage TIN': 'ND2_HED_TN',
        'Node1 Head-Stage Raster': 'ND1_HED_RA',
        'Node2 Head-Stage Raster': 'ND2_HED_RA',
        'Partition well': 'PART_WEL',
        'PPTSW TS': 'PPTSW_TS',
        'RADIUS_SKIN': 'RADIUS_SKN',
        'Recharge rate': 'RECH',
        'Recharge rate TS': 'RECH_TS',
        'ROUGHCH TS': 'ROUGHCH_TS',
        'RUNOFF TS': 'RUNOFF_TS',
        'Spec. storage': 'SS',
        'Spec. yield': 'SY',
        'To layer': 'TO_LAYER',
        'Top scr.': 'TOP_SCRN',
        'Use screen': 'USE_SCRN',
        'Vertical K': 'VK',
        'Vertical anis.': 'VANI',
        'MD_TYPE_FLAG': 'MD_TYPE',
        'MD_TYPE_FLAG TS': 'MD_TYPE_TS',
        'MD_FRACTION': 'MD_FRAC',
        'MD_FRACTION TS': 'MD_FRAC_TS',
        'MD_POROSITY': 'MD_PORO',
        'MD_PORO TS': 'MD_PORO_TS',
        'BULK_DENSITY': 'BULK_DN',
        'BULK_DENSITY TS': 'BULK_DN_TS',
        'MD_DIFF_LENGTH': 'MD_DIFL',
        'MD_DIFF_LENGTH TS': 'MD_DIFL_TS',
        'MD_TORTUOSITY': 'MD_TORT',
        'MD_TORTUOSITY TS': 'MD_TORT_TS',
        'MD_DIST_COEFF': 'MD_DIST',
        'MD_DIST_COEFF TS': 'MD_DIST_TS',
        'MD_DECAY': 'MD_DECA',
        'MD_DECAY TS': 'MD_DECA_TS',
        'MD_DIFF_COEFF': 'MD_DIFC',
        'MD_DIFF_COEFF TS': 'MD_DIFC_TS',
        'AIOLD1MD': 'AIOLD1MD',
        'AIOLD1MD TS': 'AIOLD1M_TS',
        'AIOLD2MD': 'AIOLD2MD',
        'AIOLD2MD TS': 'AIOLD2M_TS',
    }
    return fields


def layer_info_from_row(do_layers, row):
    """Returns 'from layer' and 'to layer'.

    Args:
        row:

    Returns:
        (tuple): tuple containing:
            - (int): The 'From layer'.
            - (int): The 'To layer'.
    """
    if not do_layers:
        return 0, 0

    from_layer = int(row['From layer'])
    to_layer = int(row['To layer'])
    return from_layer, to_layer
