"""Module for ReaderBase class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['FileReader', 'ParseError']

# 1. Standard Python modules
from pathlib import Path
import re
from typing import Any, Optional, TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ParseError(Exception):
    """Exception raised when a file contains an error."""
    def __init__(self, line_number: int, field_number: int, problem: str):
        """
        Initialize the class.

        Args:
            line_number: Line number the error occurred on.
            field_number: If nonzero, the field where the error was encountered. Otherwise, omitted from error message.
            problem: Description of the problem.
        """
        if field_number:
            self.message = f'Error on line {line_number}, field {field_number}: {problem}'
        else:
            self.message = f'Error on line {line_number}: {problem}'
        super().__init__(line_number, field_number, problem)


class EndOfInputError(ParseError):
    pass


class FileReader:
    """A class for reading files."""
    def __init__(self, file: Path | str | TextIO, comment_markers: Optional[list[str]] = None):
        """
        Initialize the reader.

        Args:
            file: The file to be read.
            comment_markers: List of strings to consider as comment markers. When reading a line, the first comment
                marker and everything after it will be discarded.
        """
        comment_markers = comment_markers or []

        if isinstance(file, (Path, str)):
            self._path = file
            self._file = None
        else:
            self._path = None
            self._file = file

        self._comment_markers = comment_markers
        self._exponent_pattern = re.compile('[dD]')
        self._split_pattern = re.compile(r'[ ,\t\n]')
        self._strip_pattern = re.compile(r'[^ ,\t\n]')

        self._next_index_in_line = 0
        self._current_token_in_line = 0

        self.line_number = 0
        self._line = None

    @property
    def line(self) -> str:
        """The current line in the file, with comments stripped."""
        if self._line is None:
            self._next_line()
        return self._line

    def __enter__(self):
        """Open the file."""
        self.line_number = 0
        if self._path is not None:
            self._file = open(self._path, 'r')
        return self

    def __exit__(self, _exc_type, _exc_value, _exc_tb):
        """Close the file."""
        if self._path is not None:
            self._file.close()

    def _next_line(self):
        """Read a line out of the file."""
        self._current_token_in_line = 0  # Current token appears in error, so reset it for this line before reporting.

        line = self._file.readline()

        if not line and self._line == '':
            raise self.error('Unexpected end-of-file.', end_of_input=True)

        self.line_number += 1

        had_newline = line.endswith('\n')
        original_length = len(line)

        for comment_marker in self._comment_markers:
            line = line.split(comment_marker)[0]

        if len(line) != original_length:
            line = line.rstrip()
        if len(line) != original_length and had_newline:
            line += '\n'

        self._line = line
        self._next_index_in_line = 0

    def next_line(self):
        """Move the reader to the next line of input, discarding whatever is left on the current line."""
        if self.line_number == 0:
            self._next_line()  # The reader pretends to be on line 1 initially, but isn't, so we have to fix that now.
        self._next_line()

    def read_str(self, optional: bool = False) -> str:
        """
        Read a string from the file.

        Args:
            optional: Whether the field is optional. If True and the reader is already at the end of the line, then
                None will be returned. If False, then a string is required.

        Returns:
            The string that was read, or '' if `optional` was True and the reader was at the end of the line.
        """
        line = self.line
        self._current_token_in_line += 1
        match = self._strip_pattern.search(line, self._next_index_in_line)

        if not match and not optional:
            # We stripped everything.
            self._next_index_in_line = len(line)
            raise self.error('Unexpected end-of-line.', end_of_input=True)
        elif not match:
            return ''

        start = match.start()
        if line[start] == '"':
            # We're extracting a quoted string.
            end = line.find('"', start + 1) + 1
            self._next_index_in_line = end
            piece = line[start:end].strip('"')
            return piece

        match = self._split_pattern.search(line, start)
        if not match:
            # No more whitespace, this is the last piece.
            self._next_index_in_line = len(line)
            return line[start:]

        # There was whitespace, just take up to there.
        piece = line[start:match.start()]
        self._next_index_in_line = match.start() + 1
        return piece

    def _read_wrapper(self, type_constructor, optional: bool, message: str) -> Any:
        """Wrapper for handling optional fields."""
        try:
            field = self.read_str()
        except EndOfInputError:
            if optional:
                return None
            else:
                raise self.error(message)

        try:
            return type_constructor(field)
        except ValueError:
            raise self.error(message)

    def _multi_read_wrapper(self, type_constructor, optional: bool, count: int, message: str):
        """Wrapper for reading multiple fields of the same type in a single call."""
        values = [self._read_wrapper(type_constructor, optional, message) for _ in range(count)]
        if count == 1:
            return values[0]
        else:
            return values

    def read_int(self, optional: bool = False, count: int = 1) -> Optional[int | list[int]]:
        """
        Read one or more ints from the file.

        Args:
            count: How many ints to read.
            optional: Whether the field is optional. If True and the reader is already at the end of the line, then
                None will be returned. If False, then an integer is required. In both cases, if anything is present, it
                must be an int or an exception is raised.

        Returns:
            If `count` is 1, then the single read int. If `count` is >1, then a list of `count` ints. If `optional` is
            True and the reader is at the end of the line, then None.
        """
        return self._multi_read_wrapper(int, optional, count, 'Expected an integer.')

    def read_float(self, count: int = 1) -> float | list[float]:
        """
        Read one or more floats from the file.

        Args:
            count: How many floats to read.

        Returns:
            If `count` is 1, then the single read float. If `count` is >1, then a list of `count` floats.
        """
        return self._multi_read_wrapper(float, False, count, 'Expected a double.')

    def read_remainder(self) -> str:
        """
        Read the remainder of a line.

        There must actually be a remainder of the line or an exception will be raised.
        """
        line = self.line
        self._current_token_in_line += 1
        match = self._strip_pattern.search(line, self._next_index_in_line)

        if not match:
            # We stripped everything.
            self._next_index_in_line = len(line)
            raise self.error('Unexpected end-of-line.')

        remainder = line[match.start():].strip('"\n')
        return remainder

    def error(self, reason: str, end_of_input: bool = False) -> ParseError:
        """
        Get an exception that reports an error.

        Args:
            reason: The reason for the error, e.g. "The thing was an invalid value".
            end_of_input: Whether to raise an EndOfInputError, as opposed to ParseError.

        Returns:
            An exception that can be raised to report an error.
        """
        line_number = self.line_number
        token_number = self._current_token_in_line
        if end_of_input:
            return EndOfInputError(line_number, token_number, reason)
        else:
            return ParseError(line_number, token_number, reason)
