"""Module defining the contents of a .traps file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['Direction', 'LineTrap', 'PolygonTrap', 'Traps']

# 1. Standard Python modules
from dataclasses import dataclass, field
from enum import auto, Enum
from typing import TypeAlias

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

Pt2d: TypeAlias = tuple[float, float]


class Direction(Enum):
    """
    Trap directions.

    It's unclear from the documentation what these actually mean.
    """
    decreasing_x = auto()
    increasing_x = auto()
    any = auto()


@dataclass
class LineTrap:
    """
    A single line trap.

    Args:
        trap_id: The trap's ID.
        label: The trap's human-readable label.
        direction: The trap's direction.
        bottom: The trap's bottom.
        top: The trap's top.
        is_open: Whether the trap is open. If open, parcels that enter the trap can leave it. If closed, parcels that
            enter the trap can't leave.
        is_single: Whether the trap is single. If single, parcels that enter the trap are only counted once per
            simulation. Otherwise, parcels are counted every time they enter.
        start: The trap's start point. In XMS, this is the x,y coordinates of the arc's start node. PTM doesn't support
            traps with vertices.
        end: The trap's end point. In XMS, this is the x,y coordinates of the arc's end node. PTM doesn't support traps
            with vertices.
    """
    trap_id: int = -1
    label: str = ''
    direction: Direction = Direction.any
    bottom: float = 0.0
    top: float = 0.0
    is_open: bool = False
    is_single: bool = False
    start: Pt2d = (0.0, 0.0)
    end: Pt2d = (0.0, 0.0)


@dataclass
class PolygonTrap:
    """
    A single polygon trap.

    Args:
        trap_id: The trap's ID.
        label: The trap's human-readable label.
        bottom: The trap's bottom.
        top: The trap's top.
        is_open: Whether the trap is open. If open, parcels that enter the trap can leave it. If closed, parcels that
            enter the trap can't leave.
        is_single: Whether the trap is single. If single, parcels that enter the trap are only counted once per
            simulation. Otherwise, parcels are counted every time they enter.
        points: List of x,y coordinate pairs defining the trap's boundary. In XMS, this would be the points defining
            the polygon's outer ring. PTM doesn't support polygon traps with holes.
    """
    trap_id: int = -1
    label: str = ''
    bottom: float = 0.0
    top: float = 0.0
    is_open: bool = False
    is_single: bool = False
    points: list[Pt2d] = field(default_factory=list)


@dataclass
class Traps:
    """
    A collection of line and polygon traps.

    Args:
        line_traps: List of line traps. May be empty.
        polygon_traps: List of polygon traps. May be empty.
    """
    line_traps: list[LineTrap] = field(default_factory=list)
    polygon_traps: list[PolygonTrap] = field(default_factory=list)
