"""Class for writing canal coverage data to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation

# 4. Local modules
from xms.rsm.data import cell_monitor_data_def as cmdd
from xms.rsm.file_io.monitor_writer import MonitorWriter


class CellMonitorWriter(MonitorWriter):
    """Writer class for the RSM control file."""
    def __init__(self, xml_parent, xms_data):
        """Constructor.

        Args:
            xml_parent (xml.etree.cElementTree.Element): xml parent element
            xms_data (XmsData): The XMS interprocess communication object
        """
        super().__init__(xml_parent, xms_data.cell_monitor_coverages)
        self._monitor_xml_name = 'cellmonitor'
        self._xy_to_cell = None
        self._xms_data = xms_data
        self._ugrid = xms_data.xmugrid

    def _build_xy_to_cell(self):
        """Create a lookup for x, y coordinates to cell ids."""
        self._xy_to_cell = {}
        xy = []
        for cov, _ in self._coverages:
            points = cov.get_points(FilterLocation.PT_LOC_DISJOINT)
            xy.extend([(pt.x, pt.y) for pt in points])

        extractor = self._xms_data.ugrid_extractor
        if extractor:
            extractor.extract_locations = [(p[0], p[1], 0.0) for p in xy]
            extractor.extract_data()
            cell_idx = extractor.cell_indexes
        else:
            cell_idx = [-1] * len(xy)
        cell_idx = [self._invalid_id - 1 if x < 0 else x for x in cell_idx]
        self._xy_to_cell = {p: idx + 1 for p, idx in zip(xy, cell_idx)}

    def _get_monitor_id(self, x: float, y: float) -> int:
        """Get a Section with parameters for a given target."""
        if self._xy_to_cell is None:
            self._build_xy_to_cell()
        return self._xy_to_cell[(x, y)]

    def _generic_model(self):
        """Get the generic model."""
        return cmdd.generic_model()

    def _log_invalid_monitor_id(self, pt_id, cov_name):
        """Log an invalid monitor id.

        Args:
            pt_id (int): point id
            cov_name (str): coverage name
        """
        msg = f'Point id: {pt_id} in coverage: "{cov_name}" is outside the grid and will not be written to the file.'
        self._logger.warning(msg)
