"""Model parameters."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"
__all__ = [
    'FLAG_IC_NAMES',
    'INU_TR_NAMES',
    'SPECIAL_FLAG_IC_NAMES',
    'elevation_flag',
    'enabled',
    'get_model',
    'needed_files',
    'parameter_for_file',
    'salinity_flag',
    'temperature_flag',
    'velocity_flag',
]

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel, Parameter, Section
from xms.tool_core.table_definition import FloatColumnType, IntColumnType, StringColumnType, TableDefinition

# 4. Local modules

WSE_GROUP = 'wse'
FLOW_GROUP = 'flow'
SALINITY_GROUP = 'salinity'
TEMPERATURE_GROUP = 'temperature'

INU_TR_NAMES: list[str] = [f'inu_tr({i})' for i in range(1, 13)]
FLAG_IC_NAMES: list[str] = [f'flag_ic({i})' for i in range(1, 13)]
SPECIAL_FLAG_IC_NAMES: list[str] = ['flag_ic(1)', 'flag_ic(2)']
MOD_NAMES: list[str] = ['T', 'S', 'GEN', 'Age', 'SED3D', 'EcoSim', 'ICM', 'CoSINE', 'FIB', 'TIMOR', 'FABM', 'DVD']


def get_model() -> GenericModel:
    """Get the model used by SCHISM."""
    model = GenericModel(exclusive_material_conditions=True)
    _add_boundary_conditions(model)
    _add_core_group(model)
    _add_opt_group(model)
    _add_schout_group(model)
    _add_other_group(model)
    _add_materials(model)
    return model


def _add_boundary_conditions(model: GenericModel):
    """Add the arc boundary conditions."""
    options = ['0', '1', '2', '3', '4', '5']
    flags = {option: False for option in options}

    group = model.arc_parameters.add_group('open', 'Open', is_default=True)

    # Water surface elevation
    selector = group.add_option('wse-type', '\u03B7', '0', options)
    series = group.add_curve('wse-series', 'Time series', ['Elevation', 'Time (seconds)'])
    series.add_dependency(selector, flags | {'1': True})
    constant = group.add_float('wse-constant', 'Constant', 0.0)
    constant.add_dependency(selector, flags | {'2': True})
    dataset = group.add_text('wse-dataset', 'Dataset')
    dataset.add_dependency(selector, flags | {'4': True, '5': True})

    # Flow type
    selector = group.add_option('flow-type', 'Flow (u,v)', '0', options)
    series = group.add_curve('flow-series', 'Time series', ['Elevation', 'Time (seconds)'])
    series.add_dependency(selector, flags | {'1': True})
    constant = group.add_float('flow-constant', 'Constant', 0.0)
    constant.add_dependency(selector, flags | {'2': True})
    dataset = group.add_text('flow-dataset', 'Dataset')
    dataset.add_dependency(selector, flags | {'4': True, '5': True})

    options = ['0', '1', '2', '3', '4']
    flags = {option: False for option in options}

    selector = group.add_option('salinity-type', 'Salinity', '0', options)
    constant = group.add_float('salinity-constant', 'Constant', 0.0)
    constant.add_dependency(selector, flags | {'2': True})
    nudging = group.add_float('salinity-nudging', 'Nudging factor', 0.0)
    nudging.add_dependency(selector, flags | {'1': True, '2': True, '3': True, '4': True})

    selector = group.add_option('temperature-type', 'Temperature', '0', options)
    constant = group.add_float('temperature-constant', 'Constant', 0.0)
    constant.add_dependency(selector, flags | {'2': True})
    nudging = group.add_float('temperature-nudging', 'Nudging factor', 0.0)
    nudging.add_dependency(selector, flags | {'1': True, '2': True, '3': True, '4': True})

    group.add_text(
        'name', 'Name (optional)', description='Added as a comment in some output files for debugging purposes.'
    )


def _add_core_group(model: GenericModel):
    """Add the core group to the global parameters."""
    group = model.global_parameters.add_group('core', 'CORE')

    group.add_boolean(
        'ipre',
        'Pre-processing flag (ipre)',
        False,
        description='Pre-processing flag is very useful for checking integrity of the horizontal '
        'grid and some inputs. ipre=1: code will output centers.bp, sidecenters.bp, '
        '(centers build point, sidcenters build point), and mirror.out and stop. '
        'Check errors in fatal.error or system outputs.'
    )

    group.add_boolean(
        'ibc',
        'Barotropic flag (ibc)',
        False,
        description='If ibc=0, a baroclinic model is used and regardless of the value for ibtp, '
        'the transport equation is solved. If ibc=1, a barotropic model is used, '
        'and the transport equation may (when ibtp=1) or may not (when ibtp=0) be '
        'solved; in the former case, S and T are treated as passive tracers.'
    )
    group.add_boolean(
        'ibtp',
        'Baroclinic flag (ibtp)',
        True,
        description='If ibc=0, a baroclinic model is used and regardless of the value '
        'for ibtp, the transport equation is solved. If ibc=1, a barotropic '
        'model is used, and the transport equation may (when ibtp=1) or may '
        'not (when ibtp=0) be solved; in the former case, S and T are treated '
        'as passive tracers.'
    )

    group.add_float('rnday', 'Total simulation time in days (rnday)', 30.0)
    group.add_float(
        'dt',
        'Time step in seconds (dt)',
        100.0,
        description='This is the main time step in SCHISM. The transport solvers have their own '
        'adaptive time step for subcycling to satisfy the stability constraint.'
    )

    # TODO: WWM dependency
    group.add_integer(
        'msc2',
        'msc2',
        24,
        description='These two parameters are only used if the wave module WWM is invoked '
        '(USE_WWM is on and icou_elfe_wwm=1). The values represent the spectral '
        'resolution used in WWM and must match those in wwminput.nml;'
    )
    group.add_integer(
        'mdc2',
        'mdc2',
        30,
        description='These two parameters are only used if the wave module WWM is invoked '
        '(USE_WWM is on and icou_elfe_wwm=1). The values represent the spectral '
        'resolution used in WWM and must match those in wwminput.nml;'
    )

    # TODO: USE_GEN, USE_AGE, USE_SED, USE_ECO dependencies
    group.add_integer('ntracer_gen', '# of tracers for GEN (ntracer_gen)', 2)
    group.add_integer(
        'ntracer_age', '# of tracers for AGE (ntracer_age)', 4, description='Must be =2*N where N is # of age tracers'
    )
    group.add_integer('sed_class', '# of tracers for SED3D (sed_class)', 5)
    group.add_integer('eco_class', '# of tracers for EcoSim (eco_class)', 27, low=25, high=60)

    group.add_integer(
        'nspool', 'Global output step spool (nspool)', 36, description='Output is done every nspool steps.'
    )
    group.add_integer(
        'ihfskip',
        'Global stack spool (ihfskip)',
        864,
        description='A new output stack is opened every ihfskip steps. Must be a multiple of nspool'
    )


def _add_opt_group(model: GenericModel):
    """Add the opt group to the global parameters."""
    group = model.global_parameters.add_group('opt', 'OPT')
    tracer_modules = ['T', 'S', 'GEN', 'AGE', 'SED3D', 'EcoSim', 'ICM', 'CoSINE', 'Feco', 'TIMOR', 'FABM', 'DVD']

    # The order is following the sample param.nml. Online documentation is in alphabetical order.
    # Some parameters are "basic" and always visible. Others are "advanced" and hidden behind a toggle.
    # Some pseud-basic parameters were moved closer to the top to be above the advanced toggle.
    group.add_integer('start_year', 'Start Year (start_year)', 2000)
    group.add_integer('start_month', 'Start Month (start_month)', 1)
    group.add_integer('start_day', 'Start Day (start_day)', 1)
    group.add_float('start_hour', 'Start Hour (start_hour)', 0.0)
    group.add_float(
        'utc_start',
        'UTC Start (utc_start)',
        8.0,
        description='utc_start is hours behind the GMT, and is used to adjust time zone.'
    )
    # Needs to be 1 or 2, but maybe this should be a boolean?
    ics_options = ['1', '2']
    ics = group.add_option('ics', 'Coordinate frame flag (ics)', '1', options=ics_options)
    slam0 = group.add_float(
        'slam0',
        'Centers of projection used to convert lon to Cartesian coordinates (slam0)',
        -124.0,
        description='Not really used'
    )
    _add_dependency(ics, slam0, {1})
    sfea0 = group.add_float('sfea0', 'Centers of projection used to convert lat to Cartesian coordinates (sfea0)', 45.0)
    _add_dependency(ics, sfea0, {1})

    options = ['0', '1', '2']
    ihot = group.add_option('ihot', 'Hotstart option (ihot)', '0', options=options)
    ihot_file = group.add_input_file(
        name='ihot_file', label='Hotstart file', default='', file_filter='NetCDF hotstart (*.nc);;All files (*.*)'
    )
    _add_dependency(ihot, ihot_file, {1, 2})

    group.add_float('dramp', 'Ramp-up period in days for B.C. (dramp)', 1.0)

    options = ['-1', '0', '1']
    nchi = group.add_option('nchi', 'Bottom friction option (nchi)', '0', options=options)
    dzb_min = group.add_float('dzb_min', 'Min. bottom boundary layer thickness [m] (dzb_min)', 0.5)
    _add_dependency(nchi, dzb_min, {1})
    dzb_decay = group.add_float('dzb_decay', 'A decay const. [-]. should =0 (dzb_decay)', 0.0)
    _add_dependency(nchi, dzb_decay, {1})
    hmin_man = group.add_float('hmin_man', 'Min. depth in Manning\'s formulation [m] (hmin_man)', 1.0)
    _add_dependency(nchi, hmin_man, {-1})
    manning_gr3 = group.add_dataset('manning_ds', "Manning's n dataset")
    _add_dependency(nchi, manning_gr3, {-1})
    drag_gr3 = group.add_dataset('drag_ds', "Drag coefficient dataset")
    _add_dependency(nchi, drag_gr3, {0})
    rough_gr3 = group.add_dataset('rough_ds', 'Bottom roughness dataset')
    _add_dependency(nchi, rough_gr3, {1})

    options = ['-1', '0', '1']
    ncor = group.add_option('ncor', 'Coriolis option (ncor)', '0', options=options)
    rlatitude = group.add_float('rlatitude', 'Mean latitude used to calculate the Coriolis factor (rlatitude)', 46.0)
    _add_dependency(ncor, rlatitude, {-1})
    coricoef = group.add_float('coricoef', 'Coriolis parameter (coricoef)', 0.0)
    coricoef.add_dependency(ncor, {'-1': False, '0': True, '1': False})

    options = ['-1', '0', '2', '4']
    nws = group.add_option('nws', 'Atmos. option (nws)', '0', options=options)
    windrot = group.add_dataset('windrot_geo2proj_ds', 'windrot_geo2proj')
    _add_dependency(nws, windrot, {-1, 2, 4})
    wtiminc = group.add_float('wtiminc', 'Time step for atmos. forcing (wtiminc)', 300.0)
    _add_dependency(nws, wtiminc, {1, 2, 3})
    nrampwind = group.add_float('nrampwind', 'Ramp-up option for atmps. forcing (nrampwind)', 1.0)
    _add_dependency(nws, nrampwind, {1, 2, 3})
    drampwind_flags = {option: option[0] in {'1', '2', '3'} for option in nws.options}
    drampwind = group.add_float('drampwind', 'Ramp-up period in days for wind (drampwind)', 1.0)
    drampwind.add_dependency(nws, drampwind_flags)
    iwindoff = group.add_boolean('iwindoff', 'Option to scale the wind speed (iwindoff)', False)
    _add_dependency(nws, iwindoff, {-1, 1, 2, 3})
    iwind_form = group.add_integer('iwind_form', '(iwind_form)', 1, low=-3, high=0)
    _add_dependency(nws, iwind_form, {-1, 1, 2, 3})
    options = ['1', '10']
    model_type_pahm = group.add_option(
        'model_type_pahm', 'Hurricane model type (model_type_pahm)', '10', options=options
    )
    _add_dependency(nws, model_type_pahm, {-1})

    ihconsv = group.add_boolean('ihconsv', 'Heat budget model option (ihconsv)', False)
    albedo_dataset = group.add_dataset('albedo_ds', 'Albedo dataset')
    albedo_dataset.add_dependency(ihconsv, {True: True, False: False})
    watertype_dataset = group.add_dataset('watertype_ds', 'Water type dataset')
    watertype_dataset.add_dependency(ihconsv, {True: True, False: False})

    options = ['0', '1', '2']
    i_hmin_airsea_ex = group.add_option(
        'i_hmin_airsea_ex', 'Locally turn off heat exchange (i_hmin_airsea_ex)', '2', options=options
    )
    i_hmin_airsea_ex.add_dependency(ihconsv, {True: True, False: False})
    hmin_airsea_ex = group.add_float('hmin_airsea_ex', 'hmin_airsea_ex', 0.2)
    hmin_airsea_ex.add_dependency(ihconsv, {False: False, True: True})

    i_hmin_salt_ex = group.add_option(
        'i_hmin_salt_ex', 'Locally turn off salt exchange (i_hmin_salt_ex)', '2', options=options
    )
    i_hmin_salt_ex.add_dependency(ihconsv, {True: True, False: False})

    hmin_salt_ex = group.add_float('hmin_salt_ex', 'hmin_salt_ex', 0.2)
    hmin_salt_ex.add_dependency(ihconsv, {False: False, True: True})

    group.add_boolean(
        'isconsv',
        'Evaporation and precipitation model (isconsv)',
        False,
        description='PREC_EVAP pre-processing flag in makefile needs to be turned on'
    )

    itur_options = ['-2', '-1', '0', '1', '2', '3', '4', '5']
    itur = group.add_option('itur', 'Turbulence closure (itur)', '0', options=itur_options)

    flags = {option: option in ['3', '4'] for option in itur_options}
    diffmax = group.add_dataset('diffmax_ds', 'diffmax')
    diffmax.add_dependency(itur, flags)
    diffmin = group.add_dataset('diffmin_ds', 'diffmin')
    diffmin.add_dependency(itur, flags)

    flags = {option: option == '0' for option in itur.options}
    dfv0 = group.add_float('dfv0', 'dfv0', 0.01)
    dfv0.add_dependency(itur, flags)
    dfh0 = group.add_float('dfh0', 'dfh0', 0.0001)
    dfh0.add_dependency(itur, flags)
    options = ['MY', 'KL', 'KE', 'KW', 'UB']
    mid = group.add_option('mid', 'Closure scheme used (mid)', 'KL', options=options, description='Use KE if itur=5')
    _add_dependency(itur, mid, {3, 5})
    options = ['GA', 'KC']
    stab = group.add_option(
        'stab',
        'Stability function used (stab)',
        'KC',
        options=options,
        description="Use 'GA' if turb_met='MY'; otherwise use 'KC'"
    )
    _add_dependency(itur, stab, {3, 5})

    xlsc0 = group.add_float('xlsc0', 'Scale for surface & bottom mixing length (xlsc0)', 0.1, low=0)
    _add_dependency(itur, xlsc0, {3, 5})
    h1_pp = group.add_float('h1_pp', 'h1_pp', 20.0)
    _add_dependency(itur, h1_pp, {2})
    vdmax_pp1 = group.add_float('vdmax_pp1', 'vdmax_pp1', 1e-2)
    _add_dependency(itur, vdmax_pp1, {2})
    vdmin_pp1 = group.add_float('vdmin_pp1', 'vdmin_pp1', 1e-5)
    _add_dependency(itur, vdmin_pp1, {2})
    tdmin_pp1 = group.add_float('tdmin_pp1', 'tdmin_pp1', 1e-5)
    _add_dependency(itur, tdmin_pp1, {2})
    h2_pp = group.add_float('h2_pp', 'h2_pp', 50.0)
    _add_dependency(itur, h2_pp, {2})
    vdmax_pp2 = group.add_float('vdmax_pp2', 'vdmax_pp2', 1e-2)
    _add_dependency(itur, vdmax_pp2, {2})
    vdmin_pp2 = group.add_float('vdmin_pp2', 'vdmin_pp2', 1e-5)
    _add_dependency(itur, vdmin_pp2, {2})
    tdmin_pp2 = group.add_float('tdmin_pp2', 'tdmin_pp2', 0.0)
    _add_dependency(itur, tdmin_pp2, {2})

    advanced = group.add_boolean('advanced', 'Advanced', False)

    group.add_boolean(
        'ipre2',
        '2nd Pre-processing flag (ipre2)',
        False,
        description='If ipre2/=0, code will output some diagnotic outputs and stop.'
    )

    # TODO: itransport_only=2 needs totalSuspendedLoad and sedBedStress
    # From OPT
    options = ['0', '1', '2']
    group.add_option('itransport_only', 'Only solve tracer transport (intransport_only)', '0', options=options)

    # If iloadtide=1, needs inputs: loadtide_[FREQ].gr3,
    # ! where [FREQ] are freq names (shared with tidal potential, in upper cases)
    # ! and the _two_ 'depths' inside are amplitude (m) and phases (degrees behind GMT),
    # ! interpolated from global tide model (e.g. FES2014).
    options = ['0', '1', '2', '3']
    group.add_option('iloadtide', 'Add SAL into tidal potential (iloadtide)', '0', options=options)
    # if 3, default should be 0.12
    group.add_float('loadtide_coef', 'Loadtide coefficient (loadtide_coef)', 0.1)

    options = ['0', '1']
    ieos_type = group.add_option('ieos_type', 'Equation of State type used (ieos_type)', '0', options=options)
    ieos_pres = group.add_boolean('ieos_pres', 'Hydrostatic pressure effects (ieos_pres)', False)
    _add_dependency(ieos_type, ieos_pres, {0})
    eos_a = group.add_float('eos_a', 'eos_a', -0.1, high=0)
    _add_dependency(ieos_type, eos_a, {1})
    eos_b = group.add_float('eos_b', 'eos_b', 1001.0)
    _add_dependency(ieos_type, eos_b, {1})

    group.add_boolean('nramp', 'Ramp-up flag (nramp)', True)
    group.add_boolean('nrampbc', 'Ramp-up flag (nrampbc)', False)
    group.add_float('drampbc', 'Ramp-up period in days for baroclinic force (drampbc)', 0.0)

    options = ['0', '1']
    group.add_option('iupwind_mom', 'Method for momentum advection (iupwind_mom)', '0', options=options)

    # This until end of shapiro0: control the numerical dissipation in momentum solver. Maybe have in another group?
    options = ['0', '1']
    group.add_option('indvel', 'Method for computing velocity at nodes (indvel)', '0', options=options)

    options = ['0', '1', '2']
    # 1st Stabilization method, Horizontal viscosity option
    ihorcon = group.add_option('ihorcon', 'Horizontal viscosity option (ihorcon)', '0', options=options)
    hvis_coef0 = group.add_float(
        'hvis_coef0',
        'Horizontal viscosity (hvis_coef0)',
        0.025,
        description='Const. diffusion # if ihorcon/=0; <=0.025 for ihorcon=2, <=0.125 '
        'for ihorcon=1'
    )
    _add_dependency(ihorcon, hvis_coef0, {1, 2})
    flags = {'0': False, '1': True, '2': True}
    cdh = group.add_float(
        'cdh',
        'Land friction coefficent (cdh) (not active yet)',
        0.01,
        description='Needed only if ihorcon/=0. Not active yet',  # parent_name='ihorcon',
        # dependency_flags=flags
    )
    cdh.add_dependency(ihorcon, flags)

    # 2nd Stabilization method, Shapiro filter. This should normally be used if indvel=0
    # ishapiro=-1 (requiring shapiro.gr3)
    options = ['-1', '0', '1', '2']
    ishapiro = group.add_option('ishapiro', 'Shapiro filter (ishapiro)', '1', options=options)
    niter_shap = group.add_integer('niter_shap', 'Number of iterations with Shapiro filter (niter_shap)', 1)
    _add_dependency(ishapiro, niter_shap, {-1, 1, 2})
    shapiro0 = group.add_float('shapiro0', 'Shapiro filter strength (shapiro0)', 0.5)
    _add_dependency(ishapiro, shapiro0, {0})

    group.add_float(
        'thetai',
        'Implicitness factor (thetai)',
        0.6,
        low=0.5,
        high=1,
        description='Recommended value: 0.6. Use "1" to get maximum stability for strong wet/dry.'
    )

    # TODO: USE_WWM pre-processor enabled dependency
    group.add_boolean('icou_elfe_wwm', 'WWM coupler flag (icou_elfe_wwm)', False)
    group.add_integer('nstep_wwm', 'Time steps to call WWM (nstep_wwm)', 1)
    options = ['0', '1', '2']
    group.add_option('iwbl', 'Wave boundary layer formation (iwbl)', '0', options=options)
    group.add_float(
        'hmin_radstress', 'Min. total water depth used only in radiation stress calculation [m] (hmin_radstress)', 1.0
    )
    group.add_boolean('nrampwafo', 'Ramp-up option for wave forces (nrampwafo)', False)
    group.add_float(
        'drampwafo', 'Ramp-up period in days for wave forces (drampwafo)', 0.0, description='No ramp-up if <= 0'
    )
    group.add_float('turbinj', '% of depth-induced wave breaking energy injected in '
                    'turbulence (turbinj)', 0.15)
    group.add_float(
        'turbinjds', '% of wave energy dissipated through whitecapping injected in turbulence (turbinjds)', 0.0
    )
    group.add_float(
        'alphaw',
        'alphaw',
        1.0,
        description='for itur=4 : scaling parameter for the surface roughness z0s = alphaw*Hm0. '
        'If negative z0s = abs(alphaw) e.g. z0s=0.2 m (Feddersen and Trowbridge, 2005)'
    )
    group.add_float('fwvor_advxy_stokes', 'Stokes drift advection (xy), Coriolis (fwvor_advxy_stokes)', 1.0)
    group.add_float('fwvor_advz_stokes', 'Stokes drift advection (z) , Coriolis (fwvor_advz_stokes)', 1.0)
    group.add_float('fwvor_gradpress', 'Pressure term (fwvor_gradpress)', 1.0)
    group.add_float('fwvor_breaking', 'Wave breaking (fwvor_breaking)', 1.0)
    group.add_float('fwvor_streaming', 'Wave streaming (fwvor_streaming)', 1.0, description='Works with iwbl /= 0')
    group.add_float('fwvor_wveg', 'Wave dissipation by vegetation acceleration term (fwvor_wveg)', 0.0)
    group.add_float('fwvor_wveg_nl', 'Non linear intrawave vegetation force (fwvor_wveg_NL)', 0.0)
    group.add_option('cur_wwm', 'Coupling current in WWM (cur_wwm)', options[0], options=options)
    group.add_boolean(
        'wafo_obcramp',
        'Ramp on wave forces at open boundary (wafo_obcramp)',
        False,
        description='This option requires the input file wafo_ramp.gr3 which defines the ramp '
        'value (between 0 and 1) at nodes over the whole domain'
    )

    options = ['0', '1', '2']
    # TODO: If imm=1, bdef.gr3 is needed
    imm = group.add_option('imm', 'Bed deformation option (imm)', '0', options=options)
    ibdef = group.add_integer('ibdef', '# of steps used in deformation (ibdef)', 10)
    _add_dependency(imm, ibdef, {1})

    group.add_boolean(
        'iunder_deep', 'Option to deal with under resolution near steep slopes in deeper depths (iunder_deep)', False
    )

    group.add_float(
        'h1_bcc',
        'How the baroclinic gradient is calculated below bottom [m] (h1_bcc)',
        50.0,
        description='The "below-bottom" gradient is zeroed out if h>=h2_bcc (i.e. like Z) or '
        'uses constant extrapolation (i.e. like terrain-following) if '
        'h<=h1_bcc(<h2_bcc). A linear transition is used if the local depth '
        'h1_bcc<h<h2_bcc.'
    )
    group.add_float(
        'h2_bcc',
        'How the baroclinic gradient is calculated below bottom [m] (h2_bcc)',
        100.0,
        description='The "below-bottom" gradient is zeroed out if h>=h2_bcc (i.e. like Z) '
        'or uses constant extrapolation (i.e. like terrain-following) if '
        'h<=h1_bcc(<h2_bcc). A linear transition is used if the local depth '
        'h1_bcc<h<h2_bcc.'
    )

    flags = {False: False, True: True}
    group.add_float('hw_depth', 'Hannah-Wright depth (hw_depth)', 1.e6)
    group.add_float('hw_ratio', 'Hannah-Wright ratio (hw_ratio)', 0.5)

    # TODO: ihydraulics/=0, hydraulics.in ! is required
    group.add_boolean('ihydraulics', 'Hydraulic model option (ihydraulics)', False)

    # TODO: If =1, needs source_sink.in (list of elements),
    # ! vsource,th, vsink.th, and msource.th (the source/sink values must be single precision).
    # ! If =-1, all info is in source.nc
    # ! and each type of volume/mass source/sink can have its own time step and
    # ! # of records.
    options = ['-1', '0', '1']
    if_source = group.add_option('if_source', 'Point sources/sinks option (if_source)', '0', options=options)
    nramp_ss = group.add_boolean('nramp_ss', 'Ramp-up flag for source/sinks (nramp_ss)', True)
    _add_dependency(if_source, nramp_ss, {1})
    # flags = self._flags_from_options(options, [0, 2])
    dramp_ss = group.add_float(
        'dramp_ss', 'Ramp-up period in days for source/sinks(dramp_ss)', 2.0, description='No ramp-up if <= 0'
    )
    dramp_ss.add_dependency(if_source, {'-1': True, '0': False, '1': True})

    options = ['0', '1']
    group.add_option('meth_sink', '(meth_sink)', options[1], options=options)

    # TODO: possibly only show the models that are invoked
    definition = [
        StringColumnType(header='Module', enabled=False),
        IntColumnType(
            header='lev_tr_source',
            tool_tip='The tracers are injected into an element at a particular level, as specified by '
            'lev_tr_source(1:ntr) (where ntr is total # of tracer modules, i.e. 1 input level per '
            'module). The code will extrapolate below bottom/above surface if necessary, so e.g., -9 '
            'means bottom. To inject at all levels, set the level at 0.'
        )
    ]
    table_def = TableDefinition(definition, fixed_row_count=0)
    default = [[module, -9] for module in []]
    group.add_table(
        'lev_tr_source',
        'Vertical level to inject source concentration (lev_tr_source)',
        default=default,
        table_definition=table_def
    )

    group.add_integer('level_age', 'Specify level #\'s if age module is invoked (level_age)', -999)

    # TODO: dependency for hdif.gr3
    group.add_boolean(
        'ihdif',
        'Horizontal diffusivity option (ihdif)',
        False,
        description='If ihdif=1, horizontal diffusivity is given in hdif.gr3'
    )

    # TODO: If ic_elev=1, elev.ic (in *.gr3 format) is needed
    # ! to specify the initial elevations; otherwise elevation is initialized to 0 everywhere

    group.add_boolean('ic_elev', 'Elevation initial condition flag for cold start only (ic_elev)', False)

    group.add_boolean(
        'nramp_elev',
        'Elevation boundary condition ramp-up flag (nramp_elev)',
        False,
        description='If unchecked, ramp up from 0. If checked, ramp up from elev. values '
        'read in from elev.ic or hotstart.nc - if neither is present, from 0.'
    )

    group.add_boolean(
        'inv_atm_bnd',
        'Optional inverse barometric effects on the elev. (inv_atm_bnd)',
        False,
        description='If inv_atm_bnd=1, the elev.\'s at boundary are corrected by the difference '
        'between the actual atmos. pressure and a reference pressure'
    )
    group.add_float('prmsl_ref', 'Reference atmos. pressure on bnd [Pa] (prmsl_ref)', 101325.0)

    flag_ic_flags = {'0': False, '1': True, '2': False}
    flag_ic_options = ['0', '1', '2']
    for name in FLAG_IC_NAMES:
        selector = group.add_option(name, name, '1', flag_ic_options)
        dataset = group.add_dataset(f'{name}_dataset', f'{name} dataset')
        dataset.add_dependency(selector, flag_ic_flags)

    group.add_float('gen_wsett', 'Settling vel [m/s] for GEN module (positive downward) (gen_wsett)', 0.0)

    # TODO: T,S group?
    group.add_boolean(
        'ibcc_mean',
        'Mean T,S profile option (ibcc_mean)',
        False,
        description='If ibcc_mean is checked (or ihot=0 and flag_ic(1)=2), mean profile is read '
        'in from ts.ic, and will be removed when calculating baroclinic force. '
        'No ts.ic is needed if ibcc_mean is unchecked.'
    )

    group.add_float('rmaxvel', 'Maximum velocity (rmaxvel)', 5.0)

    # Following parameters control backtracking
    group.add_float(
        'velmin_btrack', 'Min. velocity for invoking backtrack and for abnormal exit in quicksearch (velmin_btrack)',
        0.0001
    )
    group.add_float('btrack_nudge', 'Nudging factors for starting side/node (btrack_nudge)', 0.009013)
    options = ['0', '1']
    group.add_option(
        'ibtrack_openbnd', 'Behavior when trajectory hits open bnd (ibtrack_openbnd)', options[1], options=options
    )

    # For Wetting and drying
    group.add_boolean(
        'ihhat', r'\hat{H} is made non-negative to enhance robustness near wetting and drying (ihhat)', True
    )
    group.add_boolean(
        'inunfl', 'More accurate wetting and drying if grid resolution is sufficiently fine (inunfl)', False
    )
    group.add_float('h0', 'Min. water depth for wetting/drying [m] (h0)', 0.01)
    # Matters only if USE_WWM
    group.add_boolean(
        'shorewafo',
        'Impose radiation stress R_s = g*grad(eta) at the numerical shoreline (shorewafo)',
        False,
        description='Matters only if USE_WWM'
    )

    # Solver options
    group.add_integer('moitn0', 'Output spool for solver info; used only with JCG (moitn0)', 50)
    group.add_integer('mxitn0', 'Max. solver iteration allowed (mxitn0)', 1500)
    group.add_float('rtol0', 'Solver error tolerance (rtol0)', 1e-12)

    options = ['0', '1', '2']
    group.add_option('nadv', 'Advection (ELM) option (nadv)', '1', options=options)
    group.add_float('dtb_max', 'Max steps allowed [seconds] (dtb_max)', 30.0)
    group.add_float('dtb_min', 'Min steps allowed [seconds] (dtb_min)', 10.0)

    options = ['-1', '0', '1']
    inter_mom = group.add_option(
        'inter_mom', 'Interpolation method at foot of characteristic line during ELM (inter_mom)', '0', options=options
    )
    kr_co = group.add_integer('kr_co', 'General covariance function (kr_co)', 1)
    _add_dependency(inter_mom, kr_co, {-1, 1})

    # Tracer transport method
    itr_met = group.add_integer(
        'itr_met',
        'Tracer transport method (itr_met)',
        3,
        description='3 for TVD, 4 for 3rd order WENO.',
        low=3,
        high=4
    )
    group.add_float('h_tvd', 'Cut-off depth [m] (h_tvd)', 5.0)
    flags = {i: i in {2, 3, 4} for i in range(-1000, 1000)}
    eps1_tvd_imp = group.add_float('eps1_tvd_imp', '1st tolerance of convergence (eps1_tvd_imp)', 1e-4)
    eps1_tvd_imp.add_dependency(itr_met, flags)
    eps2_tvd_imp = group.add_float('eps2_tvd_imp', '2nd tolerance of convergence (eps2_tvd_imp)', 1e-14)
    eps2_tvd_imp.add_dependency(itr_met, flags)
    group.add_boolean('ielm_transport', 'Hybrid ELM-FV transport for performance (ielm_transport)', False)
    group.add_integer('max_subcyc', 'Max # of subcycling per time step in transport allowed (max_subcyc)', 10)
    flags[3] = False
    itr_met_flags = {i: i == 4 for i in range(-1000, 1000)}
    options = [
        '0',
        '1',
        '2',
    ]
    ip_weno = group.add_option('ip_weno', 'Order of accuracy (ip_weno)', '2', options=options)
    ip_weno.add_dependency(itr_met, itr_met_flags)
    courant_weno = group.add_float('courant_weno', 'Courant number for weno transport (courant_weno)', 0.5)
    courant_weno.add_dependency(itr_met, itr_met_flags)
    options = ['1', '2']
    nquad = group.add_option('nquad', 'Number of quad points on each side (nquad)', '2', options=options)
    nquad.add_dependency(itr_met, itr_met_flags)
    options = ['1', '3']
    ntd_weno = group.add_option('ntd_weno', 'Order of temporal discretization (ntd_weno)', '1', options=options)
    ntd_weno.add_dependency(itr_met, itr_met_flags)
    epsilon1 = group.add_float(
        'epsilon1',
        'Coefficient for 2nd order weno smoother (epsilon1)',
        1e-15,
        description='Larger values are more prone to numerical dispersion'
    )
    epsilon1.add_dependency(itr_met, flags)
    group.add_integer('i_epsilon2', 'i_epsilon2', 1, low=1, high=2)
    epsilon2 = group.add_float(
        'epsilon2',
        '1st coefficient for 3rd order weno smoother (epsilon2)',
        1e-10,
        description='Larger values are more prone to numerical dispersion. 1.e-10 '
        'should be fairly safe, recommended values: 1.e-8 ~ 1.e-6'
    )
    epsilon2.add_dependency(itr_met, itr_met_flags)
    i_prtnftl_weno = group.add_boolean(
        'i_prtnftl_weno', 'Option for writing nonfatal errors on invalid temp. or salinity for density '
        '(i_prtnftl_weno)', False
    )
    group.add_float('epsilon3', 'epsilon3', 1.0e-25)
    i_prtnftl_weno.add_dependency(itr_met, itr_met_flags)
    # Inactive at the moment:
    # Elad filter has not been implemented yet; preliminary tests showed it might not be necessary
    group.add_integer('ielad', 'ielad', 0)
    group.add_float('small_elad', 'small_elad', 1.0e-4)
    group.add_float('ielad_weno', 'ielad_weno', 0.0)

    group.add_boolean('iprecip_off_bnd', 'Turn off precip near land bnd (iprecip_off_bnd)', False)

    group.add_boolean(
        'inu_elev',
        'Sponge layer for elevation (inu_elev)',
        False,
        description='If checked, relax. constants (in 1/sec, i.e. relax*dt<=1) are specified in '
        'elev_nudge.gr3 (thus a depth=0 means no relaxation).'
    )
    group.add_boolean(
        'inu_uv',
        'Sponge layer for velocity (inu_uv)',
        False,
        description='If checked, relax. constants (in 1/sec, i.e. relax*dt<=1) are specified in '
        'uv_nudge.gr3 (thus a depth=0 means no relaxation).'
    )

    inu_tr_flags = {'0': False, '1': True, '2': False}
    inu_tr_options = ['0', '1', '2']
    for inu_tr_name in INU_TR_NAMES:
        selector = group.add_option(inu_tr_name, inu_tr_name, '0', inu_tr_options)
        dataset = group.add_dataset(f'{inu_tr_name}_ds', f'{inu_tr_name} dataset')
        dataset.add_dependency(selector, inu_tr_flags)

    options = ['1', '2']
    group.add_option('nu_sum_mult', 'Final relaxation constant (nu_sum_mult)', options[0], options=options)
    group.add_float('vnh1', 'Vertical nudging depth 1 (vnh1)', 400.0)
    group.add_float('vnf1', 'Vertical relaxation constant 1 (vnf1)', 0.0)
    group.add_float('vnh2', 'Vertical nudging depth 2 (must >vnh1) (vnh2)', 500.0)
    group.add_float('vnf2', 'Vertical relaxation constant 2 (vnf2)', 0.0)
    group.add_float('step_nu_tr', 'Time step [sec] in all [MOD]_nu.nc (for inu_[MOD]=2) (step_nu_tr)', 86400.0)

    group.add_float(
        'h_bcc1', 'Cut-off depth for cubic spline interpolation near bottom when computing horizontal '
        'gradients (h_bcc1)', 100.0
    )

    group.add_float(
        's1_mxnbt',
        'Dimensioning parameters for inter-subdomain btrack (s1_mxnbt)',
        0.5,
        description='If error occurs like "bktrk_subs: overflow" or "MAIN: nbtrk > mxnbt" '
        'gradually increasing these will solve the problem'
    )
    group.add_float(
        's2_mxnbt',
        'Dimensioning parameters for inter-subdomain btrack (s2_mxnbt)',
        3.5,
        description='If error occurs like "bktrk_subs: overflow" or "MAIN: nbtrk > mxnbt" '
        'gradually increasing these will solve the problem'
    )

    # TODO: if True, input harm.in required
    group.add_boolean(
        'iharind',
        'Harmonic analysis flag (iharind)',
        False,
        description='If used , need to turn on USE_HA in Makefile, and input harm.in'
    )

    group.add_boolean('iflux', 'Conservation check option (iflux)', False)
    group.add_boolean('iflux_out_format', 'More detailed flux outputs (iflux_out_format)', False)

    group.add_boolean('izonal5', 'Williamson test #5 (izonal5)', False)
    group.add_boolean(
        'ibtrack_test',
        'Rotating Gausshill test with stratified T,S (ibtrack_test)',
        False,
        description='If on, ics must =2'
    )
    group.add_boolean(
        'irouse_test', 'Rouse profile test (irouse_test)', False, description='If on, must turn on USE_TIMOR'
    )

    options = ['1', '2', '3']
    group.add_option(
        'flag_fib', 'Flag to choose FIB model for bacteria decay (used with USE_FIB) (flag_fib)', '1', options=options
    )

    # TODO: Only if USE_MARSH is on
    # Marsh model parameters
    group.add_float('slr_rate', 'Sea-level rise rate in mm/yr (slr_rate)', 120.0)
    # TODO: extra inputs needed if True
    # If isav=1, need 4 extra inputs: (1) sav_D.gr3 (depth is stem diameter in meters);
    # ! (2) sav_N.gr3 (depth is # of stems per m^2);
    # ! (3) sav_h.gr3 (height of canopy in meters);
    # ! (4) sav_cd.gr3 (drag coefficient).
    group.add_boolean(
        'isav',
        '(isav)',
        False,
        description='If USE_MARSH is on and isav is checked, all .gr3 must have constant depths!'
    )

    group.add_integer('nstep_ice', 'Call ICE module every nstep_ice steps of SCHISM (nstep_ice)', 1)

    # Physical constants group
    group.add_float('rearth_pole', 'Earth\'s radii at pole (rearth_pole)', 6378206.4)
    group.add_float('rearth_eq', 'Earth\'s radii at equator (rearth_eq)', 6378206.4)
    group.add_float('shw', 'Specific heat of water (C_p) in J/kg/K (shw)', 4184.0)
    group.add_float('rho0', 'Reference water density for Boussinesq approximation in kg/m^3 (rho0)', 1000.0)
    group.add_float(
        'vclose_surf_frac',
        'vclose_surf_frac',
        1.0,
        description='Fraction of vertical flux closure adjustment applied at surface, then '
        'subtracted from all vertical fluxes. This is currently done for T,S only. '
        '1: fully from surface (i.e. no correction as before); 0: fully from bottom'
    )

    definition = [
        StringColumnType(header='Module', enabled=False),
        IntColumnType(
            header='iadjust_mass_consv0',
            low=0,
            high=1,
            checkbox=True,
            tool_tip='Option to enforce strict mass conservation. DVD and SED3D must be unchecked'
        )
    ]
    table_def = TableDefinition(definition, fixed_row_count=len(tracer_modules))
    default = [[module, 0] for module in tracer_modules]
    group.add_table(
        'iadjust_mass_consv0',
        'Option to enforce strict mass conservation (iadjust_mass_consv0)',
        default=default,
        table_definition=table_def
    )

    group.add_float(
        'h_massconsv',
        'h_massconsv [m]',
        2.0,
        description='For ICM, impose mass conservation for depths larger than a threshold by '
        'considering prism volume change from step n to n+1'
    )
    group.add_float('rinflation_icm', 'Max ratio btw H^{n+1} and H^n allowed (rinflation_icm)', 1.e-3)

    # Some parameters are considered basic, and always visible. Parameters that depend on them were also promoted to the
    # basic group due to limitations in GenericModel. It doesn't support multiple parents, so you can't have a parameter
    # that is visible only when the advanced toggle is on *and* the basic parameter it depends on has the correct value.
    #
    # Adding all of these manually was tedious, so we'll just go through all the parameters here and make anything
    # without a parent depend on the advanced toggle.
    basic_parameters = {
        'advanced', 'start_year', 'start_month', 'start_day', 'start_hour', 'utc_start', 'ics', 'ihot', 'dramp', 'nchi',
        'ncor', 'nws', 'ihconsv', 'isconsv', 'itur'
    }

    flags = {True: True, False: False}
    for name in group.parameter_names:
        parameter = group.parameter(name)
        if name not in basic_parameters and parameter.parent is None:
            parameter.add_dependency(advanced, flags)


def _add_schout_group(model: GenericModel):
    """Add the schout group to the global parameters."""
    group = model.global_parameters.add_group('schout', 'SCHOUT')

    group.add_boolean(
        'nc_out', 'Main switch to turn on/off netcdf outputs, useful for other programs (e.g., ESMF) '
        'to control outputs (nc_out)', True
    )

    group.add_boolean(
        'iof_ugrid',
        'UGRID option for _3D_ outputs under scribed IO (iof_ugrid)',
        False,
        description='If checked, 3D outputs will also have UGRID metadata (at the expense of '
        'file size)'
    )

    group.add_boolean(
        'nhot', 'Option for hotstart outputs (nhot)', False, description='Output *_hotstart every "nhot_write" steps'
    )
    group.add_integer(
        'nhot_write',
        'Steps to output *_hotstart (nhot_write)',
        8640,
        description='must be a multiple of ihfskip or nspool_sta if iout_sta is checked'
    )

    iout_sta = group.add_boolean('iout_sta', 'Station output option (iout_sta)', False)
    flags = {True: True, False: False}
    nspool_sta = group.add_integer(
        'nspool_sta', 'Output skip (nspool_sta)', 10, description='mod(nhot_write,nspool_sta) must=0'
    )
    nspool_sta.add_dependency(iout_sta, flags)

    iof_columns = [
        StringColumnType(enabled=False, header='Variable'),
        StringColumnType(enabled=False, header='Description (nc output variable name)'),
        StringColumnType(enabled=False, header='2D/3D'),
        IntColumnType(header='I/O Flag', high=1, low=0, checkbox=True),
    ]
    iof_def = TableDefinition(iof_columns, fixed_row_count=31)
    iof_defaults = [
        ['iof_hydro(1)', 'Elevation (elevation)', '2D', 1],
        ['iof_hydro(2)', 'Air pressure [Pa] (airPressure)', '2D', 0],
        ['iof_hydro(3)', 'Air temperature [C] (airTemperature)', '2D', 0],
        ['iof_hydro(4)', 'Specific humidity [-] (specificHumidity)', '2D', 0],
        ['iof_hydro(5)', 'Net downward solar (shortwave) radiation after albedo [W/m/m] (solarRadiation)', '2D', 0],
        ['iof_hydro(6)', 'Sensible flux (positive upward) [W/m/m]  (sensibleHeat)', '2D', 0],
        ['iof_hydro(7)', 'Latent heat flux (positive upward) [W/m/m] (latentHeat)', '2D', 0],
        ['iof_hydro(8)', 'Upward longwave radiation (positive upward) [W/m/m] (upwardLongwave)', '2D', 0],
        ['iof_hydro(9)', 'Downward longwave radiation (positive downward) [W/m/m] (downwardLongwave)', '2D', 0],
        ['iof_hydro(10)', 'Total flux=-flsu-fllu-(radu-radd) [W/m/m] (totalHeat)', '2D', 0],
        ['iof_hydro(11)', 'Evaporation rate [kg/m/m/s] (evaporationRate)', '2D', 0],
        ['iof_hydro(12)', 'Precipitation rate [kg/m/m/s] (precipitationRate)', '2D', 0],
        ['iof_hydro(13)', 'Bottom stress vector [kg/m/s^2(Pa)] (bottomStressX,Y)', '2D vector', 0],
        ['iof_hydro(14)', 'Wind velocity vector [m/s] (windSpeedX,Y)', '2D vector', 0],
        ['iof_hydro(15)', 'Wind stress vector [m^2/s/s] (windStressX,Y)', '2D vector', 0],
        ['iof_hydro(16)', 'Depth-averaged vel vector [m/s] (depthAverageVelX,Y)', '2D vector', 0],
        ['iof_hydro(17)', 'Vertical velocity [m/s] (verticalVelocity)', '3D', 0],
        ['iof_hydro(18)', 'Water temperature [C] (temperature)', '3D', 0],
        ['iof_hydro(19)', 'Water salinity [PSU] (salinity)', '3D', 0],
        ['iof_hydro(20)', 'Water density [kg/m^3] (waterDensity)', '3D', 0],
        ['iof_hydro(21)', 'Vertical eddy diffusivity [m^2/s] (diffusivity)', '3D', 0],
        ['iof_hydro(22)', 'Vertical eddy viscosity [m^2/s] (viscosity)', '3D', 0],
        ['iof_hydro(23)', 'Turbulent kinetic energy (turbulentKineticEner)', '3D', 0],
        ['iof_hydro(24)', 'Turbulent mixing length [m] (mixingLength)', '3D', 0],
        ['iof_hydro(25)', 'Z-coord (this flag should be on for visIT, etc) (zCoordinates) ', '3D', 1],
        ['iof_hydro(26)', 'Horizontal vel vector [m/s] (horizontalVelX,Y)', '3D vector', 1],
        ['iof_hydro(27)', 'Horizontal vel vector defined @side [m/s] (horizontalSideVelX,Y', '3D vector', 0],
        ['iof_hydro(28)', 'Vertical vel. @elem [m/s] (verticalVelAtElement)', '3D vector', 0],
        ['iof_hydro(29)', 'T @prism centers [C] (temperatureAtElement)', '3D vector', 0],
        ['iof_hydro(30)', 'S @prism centers [PSU] (salinityAtElement)', '3D vector', 0],
        [
            'iof_hydro(31)', 'Barotropic pressure gradient force vector (m.s-2) @side centers  (pressure_gradient)',
            '2D vector', 0
        ]
    ]
    group.add_table('iof_hydro', 'Global output options (iof_hydro)', iof_defaults, iof_def)


def _add_other_group(model: GenericModel):
    """Add the other group to the global parameters."""
    group = model.global_parameters.add_group('other', 'OTHER')

    group.add_option(
        'executable',
        'Model executable',
        default='OLDIO PREC_EVAP TVD-VL',
        options=['OLDIO PREC_EVAP TVD-VL', 'OLDIO TVD-VL']
    )

    group.add_float(
        'tip_dp', 'Cut-off depth for applying tidal potential (i.e., it is not calculated when depth < tip_dp', 0.0
    )

    group.add_option('ivcor', 'ivcor', '2', options=['1', '2'])

    definition = [FloatColumnType(header='Depth', low=0.0)]
    table_def = TableDefinition(definition)
    default = []
    group.add_table('z_levels', 'Z levels', default=default, table_definition=table_def)

    group.add_float('hc', 'hc', 0.0)
    group.add_float('theta_b', 'theta_b', 0.0)
    group.add_float('theta_f', 'theta_f', 0.0)

    definition = [FloatColumnType(header='Depth', low=0.0)]
    table_def = TableDefinition(definition)
    default = []
    group.add_table('s_levels', 'S levels', default=default, table_definition=table_def)

    group.add_integer('processes', 'Number of processes', default=4, low=1)


def _add_materials(model: GenericModel):
    """Add materials to the model."""
    model.material_parameters.add_group('upwind', 'Upwind')
    model.material_parameters.add_group('higher_order', 'Higher order')


def _add_dependency(parent: Parameter, child: Parameter, on: set[int]):
    """
    Make a parameter dependent on another parameter.

    Args:
        parent: The parent parameter.
        child: The child parameter.
        on: Values of the parent for which the child should be enabled.
    """
    flags = {key: int(key) in on for key in parent.options}
    child.add_dependency(parent, flags)


def elevation_flag(section: Section) -> int:
    """
    Get the elevation flag.

    Args:
        section: Arc parameters.

    Returns:
        The elevation type flag.
    """
    parameter = section.group('open').parameter('wse-type')
    return parameter.options.index(parameter.value)


def velocity_flag(section: Section) -> int:
    """
    Get the velocity flag.

    Args:
        section: Arc parameters.

    Returns:
        The velocity type flag.
    """
    parameter = section.group('open').parameter('flow-type')
    return parameter.options.index(parameter.value)


def salinity_flag(section: Section) -> int:
    """
    Get the salinity flag.

    Args:
        section: Arc parameters.

    Returns:
        The salinity type flag.
    """
    parameter = section.group('open').parameter('salinity-type')
    return parameter.options.index(parameter.value)


def temperature_flag(section: Section) -> int:
    """
    Get the temperature flag.

    Args:
        section: Arc parameters.

    Returns:
        The temperature type flag.
    """
    parameter = section.group('open').parameter('temperature-type')
    return parameter.options.index(parameter.value)


def _add_initial_conditions(section: Section, files: set[str]):
    """
    Add initial condition files.

    salt.ic, temp.ic, ts.ic, [mod]_hvar_[class].ic, [mod]_vvar_[class].ic.

    Args:
        section: Global parameters.
        files: Where to add files to.
    """
    opt = section.group('opt')

    if opt.parameter('flag_ic(1)').value != opt.parameter('flag_ic(2)').value:
        raise ValueError('flag_ic(1) must equal flag_ic(2)')

    for name in FLAG_IC_NAMES:
        flag_ic = section.group('opt').parameter(name).value
        assert isinstance(flag_ic, str)
        if flag_ic == '1' and name in SPECIAL_FLAG_IC_NAMES:
            files.update({'salt.ic', 'temp.ic'})
        elif flag_ic == '1':
            raise ValueError(f'{name}=1 is not currently supported')
        elif flag_ic == '2':
            raise ValueError(f'{name}=2 is not currently supported')


def _add_nudging(section: Section, files: set[str]):
    """
    Add nudging files.

    [mod]_nudge.gr3, and [mod]_nu.nc.

    Args:
        section: Global parameters.
        files: Where to add new files to.
    """
    for parameter_name in INU_TR_NAMES:
        inu_tr = section.group('opt').parameter(parameter_name).value
        assert isinstance(inu_tr, str)
        if inu_tr == '2':
            # files.add(f'{tracer_name}_nu.nc')
            raise ValueError(f'{parameter_name}=2 is not currently supported')
        if inu_tr != '0':
            # files.add(f'{tracer_name}_nudge.gr3')
            raise ValueError(f'{parameter_name}/=0 is not currently supported')


def needed_files(section: Section) -> set[str]:
    """
    Get all the files that are needed by a section.

    Args:
        section: Global parameters for the model.

    Returns:
        Set of file names that are needed.
    """
    files = set()

    ihot = section.group('opt').parameter('ihot').value
    assert isinstance(ihot, str)
    is_hotstart = ihot != '0'
    if is_hotstart:
        files.add('hotstart.nc')
        # raise ValueError('ihot!=0 is not currently supported')

    ibc = section.group('core').parameter('ibc').value
    ibtp = section.group('core').parameter('ibtp').value
    assert isinstance(ibc, bool) and isinstance(ibtp, bool)
    if not is_hotstart and (ibtp or not ibc):
        files.add('tvd.prop')
        # Adds salt.ic, temp.ic, ts.ic, [mod]_hvar_[class].ic, [mod]_vvar_[class].ic
        _add_initial_conditions(section, files)

    if_source = section.group('opt').parameter('if_source').value
    assert isinstance(if_source, str)
    if if_source == '-1':
        # files.add('source.nc')
        raise ValueError('if_source=-1 is not currently supported')
    if if_source == '1':
        # files.add('msource.th')
        # files.add('source_sink.in')
        # files.add('vsink.th')
        # files.add('vsource.th')
        raise ValueError('if_source=1 is not currently supported')

    iflux = section.group('opt').parameter('iflux').value
    assert isinstance(iflux, bool)
    if iflux:
        # files.add('fluxflag.prop')
        raise ValueError('iflux=1 is not currently supported')

    iharind = section.group('opt').parameter('iharind').value
    assert isinstance(iharind, bool)
    if iharind:
        # files.add('harm.in')
        raise ValueError('iharind=1 is not currently supported')

    ihconsv = section.group('opt').parameter('ihconsv').value
    assert isinstance(ihconsv, bool)
    if ihconsv:
        files.add('albedo.gr3')
        files.add('watertype.gr3')

    ihdif = section.group('opt').parameter('ihdif').value
    assert isinstance(ihdif, bool)
    if ihdif:
        # files.add('hdif.gr3')
        raise ValueError('ihdif=1 is not currently supported')

    ihydraulics = section.group('opt').parameter('ihydraulics').value
    assert isinstance(ihydraulics, bool)
    if ihydraulics:
        # files.add('hydraulics.in')
        raise ValueError('ihydraulics=1 is not currently supported')

    iloadtide = section.group('opt').parameter('iloadtide').value
    assert isinstance(iloadtide, str)
    if iloadtide == '1':
        # files.add('loadtide_[freq].gr3')
        raise ValueError('iloadtide=1 is not currently supported')

    inter_mom = section.group('opt').parameter('inter_mom').value
    assert isinstance(inter_mom, str)
    if inter_mom == '-1':
        # files.add('krvel.gr3')
        raise ValueError('inter_mom=-1 is not currently supported')

    if not is_hotstart:  # Nudging is stored in the hotstart file for hotstart cases.
        _add_nudging(section, files)  # Adds [mod]_nudge.gr3, [mod]_nu.nc

    inu_uv = section.group('opt').parameter('inu_uv').value
    assert isinstance(inu_uv, bool)
    if inu_uv:
        # files.add('uv_nudge.gr3')
        raise ValueError('inu_uv=1 is not currently supported')

    iout_sta = section.group('schout').parameter('iout_sta').value
    assert isinstance(iout_sta, bool)
    if iout_sta:
        # files.add('station.in')
        raise ValueError('iout_sta=1 is not currently supported')

    isav = section.group('opt').parameter('isav').value
    assert isinstance(isav, bool)
    if isav:
        # files.add('sav_cd.gr3')
        # files.add('sav_d.gr3')
        # files.add('sav_n.gr3')
        # files.add('sav_h.gr3')
        raise ValueError('isav=1 is not currently supported')

    ishapiro = section.group('opt').parameter('ishapiro').value
    assert isinstance(ishapiro, str)
    if ishapiro == '-1':
        # files.add('shapiro.gr3')
        raise ValueError('ishapiro=-1 is not currently supported')

    itur = section.group('opt').parameter('itur').value
    assert isinstance(itur, str)
    if itur == '-2':
        # files.add('hvd.mom')
        # files.add('hvd.tran')
        raise ValueError('itur=-2 is not currently supported')
    if itur == '-1':
        # files.add('vvd.dat')
        raise ValueError('itur=-1 is not currently supported')

    nchi = section.group('opt').parameter('nchi').value
    assert isinstance(nchi, str)
    if nchi == '-1':
        files.add('manning.gr3')
    elif nchi == '0':
        files.add('drag.gr3')
    elif nchi == '1':
        files.add('rough.gr3')
    else:
        # At least one of these is required.
        raise AssertionError('Unrecognized value for nchi')

    nws = section.group('opt').parameter('nws').value
    assert isinstance(nws, str)
    nws = int(nws)
    iwindoff = section.group('opt').parameter('iwindoff').value
    assert isinstance(iwindoff, bool)
    if nws == -1:
        # files.add('hurricane-track.dat')
        raise ValueError('nws=-1 is not currently supported')
    if nws == 2:
        files.add('sflux/sflux_air_1.*.nc')
        isconsv = section.group('opt').parameter('isconsv').value
        assert isinstance(isconsv, bool)
        if isconsv:
            files.add('sflux/sflux_prc_1.*.nc')
        if ihconsv:
            files.add('sflux/sflux_rad_1.*.nc')
    if nws != 0:
        files.add('windrot_geo2proj.gr3')

    if nws == 1 or nws == 4:
        # files.add('wind.th')
        raise ValueError('nws=1 or nws=4 is not currently supported')
    if nws > 0 and iwindoff:
        # files.add('windfactor.gr3')
        raise ValueError('nws>0 with iwindoff=1 is not currently supported')

    return files


def parameter_for_file(file: str) -> tuple[str, str]:
    """
    Get the parameter that holds the dataset UUID for a file.

    Args:
        file: Name of the file. Does not include full path, just something like 'file.ext'.

    Returns:
        Tuple of (group_name, parameter_name) describing the parameter the dataset's UUID should go in.
    """
    mapping = {
        'albedo.gr3': ('opt', 'albedo_ds'),
        'drag.gr3': ('opt', 'drag_ds'),
        'manning.gr3': ('opt', 'manning_ds'),
        'rough.gr3': ('opt', 'rough_ds'),
        'salt.ic': ('opt', 'flag_ic(1)_dataset'),
        'temp.ic': ('opt', 'flag_ic(2)_dataset'),
        'watertype.gr3': ('opt', 'watertype_ds'),
        'windrot_geo2proj.gr3': ('opt', 'windrot_geo2proj_ds'),
    }

    if file in mapping:
        return mapping[file]

    raise AssertionError(f'No parameter for file {file}')


def enabled(section: Section, group: str, parameter: str) -> bool:
    """
    Check whether a parameter is enabled.

    Args:
        section: The section containing the parameter.
        group: Name of the group the parameter is in.
        parameter: Name of the parameter.

    Returns:
        Whether the parameter is enabled.
    """
    parameter = section.group(group).parameter(parameter)
    if not parameter.parent:
        # A parent can only be disabled if its parent has a bad value. If no parent, always enabled.
        return True

    _, parent_group, parent_parameter = parameter.parent
    parent = section.group(parent_group).parameter(parent_parameter)
    flags = parameter.dependency_flags

    if not flags[parent.value]:
        return False

    # The parent had an acceptable value, but it may have its own parent that disabled it.
    return enabled(section, parent_group, parent_parameter)
