"""Module for the SimRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ExecutableCommand, Query
from xms.components.bases.run_base import RunBase
from xms.data_objects.parameters import Simulation
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.schism.data.model import get_model
from xms.schism.data.sim_data import SimData
from xms.schism.entry_points import run_model as runner_module

_SECONDS_PER_DAY = 24 * 60 * 60


class SimRunner(RunBase):
    """Class that finds executables for running SCHISM."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

    def get_executables(self, do_sim: Simulation, query: Query, file_location) -> list[ExecutableCommand]:
        """
        Get the executable commands for any Simulation object given.

        Args:
            do_sim: The Simulation you want to load the solution for.
            query: Inter-process communication object.
            file_location: The location of input files for the simulation.

        Returns:
            The executable objects to run and the action requests that go with it.
        """
        simulation = query.item_with_uuid(do_sim.uuid, model_name='SCHISM', unique_name='SimComponent')
        main_file = simulation.main_file
        data = SimData(main_file)

        model = get_model()
        model.global_parameters.restore_values(data.global_values)

        rnday = model.global_parameters.group('core').parameter('rnday').value
        rnsec = rnday * _SECONDS_PER_DAY
        dt = model.global_parameters.group('core').parameter('dt').value
        time_steps = int(rnsec // dt)

        processes = model.global_parameters.group('other').parameter('processes').value
        executable = model.global_parameters.group('other').parameter('executable').value
        key = f'SCHISM - SCHISM {executable}'

        settings = SettingsManager(python_path=False)
        package = 'File_Location_Preferences'
        schism_path = str(settings.get_setting(package, key, ''))
        mpi_path = str(settings.get_setting(package, 'MsMPI', ''))

        params = {
            'schism': schism_path,
            'mpi': mpi_path,
            'steps': time_steps,
            'processes': processes,
        }
        params_path = Path(file_location) / 'xms_params.txt'
        with open(params_path, 'w') as f:
            f.write(repr(params))

        cmd = ExecutableCommand(
            executable=runner_module.__file__,
            model='SCHISM',
            display_name='Running the SCHISM model',
            run_weight=100,
            progress_script='entry_points/progress.py',
            executable_is_script=True,
        )

        return [cmd]
