"""Progress reporter for running SCHISM."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import ast
import logging
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.core.filesystem.filesystem import removefile

# 4. Local modules


class FileLogger:
    """Class that looks like a logging.logger but logs its messages to a text file."""
    def __init__(self):
        """Initialize the file logger."""
        log_file = 'xms_log.txt'
        removefile(log_file)

        module = self.__module__
        first_two_components = module.split('.')[:2]
        logger_name = '.'.join(first_two_components)
        self._logger = logging.getLogger(logger_name)

        handler = logging.FileHandler(log_file)
        self._logger.addHandler(handler)

        self._logger.setLevel(logging.INFO)

        self.info = self._logger.info


class SchismTracker:
    """Class that tracks a SCHISM model running in SMS."""
    query = None
    logger = None
    model_log_path = None
    echo_file_pos = 0
    num_time_steps = 30
    current_time_step = 0
    last_reported_time_step = 0

    pest_iter = 0
    prog_increment = 0
    model_prog_increment = 0
    current_progress = 0

    @staticmethod
    def progress_function():
        """Calculates the progress and sends it to SMS."""
        if not os.path.exists(SchismTracker.model_log_path):
            return

        with open(SchismTracker.model_log_path, "r") as model_output:
            model_output.seek(SchismTracker.echo_file_pos)

            while echo_line := model_output.readline():
                if not echo_line.endswith('\n'):
                    return
                SchismTracker.echo_file_pos = model_output.tell()

                if echo_line.startswith('Finished step '):
                    echo_line = echo_line.removeprefix('Finished step ').strip()
                    SchismTracker.current_time_step = int(echo_line)

        if SchismTracker.current_time_step != SchismTracker.last_reported_time_step:
            SchismTracker.last_reported_time_step = SchismTracker.current_time_step
            fractional_percent = SchismTracker.current_time_step / SchismTracker.num_time_steps
            whole_percent = int(fractional_percent * 100)
            SchismTracker.query.update_progress_percent(whole_percent)

    @staticmethod
    def start_tracking():
        """Entry point for the SRH-Post progress script."""
        SchismTracker.model_log_path = 'schism_log.txt'
        removefile(SchismTracker.model_log_path)
        SchismTracker.logger = FileLogger()

        with open('xms_params.txt') as f:
            content = f.read()
            values = ast.literal_eval(content)
            SchismTracker.num_time_steps = values['steps']

        # do this last
        SchismTracker.query = Query(progress_script=True)
        prog = SchismTracker.query.xms_agent.session.progress_loop
        prog.set_progress_function(SchismTracker.progress_function)
        prog.start_loop()


if __name__ == '__main__':
    tracker = SchismTracker()
    tracker.start_tracking()
