"""Module for projection utility."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UGrid2d
from xms.data_objects.parameters import Projection
from xms.gdal.utilities.gdal_utils import is_geographic, is_local, transform_points_from_wkt
from xms.grid.ugrid import UGrid

# 4. Local modules

GEOGRAPHIC_WKT = (
    """GEOGCS["Geographic Coordinate System",DATUM["D_NORTH_AMERICAN_1983",SPHEROID[
        "GRS_1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["Degree",0.017453292519943295]]"""
)


def make_geographic(projection: Projection, ugrid: UGrid | UGrid2d) -> tuple[bool, UGrid2d | None]:
    """
    Make a UGrid geographic.

    If the UGrid is already geographic, it is left unchanged. If reprojection is necessary, the UGrid's points will be
    reprojected into an arbitrary geographic projection (currently WGS84).

    Reprojection will fail if the projection is local or has no well_known_text.

    Args:
        projection: The current projection of the grid.
        ugrid: The ugrid to project.

    Returns:
        Whether the UGrid is now in geographic coordinates.
    """
    if projection.well_known_text == '' or is_local(projection.well_known_text):
        return False, None

    if is_geographic(projection.well_known_text):
        return True, ugrid

    if isinstance(ugrid, UGrid2d):
        ugrid = ugrid.ugrid

    points = ugrid.locations
    projected_points = transform_points_from_wkt(points, projection.well_known_text, GEOGRAPHIC_WKT)

    xm_ugrid = UGrid(points=projected_points, cellstream=ugrid.cellstream)
    ugrid_2d = UGrid2d(ugrid=xm_ugrid)
    ugrid_2d.uuid = str(uuid.uuid4())
    return True, ugrid_2d


def looks_geographic(ugrid: UGrid2d):
    """
    Check whether a UGrid looks like it's in a geographic projection.

    SCHISM has a habit of not saving its coordinate system anywhere, so we have no idea what projection its geometry is
    in when we import its model-native files. It often makes use of our tidal databases, which expect UGrids in
    geographic coordinate systems, so it's convenient for the user if we can at least identify those and automatically
    assign their projection.

    This function is responsible for attempting to identify them. It makes a best-effort at guessing.

    A UGrid "looks geographic" if all its X-coordinates are in the range [-180, 180] and all its Y-coordinates are in
    the range [-90, 90]. It is possible for a UGrid to be in a non-geographic coordinate system and still have all its
    coordinates in this range, so a return value of True is not a guarantee that the UGrid is actually geographic. A
    return value of False, however, means it is almost certainly non-geographic (or just corrupted, which is arguably
    not geographic anyway).

    Args:
        ugrid: The UGrid to check.

    Returns:
        Whether the UGrid looks geographic.
    """
    locations = ugrid.ugrid.locations

    xs = locations[:, 0]
    if (xs < -180.0).any() or (xs > 180.0).any():
        return False

    ys = locations[:, 1]
    if (ys < -90.0).any() or (ys > 90.0).any():
        return False

    return True
