"""Module for the MapTidesRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"
__all__ = ['MapTidesRunner']

# 1. Standard Python modules
from functools import cached_property
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.data_objects.parameters import Component, UGrid
from xms.gmi.components.utils import new_component_dir
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread
from xms.tides.data.tidal_data import TidalData

# 4. Local modules
from xms.schism.data.mapped_bc_data import MappedBcData
from xms.schism.data.sim_data import SimData
from xms.schism.external.crc import compute_crc
from xms.schism.external.mapped_tidal_data import MappedTidalData
from xms.schism.tides.tidal_mapper import TidalMapper


class MapTidesRunner(FeedbackThread):
    """Worker thread for mapping the tides."""
    def __init__(self, tide_uuid: str, query: Query):
        """
        Initialize the runner.

        Args:
            tide_uuid: UUID of the domain to map.
            query: Interprocess communication object.
        """
        super().__init__(query)

        self.display_text = {
            'title':
                'SCHISM Map Tidal Constituents',
            'working_prompt':
                'Mapping tidal constituents, please wait...',
            'warning_prompt':
                'Warning(s) encountered while mapping tidal constituents. Review log output for more details.',
            'error_prompt':
                'Error(s) encountered while mapping tidal constituents. Review log output for more details.',
            'success_prompt':
                'Successfully mapped tidal constituents',
            'note':
                '',
            'auto_load':
                'Close this dialog automatically when task is finished.'
        }

        self._mapped_tide_dir = new_component_dir()
        self._mapped_tide_main_file = self._mapped_tide_dir / 'schism_mapped_tides.nc'
        self._current_domain_hash = ''

        # Information for unmapped tides
        self._tide_uuid = tide_uuid

    @cached_property
    def _current_item(self):
        """The data object for the sim component."""
        return self._query.current_item()

    @cached_property
    def _parent_item(self):
        """
        The data object container for the simulation.

        (not the component)
        """
        return self._query.parent_item()

    @cached_property
    def _parent_item_uuid(self):
        """
        The UUID of the simulation.

        (not the component)
        """
        return self._parent_item.uuid

    @cached_property
    def _project_tree(self):
        """The project tree."""
        return self._query.copy_project_tree()

    @cached_property
    def _mapped_bc_data(self) -> MappedBcData:
        """The data manager for the mapped boundary condition coverage."""
        project_tree = self._project_tree
        sim_node = tree_util.find_tree_node_by_uuid(project_tree, self._parent_item_uuid)
        mapped_bc_node = tree_util.descendants_of_type(sim_node, unique_name='MappedBcComponent', only_first=True)
        if not mapped_bc_node:
            raise ExpectedError('Applying tidal constituents requires a mapped BC coverage.')
        main_file = mapped_bc_node.main_file
        mapped_coverage_data = MappedBcData(main_file)
        return mapped_coverage_data

    def _run(self):
        """Run the mapper."""
        self._get_sim_data()
        self._unlink_tides()
        self._map_tides()
        self._link_component()

    def _get_sim_data(self):
        """Get the data manager for the simulation."""
        main_file = self._current_item.main_file
        self._sim_data = SimData(main_file)

    def _unlink_tides(self):
        """Unlink the unmapped tidal simulation."""
        self._query.unlink_item(self._parent_item_uuid, self._tide_uuid)

    def _map_tides(self):
        """Run the TidalMapper to get tide data.."""
        tidal_data = self._get_tidal_data()
        domain = self._get_domain()

        bc_data = self._mapped_bc_data
        if bc_data.domain_hash != self._current_domain_hash:
            raise ExpectedError(
                'The domain was changed after applying the boundary condition coverage to the simulation.\n'
                'Reapply the coverage to the simulation to update it.'
            )

        ocean_node_indexes = self._get_ocean_node_ids()

        mapper = TidalMapper(self._query, self._log, tidal_data, ocean_node_indexes, domain)
        mapper.map()

        mapped_data = MappedTidalData(self._mapped_tide_main_file)
        mapped_data.properties = mapper.properties.sortby('name')
        mapped_data.elevation = mapper.elevation.sortby('name')
        mapped_data.velocity = mapper.velocity.sortby('name')
        mapped_data.source = mapper.source
        mapped_data.domain_hash = self._current_domain_hash
        mapped_data.coverage_uuid = self._mapped_bc_data.uuid
        mapped_data.commit()

        self._sim_data.mapped_tides_uuid = mapped_data.uuid
        self._sim_data.commit()

    def _get_tidal_data(self) -> TidalData:
        """Get the TidalData for the tidal simulation being mapped."""
        do_tides = self._query.item_with_uuid(
            self._tide_uuid, model_name='Tidal Constituents', unique_name='Tidal_Component'
        )
        main_file = do_tides.main_file

        # I'm not sure why copying the main file is necessary. ADCIRC does it.
        new_main_file = self._mapped_tide_dir / 'schism_unmapped_tides.nc'
        shutil.copy(main_file, new_main_file)

        return TidalData(str(new_main_file))

    def _get_ocean_node_ids(self):
        """Get the IDs of nodes in the mesh that are part of ocean boundaries."""
        return self._mapped_bc_data.open_nodes

    def _get_domain(self) -> UGrid:
        """Get the domain being mapped to."""
        self._log.info('Reading domain...')
        tree = self._project_tree
        sim_node = tree_util.find_tree_node_by_uuid(tree, self._parent_item_uuid)
        domain_node = tree_util.descendants_of_type(
            sim_node, xms_types=['TI_MESH2D_PTR'], only_first=True, allow_pointers=True
        )
        if not domain_node:
            raise ExpectedError('Applying tidal constituents requires a mapped mesh.')
        domain_uuid = domain_node.uuid
        domain = self._query.item_with_uuid(domain_uuid)
        ugrid = read_grid_from_file(domain.cogrid_file)
        self._current_domain_hash = compute_crc(domain.cogrid_file)
        return ugrid.ugrid

    def _link_component(self):
        """Add the mapped tidal component and link it to the simulation."""
        tide_name = f'{self._query.item_with_uuid(self._tide_uuid).name} (applied)'

        do_component = Component(
            main_file=str(self._mapped_tide_main_file),
            name=tide_name,
            model_name='SCHISM',
            unique_name='MappedTidalComponent',
            comp_uuid=self._mapped_tide_dir.name
        )

        self._query.add_component(do_component=do_component)
        self._query.link_item(taker_uuid=self._parent_item_uuid, taken_uuid=do_component.uuid)
