"""CheckRunner class. Runs the model check."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.srh.check.check_simulation import CheckSimulation
from xms.srh.components.sim_query_helper import SimQueryHelper


class CheckRunner:
    """Writes SRH-2D input files."""
    def __init__(self, query_helper=None):
        """Constructor.

        Args:
            query_helper (SimQueryHelper): class for getting data from xms
        """
        super().__init__()
        self._sim_query_helper = query_helper
        self._query = None
        self._simulation_name = None
        self.sim_component = None
        self.wse_dataset = False
        self.co_grid = None
        self.ugrid = None
        self.grid_units = None
        self.bc_data = None
        self.bc_comp_ids_to_arc_ids = None
        self.mat_data = None
        self.nlcd_raster = ''

    def run_check(self):
        """Creates the snap preview of coverages onto the mesh."""
        self._setup_query()
        self._get_data()

        checker = CheckSimulation(self)
        checker.run_check()

    def _setup_query(self):
        """Setup communication with XMS."""
        if not self._sim_query_helper:
            self._query = Query()
            self._sim_query_helper = SimQueryHelper(self._query, at_sim=True)
            self._sim_query_helper.get_sim_data()
        self._simulation_name = self._sim_query_helper.sim_tree_item.name

    def _get_data(self):
        """Set member variables from data in SimQueryHelper."""
        self.sim_component = self._sim_query_helper.sim_component
        self.grid_units = self._sim_query_helper.grid_units
        self.co_grid = self._sim_query_helper.co_grid
        if self._sim_query_helper.co_grid:
            self.ugrid = self._sim_query_helper.co_grid.ugrid
        bc = self._sim_query_helper.bc_component
        if bc:
            self.bc_data = bc.data
            if bc.cov_uuid in bc.comp_to_xms:
                self.bc_comp_ids_to_arc_ids = bc.comp_to_xms[bc.cov_uuid]
        mat = self._sim_query_helper.material_component
        if mat:
            self.mat_data = mat.data
        self.nlcd_raster = self._sim_query_helper.nlcd_raster_file
        if self._sim_query_helper.wse_dataset is not None:
            self.wse_dataset = self._sim_query_helper.wse_dataset
