"""Module for generating pressure datasets from 3D Bridge coverages via the SMS GUI."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json
import logging
import os

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.bridge.bridge_component import BridgeComponent
from xms.bridge.bridge_io import BridgeIo
from xms.bridge.grid_builder import GridBuilder
from xms.grid.ugrid.ugrid_utils import read_ugrid_from_ascii_file
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.srh.bridges.ceiling_generator import CeilingGenerator
from xms.srh.components.sim_query_helper import SimQueryHelper


class GenerateCeilingThread(QThread):
    """Worker thread for generating pressure datasets from 3D Bridge coverages."""
    processing_finished = Signal()

    def __init__(self, target_ugrid, bridge_data, ceiling_file_name, grid_units):
        """Construct the worker.

        Args:
            target_ugrid (:obj:`xms.grid.ugrid.UGrid`): Geometry to generate the dataset for
            bridge_data (:obj:`list[xms.grid.ugrid.UGrid]`): Bridge bottom UGrids.
            ceiling_file_name (:obj:`str`): file to write ceiling data
            grid_units (:obj:`str`): Units for the grid (METERS or FEET)
        """
        super().__init__()
        self._target_ugrid = target_ugrid
        self._grid_units = grid_units
        self._bridge_data = bridge_data
        self._ceiling_file_name = ceiling_file_name
        self._logger = logging.getLogger('xms.srh')

    def run(self):
        """Run the pressure dataset generation operation."""
        try:
            generator = CeilingGenerator(
                self._target_ugrid, self._bridge_data, self._ceiling_file_name, self._grid_units
            )
            generator.generate()
        except Exception:  # pragma no cover - hard to compare logging files while testing
            self._logger.exception('Error generating pressure data sets from 3D Bridge coverage(s).')
        finally:
            self.processing_finished.emit()


class GenerateCeilingRunner:
    """Retrieves user input and performs XMS interprocess communication for the generate pressure datasets command."""
    def __init__(self, sim_component, query, parent):
        """Constructor.

        Args:
            sim_component (:obj:`SimComponent`): the simulation component
            query (:obj:`Query`): XMS interprocess communication object
            parent (:obj:`object`): Qt parent for dialogs
        """
        self._sim_component = sim_component
        self._mesh_uuid = sim_component.data.mesh_uuid
        self._query = query
        self._parent = parent
        self._pe_tree = None
        self._bridges = []  # [bridge bottom ugrid]
        self.error_msg = ''
        self._logger = logging.getLogger('xms.srh')

    def _get_and_filter_project_tree(self):
        """Get the XMS project explorer tree and filter out non-3D Bridge coverage items.

        Also ensures the target SRH mesh exists. Right now has to be the mesh linked
        to the simulation this command came from, but that might change.

        Returns:
            (:obj:`bool`): True if no problems
        """
        # Get the XMS project explorer tree.
        self._pe_tree = self._query.copy_project_tree()
        if not self._pe_tree:
            self.error_msg = 'Unable to retrieve SMS project explorer tree.'
            self._logger.error(self.error_msg)
            return False

        # Ensure the target geometry exists.
        if not self._mesh_uuid or not tree_util.find_tree_node_by_uuid(self._pe_tree, self._mesh_uuid):
            msg = 'A mesh must be linked to the simulation before generating pressure data sets.'
            self.error_msg = msg
            self._logger.error(self.error_msg)
            return False

        # Filter the tree for the dataset selector dialog.
        tree_util.filter_project_explorer(self._pe_tree, self.is_3d_bridge_if_coverage)
        return True

    def _get_inputs_from_xms(self, bridge_coverages):
        """Get the target geometry, input coverage geometries, and input coverage component main files.

        Also retrieve the target mesh geometry here.

        Args:
            bridge_coverages (:obj:`list[str]`): UUIDs of the input 3D Bridge coverages

        Returns:
            (:obj:`bool`): True if component main files were successfully retrieved
        """
        bridge_data = []

        try:
            for bridge_cov in bridge_coverages:
                cov_geom = self._query.item_with_uuid(bridge_cov)
                cov_comp = self._query.item_with_uuid(bridge_cov, model_name='3D Bridge', unique_name='Bridge')
                bridge_data.append((cov_geom, cov_comp.main_file))
        except Exception:
            self.error_msg = 'Unable to retrieve selected 3D Bridge coverage data.'
            self._logger.error(self.error_msg)
            return False

        for b in bridge_data:
            try:
                # self._logger.info('Loading 3D Bridge coverage attributes...')
                geometry = b[0]
                main_file = b[1]
                # Load the 3D Bridge coverage component data.
                bridge_reader = BridgeIo()
                bridge = bridge_reader.read(main_file)
                bridge_comp = BridgeComponent(main_file)

                # Find the feature arcs that are defined as the bridge upstream and downstream footprint lines.
                coverage_arcs = geometry.arcs
                upstream_arc = bridge_comp.get_arc(bridge.arc_id_upstream, coverage_arcs)
                downstream_arc = bridge_comp.get_arc(bridge.arc_id_downstream, coverage_arcs)
                if not upstream_arc or not downstream_arc:
                    msg = f'Invalid 3D Bridge definition detected for "{geometry.name}". Aborting.'
                    self.error_msg = msg
                    self._logger.error(self.error_msg)
                    return False

                # Create the 3D Bridge geometry.
                # self._logger.info('Constructing 3D Bridge geometry...')
                grid_builder = GridBuilder()
                upstream_pts = bridge_comp.get_arc_points(upstream_arc)
                downstream_pts = bridge_comp.get_arc_points(downstream_arc)
                wkt = ''
                grid_builder.build_top_and_bottom(bridge, upstream_pts, downstream_pts, wkt, main_file)

                # Read the UGrid representing the bottom of the bridge.
                bottom_ugrid = read_ugrid_from_ascii_file(os.path.join(os.path.dirname(main_file), 'bottom2d.xmugrid'))
                self._bridges.append((bottom_ugrid, bridge.manning_n))
            except Exception:
                self.error_msg = f'Error with coverage: {geometry.name}. Unable to retrieve selected ' \
                                 f'3D Bridge coverage data.'
                self._logger.error(self.error_msg)
                return False

        return True

    def _select_bridges(self):
        """Display a 3D Bridge coverage selector with multi-select enabled.

        Returns:
            (:obj:`bool`): True if at least one 3D Bridge coverage was selected
        """
        # Display a dialog for selecting 3D Bridge coverages.
        selector = TreeItemSelectorDlg(
            title='Select 3D Bridge Coverages',
            target_type=xmd.CoverageItem,
            pe_tree=self._pe_tree,
            parent=self._parent,
            allow_multi_select=True
        )
        if not selector.has_selectable_items():
            msg = 'At least one 3D Bridge coverage must exist in SMS to generate pressure data sets.'
            self.error_msg = msg
            self._logger.error(self.error_msg)
            return False  # No 3D Bridge coverages currently exist in SMS.
        if not selector.exec():
            return False

        # Make sure the user selected at least one 3D Bridge coverage.
        selected_bridges = selector.get_selected_item_uuid()
        if not selected_bridges:
            self.error_msg = 'No 3D Bridge coverages selected. Operation aborted.'
            self._logger.error(self.error_msg)
            return False

        # Query SMS for the input data we need.
        return self._get_inputs_from_xms(selected_bridges)

    @staticmethod
    def is_3d_bridge_if_coverage(item):
        """Check if a tree item is a 3D Bridge, but only if it is a coverage.

        Args:
            item (:obj:`TreeNode`): The item to check

        Returns:
            (:obj:`bool`): True if the tree item is a 3D Bridge coverage type or is not a coverage.
        """
        if type(item.data) is xmd.CoverageItem:
            if item.model_name == '3D Bridge' and item.coverage_type == '3D Bridge':
                return True
            return False
        return True

    def generate_ceiling(self):
        """Run the generate pressure datasets command from XMS.

        Returns:
            (:obj:`bool`): False if the tool does not run
        """
        # Get the mesh in this simulation
        helper = SimQueryHelper(self._query)
        helper.get_geometry_data()

        # Get a dump of the current SMS project explorer tree.
        if not self._get_and_filter_project_tree():
            return False
        # Get the input 3D Bridge coverage data from the user and SMS.
        if not self._select_bridges():
            return False

        # write info about the mesh when the file was generated
        ug = helper.co_grid.ugrid
        d = {
            'mesh_uuid': self._mesh_uuid,
            'num_nodes': ug.point_count,
            'num_cells': ug.cell_count,
            'mesh_crc32': helper.co_grid_file_crc32
        }
        ceiling_mesh = os.path.join(os.path.dirname(self._sim_component.main_file), 'ceiling_mesh.json')
        with open(ceiling_mesh, 'w') as f:
            f.write(json.dumps(d))
        # Generate the pressure datasets with a feedback dialog.
        ceiling_file_name = os.path.join(os.path.dirname(self._sim_component.main_file), 'ceiling.srhceiling')
        worker = GenerateCeilingThread(helper.co_grid.ugrid, self._bridges, ceiling_file_name, helper.grid_units)
        display_text = {
            'title': 'Generating Pressure Data Sets',
            'working_prompt': 'Generating pressure data sets from 3D Bridge coverages. Please wait...',
            'error_prompt': 'Error(s) encountered generating data sets. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered generating data sets. Review log output for more details.',
            'success_prompt': 'Successfully generated pressure datasets.',
            'note': '',
            'auto_load': 'Auto load generated data sets into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, self._parent)
        feedback_dlg.exec()
        return True
