"""Read an SRH-2D *.srhhydro control file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import shlex
import sys

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.data.model_control import ModelControl
from xms.srh.file_io import io_util
from xms.srh.file_io.hydro_bc_reader import HydroBcReader


def set_args_for_key(obj, arg, word):
    """Set args from a parsed string - not really sure what this does, but I needed to cover it.

    Args:
        obj: Object with attributes specified by arg to be set.
        arg: Name of the attribute being set
        word: The value of the attribute

    Returns:
        object: The value of the key for the given word
    """
    attr = getattr(obj, arg)
    if attr is not None:
        # Cast the word to the appropriate type
        rv = type(attr)(word.strip('"\''))
    else:
        rv = word.strip('"\'')
    setattr(obj, arg, rv)
    return rv


class HydroReader:
    """Reads SRH-2D hydro file."""
    def __init__(self):
        """Initializes the class."""
        self.model_control = ModelControl()
        self.logger = logging.getLogger('xms.srh')
        self.data = {}  # Dict of lines from the file
        self.file_list = []
        self.bc_hy8_file = ''
        self.obstruction_decks = []
        self.obstruction_piers = []
        self.materials_manning = {}
        self.sed_material_data = []
        self.monitor_line_ids = set()
        self.bcs = {}
        self.structures = {}
        self.bc_arc_id_to_bc_id = {}
        self.bc_id_to_structure = {}
        self.filename = ''

    def _read_model_control(self):
        """Reads the model control data."""
        mc = self.model_control
        self._read('Case', mc.hydro, 'case_name')
        self._read('Description', mc.hydro, 'simulation_description')
        runtype = self._read('RunType', None)[0]
        if runtype.lower() == 'mobile':
            mc.enable_sediment = True
        else:
            mc.enable_sediment = False
        if not mc.enable_sediment:
            unsteady_output = self._read('unsteadyoutput', None)[0]
            if unsteady_output.lower() == 'unsteady':
                mc.advanced.unsteady_output = True
            else:
                mc.advanced.unsteady_output = False
        self._read('SimTime', mc.hydro, 'start_time', 'time_step', 'end_time')
        turbulence_model = self._read('turbulencemodel', None)[0]
        if turbulence_model:
            for item in mc.advanced.param.turbulence_model.objects:
                if item.lower() == turbulence_model.lower():
                    mc.advanced.turbulence_model = item
                    break
            if mc.advanced.turbulence_model == 'Parabolic':
                self._read('ParabolicTurbulence', mc.advanced, 'parabolic_turbulence')

        if mc.enable_sediment:
            self._read_sediment()
        self._read_file_list()
        self._read_initial_conditions()
        self._read_output()

    def _read_material(self, hydro_filename, value):
        """Reads the materials.

        Args:
            hydro_filename (:obj:`str`): Path to the hydro file being read
            value (:obj:`str`): The part of the line that follows the card.
        """
        # Use shlex to handle quoted strings
        words = shlex.split(value, posix="win" not in sys.platform)
        if words and len(words) > 1:
            mat_id = int(words[0])
            if io_util.is_float(words[1]):
                self.materials_manning[mat_id] = float(words[1])
            else:
                self.materials_manning[mat_id] = self._read_xys(hydro_filename, words[1].strip('"\''))

    def _read_sed_bed(self, hydro_filename, value):
        """Reads the sediment materials.

        Args:
            hydro_filename (:obj:`str`): Path to the hydro file being read
            value (:obj:`str`): The part of the line that follows the card.
        """
        # Use shlex to handle quoted strings
        words = shlex.split(value, posix="win" not in sys.platform)
        if words and len(words) > 1:
            mat_id = int(words[0])
            lay_id = int(words[1])
            curve = self._read_xys(hydro_filename, words[3].strip('"\''))
            self.sed_material_data.append((mat_id, lay_id, curve))

    def _read_xys(self, hydro_filename, xys_filename):
        """Reads the xys file.

        Args:
            hydro_filename (:obj:`str`): Filepath of hydro file.
            xys_filename (:obj:`str`): Filepath of xys file.

        Returns:
            (:obj:`list[tuple]`): The XY series curve as a list of x,y tuples
        """
        filename = os.path.join(os.path.dirname(hydro_filename), xys_filename)
        _, xs, ys = io_util.read_xys(filename)
        return [(x, y) for x, y in zip(xs, ys)]

    def _read_obstructions(self):
        """Read the obstruction points and arcs."""
        pass  # These are read in self.read when putting the file lines into the data dict

    def _read_sediment(self):
        """Reads the sediment info."""
        mc = self.model_control
        gen_sed_params = self._read('gensedparams', None)
        if len(gen_sed_params) > 0:
            mc.sediment.sediment_specific_gravity = float(gen_sed_params[0])
            if len(gen_sed_params) > 1:
                diameters = [float(word) for word in gen_sed_params[1:]]
                mc.sediment.particle_diameter_threshold = pd.DataFrame({'Particle Diameter Threshold (mm)': diameters})
        bed_shear = self._read('BedShearOption', None)
        if len(bed_shear) > 2:
            mc.sediment.shear_partitioning_option = 'Percentage'
            if bed_shear[0].lower() == 'scaled_d90':
                mc.sediment.shear_partitioning_option = 'Scaled D90'
            mc.sediment.shear_partitioning_scaled_d90_factor = float(bed_shear[1])
            mc.sediment.shear_partitioning_percentage = float(bed_shear[2])
        start_time = self._read('SedimentStartTime', None)
        if len(start_time) > 0 and [''] != start_time:
            mc.sediment.specify_sediment_simulation_start_time = True
            mc.sediment.sediment_simulation_start_time = float(start_time[0])
        self._read_sediment_equation(mc.sediment.transport_equation_parameters, '')
        self._read_sediment_parameters(mc.sediment.transport_parameters)
        self._read_sediment_cohesion(mc.sediment)

    def _read_sediment_equation(self, sed_eq, card_append_str):
        """Reads the sediment equation.

        Args:
            sed_eq (:obj:`SedimentEquationParameters`): Sediment equation parameters class.
            card_append_str (:obj:`str`): string to append to the file cards when writing the mixed 'lower' and
                'higher' equation parameters.
        """
        eq_dict = {
            'EH': 'Engelund-Hansen',
            'MPM': 'Meyer-Peter-Muller',
            'PARKER': 'Parker',
            'WILCOCK': 'Wilcock-Crowe',
            'WU': 'Wu-et-al',
            'YANG73': 'Yang 1973 Sand w 1984 Gravel',
            'YANG79': 'Yang 1979 Sand w 1984 Gravel',
            'MIXED': 'Mixed'
        }

        if not card_append_str:
            sed_equation = self._read('sedequation', None)[0]
            if sed_equation:
                sed_eq.sediment_transport_equation = eq_dict[sed_equation.upper()]
        if sed_eq.sediment_transport_equation == 'Meyer-Peter-Muller':
            self._read(f'CapacityHidingFactor{card_append_str}', sed_eq, 'meyer_peter_muller_hiding_factor')
        elif sed_eq.sediment_transport_equation == 'Parker':
            self._read(
                f'SedCapacityCoeff{card_append_str}', sed_eq, 'parker_reference_shields_parameter',
                'parker_hiding_coefficient'
            )
        elif sed_eq.sediment_transport_equation == 'Wilcock-Crowe':
            self._read(
                f'WilcockCoeff{card_append_str}', sed_eq, 'wilcox_t1_coefficient', 'wilcox_t2_coefficient',
                'wilcox_sand_diameter'
            )
        elif sed_eq.sediment_transport_equation == 'Wu-et-al':
            self._read(f'CritShieldsParam{card_append_str}', sed_eq, 'wu_critical_shields_parameter')
        elif sed_eq.sediment_transport_equation == 'Mixed':
            lower = sed_eq.lower_transport_parameters
            higher = sed_eq.higher_transport_parameters
            words = self._read('MixedSedParams', None)
            if len(words) > 2:
                sed_eq.mixed_sediment_size_class_cutoff = float(words[0])
                lower.sediment_transport_equation = eq_dict[words[1]]
                higher.sediment_transport_equation = eq_dict[words[2]]
            self._read_sediment_equation(lower, 'Lower')
            self._read_sediment_equation(higher, 'Higher')

    def _read_sediment_parameters(self, sed_par):
        """Reads the sediment parameters.

        Args:
            sed_par (:obj:`SedimentParameters`): Sediment equation parameters class
        """
        self._read('WaterTemperature', sed_par, 'water_temperature')
        self._read('SusLoadAdaptationCoeffs', sed_par, 'deposition_coefficient', 'erosion_coefficient')
        words = self._read('BedLoadAdaptationLength', None)
        if words:
            sed_par.adaptation_length_bedload_mode = sed_par.param.adaptation_length_bedload_mode.objects[int(words[0])]
            if sed_par.adaptation_length_bedload_mode == 'Constant Length':
                sed_par.adaptation_length_bedload_length = float(words[1])
        words = self._read('ActiveLayerThickness', None)
        if len(words) > 1:
            if int(words[0]) == 1:
                sed_par.active_layer_thickness_mode = 'Constant Thickness'
                sed_par.active_layer_constant_thickness = float(words[1])
            elif int(words[0]) == 2:
                sed_par.active_layer_thickness_mode = 'Thickness based on D90'
                sed_par.active_layer_d90_scale = float(words[1])

    def _read_sediment_cohesion(self, sed):
        """Reads the sediment cohesion.

        Args:
            sed (:obj:`ModelControlSediment`): Sediment parameters class
        """
        word = self._read('CohesiveSediment', None)[0]
        if word and int(word) == 0:
            sed.enable_cohesive_sediment_modeling = False
            return
        sed.enable_cohesive_sediment_modeling = True

        word = self._read('FallVelocity', None)[0]
        if word:
            try:
                opt = int(word)
                if opt == 0:
                    sed.cohesive.fall_velocity = 'Severn River Properties'
                elif opt == 1:  # for backwards compatibility, but this format does not work with SRH PRE
                    sed.cohesive.fall_velocity = 'Velocity Data File (mm/sec)'
                    sed.cohesive.fall_velocity_data_file = self._read('FallVelocityFile', None)[0]
            except ValueError:
                sed.cohesive.fall_velocity = 'Velocity Data File (mm/sec)'
                sed.cohesive.fall_velocity_data_file = word
        #     if int(word) == 0:
        #         sed.cohesive.fall_velocity = 'Severn River Properties'
        #     elif int(word) == 1:
        #         sed.cohesive.fall_velocity = 'Velocity Data File (mm/sec)'
        # if sed.cohesive.fall_velocity == 'Velocity Data File (mm/sec)':
        #     sed.cohesive.fall_velocity_data_file = self._read('FallVelocityFile', None)[0]

        erosion_option = 0
        word = self._read('CohesiveErosionRate', None)[0]
        # new in SRH 3.3.0 - line changed in hydro file 2 examples below
        # CohesiveErosionRate 0
        # CohesiveErosionRate "c:/temp/erosion_rate_file.txt"
        if word:
            try:
                erosion_option = int(word)
            except ValueError:
                sed.cohesive.erosion_rate = 'Erosion Data File'
                sed.cohesive.erosion_rate_data_file = word

        if erosion_option == 1:
            sed.cohesive.erosion_rate = 'Erosion Data File'
            sed.cohesive.erosion_rate_data_file = self._read('CohesiveErosionRateFile', None)[0]
        else:
            words = self._read('CohesiveErosionParams', None)
            if len(words) > 4:
                sed.cohesive.surface_erosion = float(words[0])
                sed.cohesive.mass_erosion = float(words[1])
                sed.cohesive.surface_erosion_constant = float(words[2])
                sed.cohesive.mass_erosion_constant = float(words[3])
                sed.cohesive.erosion_units = 'SI' if words[4] == 'SI' else 'English'

        words = self._read('CohesiveDepositionParams', None)
        if len(words) > 3:
            sed.cohesive.full_depostion = float(words[0])
            sed.cohesive.partial_depostion = float(words[1])
            sed.cohesive.equilibrium_concentration = float(words[2])
            sed.cohesive.deposition_units = 'SI' if words[3] == 'SI' else 'English'

    def _add_file_if_found(self, file_card):
        """Adds a file to the list of files if it is in the dict.

        Args:
            file_card (:obj:`str`): dict key and file card.
        """
        file = self._read(file_card, None)
        if file and file[0]:
            self.file_list.append(f'{file_card} {file[0]}')

    def _read_file_list(self):
        """Reads the files listed and stores them in self.file_list."""
        self._add_file_if_found('Grid')
        self._add_file_if_found('HydroMat')
        self._add_file_if_found('SubsurfaceBedFile')
        self._add_file_if_found('MonitorPtFile')
        self._add_file_if_found('PressureDatasetFile')

    def _read_initial_conditions(self):
        """Reads the initial conditions."""
        mc = self.model_control
        word = self._read('InitCondOption', None)[0].lower()
        if word == 'dry':
            mc.hydro.initial_condition = 'Dry'
        elif word == 'auto':
            mc.hydro.initial_condition = 'Automatic'
        elif word == 'wse':
            mc.hydro.initial_condition = 'Initial Water Surface Elevation'
            rv = self._read('InitZoneParams', mc.hydro, '', '', '', 'initial_water_surface_elevation', '')
            mc.hydro.initial_water_surface_elevation_units = 'Meters' if rv[4].upper() == 'SI' else 'Feet'
        elif word == 'rst':
            mc.hydro.initial_condition = 'Restart File'
            self._read('RestartFile', mc.hydro, 'restart_file')
        elif word == 'vary_we':
            mc.hydro.initial_condition = 'Water Surface Elevation Dataset'

    def _read_output(self):
        """Reads the output control."""
        mc = self.model_control
        rv = self._read('OutputFormat', None, '', '')
        mc.output.output_units = 'English' if 'EN' in rv[1].upper() else 'Metric'
        mc.output.output_format = 'XMDF' if rv[0].upper() == 'XMDFC' else rv[0]
        rv = self._read('OutputOption', None)[0]
        output_option = 1
        if rv != '':
            output_option = int(self._read('OutputOption', None)[0])
            if output_option == 2:
                mc.output.output_method = 'Specified Times'
            elif output_option == 3:
                mc.output.output_method = 'Simulation End'

        if output_option == 1:
            rv = self._read('OutputInterval', None)[0]
            if rv != '':
                output_interval = float(self._read('OutputInterval', None)[0])
                if output_interval < 0:
                    mc.output.output_method = 'Simulation End'
                else:
                    # output_interval is always in hours but we want to present it as minutes to the user
                    mc.output.output_frequency_units = 'Minutes'
                    mc.output.output_frequency = output_interval * 60
        elif output_option == 2:
            words = self.data.get('outputtimes', '').split()
            times = [float(word) for word in words]
            mc.output.output_specified_times = pd.DataFrame({'Times': times})

        mc.output.output_maximum_datasets = False
        if 'maxdatasetparams' in self.data:
            mc.output.output_maximum_datasets = True
            words = self.data['maxdatasetparams'].split()
            if len(words) > 1:
                min_time = mc.output.maximum_dataset_minimum_time = float(words[0])
                max_time = mc.output.maximum_dataset_maximum_time = float(words[1])
                if min_time != 0.0 or max_time != mc.hydro.end_time:
                    mc.output.specify_maximum_dataset_time_range = True

    def _read(self, key, obj, *args):
        """Reads the data with the key and sets the variables passed in via *args.

        Args:
            key: Key word for lookup in dict. Lowercase version of card from file.
            obj: Optional object with attributes specified in *args to be set.
            *args: Attributes of obj to be set, or ''.

        Returns:
            (:obj:`list`): List of values set with length = len(*args)
        """
        value = self.data.get(key.lower(), '')
        if not value:
            return ['']

        # words = value.split()
        # Use shlex to handle quoted strings
        words = shlex.split(value, posix="win" not in sys.platform)
        if obj:
            assert len(words) == len(args)

        rv = []  # List of return values
        for i in range(len(words)):
            if obj and args[i]:
                rv.append(set_args_for_key(obj, args[i], words[i]))
            else:
                rv.append(words[i].strip('"\''))
        return rv

    def read(self, filename):
        """Reads a *.srhhydro file.

        Args:
            filename (:obj:`str`): Path to the *.srhhydro file to read
        """
        self.filename = filename
        try:
            with open(filename, 'r') as f:
                lines = f.read().splitlines()

            if lines[0].upper() != 'SRHHYDRO 30':
                raise RuntimeError('')

            bc_reader = HydroBcReader(self)
            for line in lines:
                words = line.split(maxsplit=1)
                # self.data[words[0].lower()] = words[1].strip('"') Done later

                # There can be multiple 'DeckParams' and 'PierParams' so they won't work in the data dict.
                if words[0].lower() == 'deckparams':
                    self.obstruction_decks.append(f'{words[0]} {words[1]}')
                elif words[0].lower() == 'pierparams':
                    self.obstruction_piers.append(f'{words[0]} {words[1]}')
                elif words[0].lower() == 'manningsn':
                    self._read_material(filename, words[1])
                elif words[0].lower() == 'subsurfacethickness':
                    self.sed_material_data.append(line)
                elif words[0].lower() == 'bedsedcomposition':
                    self._read_sed_bed(filename, words[1])
                elif words[0].lower() == 'numsubsurfacelayers':
                    self.sed_material_data.append(line)
                elif words[0].lower() == 'hy8file':
                    self.bc_hy8_file = words[1].strip('"')
                elif words[0].lower() == 'bc':
                    bc_reader.add_bc(line)
                elif words[0].lower() in bc_reader.struct_type:
                    bc_reader.add_bc_struct(line)
                elif bc_reader.is_bc_key_word(words[0].lower()):
                    bc_reader.lines.append(line)
                elif len(words) > 1:
                    self.data[words[0].lower()] = words[1]

            bc_reader.read_bc_data()
            self._read_model_control()
            self._read_obstructions()

        except Exception as error:  # pragma: no cover
            self.logger.exception(f'Error reading hydro file: {str(error)}')
            raise error
