"""Routines to generate plots for the SRH-2D project summary report."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules
import folium
import folium.plugins
from plotly import graph_objects as go
import plotly.express as px

# 3. Aquaveo modules
from xms.coverage.polygons import polygon_orienteer
from xms.data_objects.parameters import FilterLocation, Projection

# 4. Local modules
from xms.srh.file_io.report import report_util
from xms.srh.file_io.report.extents import Extents
from xms.srh.gui.simulation_plots import SimulationPlots


class SummaryReportPlotter:  # pragma: no cover
    """Handles creation of all the plots used in the summary report.

    We use a mix of folium and plotly. Folium is used to display things on top of a background map. We think GeoViews
    might work better for that but we haven't figured out how to install it yet. See
    https://aquaveo.github.io/xmsmesh/examples/MeshSanDiego.html

    Plotly can also do background maps but it doesn't do them as well as Folium. In particular, framing and zooming
    is more challenging with Plotly and labels on lines don't show up. But plotly is great for simple plots like the
    ARR plot and the bridge plot.
    """
    def __init__(self):  # pragma: no cover
        """Initializes the class."""
        self._simulation_plots = None

    def plot_scatter(self, ugrid, projection, ugrid_name, report_dir):
        """Creates a plot of the points of the ugrid and returns the html.

        Args:
            ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.
            projection (:obj:`xms.data_objects.parameters.Projection`): Projection.
            ugrid_name (:obj:`str`): Name of the UGrid (used to create the filename).
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Convert to geographic if necessary
        rv, points = self._get_ugrid_points_in_geographic(ugrid, projection)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')
            return ''

        # Add points to plot
        dataframe = {'lat': [pt[1] for pt in points], 'lon': [pt[0] for pt in points]}
        # TODO: Figure out zoom level
        fig = px.scatter_mapbox(dataframe, lat='lat', lon='lon', zoom=14, height=400, width=500)
        fig.update_layout(mapbox_style='open-street-map')
        fig.update_layout(margin={"r": 0, "t": 0, "l": 0, "b": 0})

        # Get HTML
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{ugrid_name}_plot.html'))
        fig.write_html(filepath, auto_open=False)
        return os.path.basename(filepath)

    def plot_mesh(self, ugrid, projection, mesh_name, report_dir):
        """Creates a plot of the ugrid cells and returns the html.

        Args:
            ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.
            projection (:obj:`xms.data_objects.parameters.Projection`): Projection.
            mesh_name (:obj:`str`): Name of the mesh (used to create the filename).
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            filepath (:obj:`str`): Filepath to html file where the plot is saved.
        """
        # Convert to geographic if necessary
        rv, ugrid_points = self._get_ugrid_points_in_geographic(ugrid, projection)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')
            return ''

        # Hash the edges
        cell_count = ugrid.cell_count
        edge_set = set()  # Use this set to remove duplicate edges
        for cell_idx in range(cell_count):
            cell_edges = ugrid.get_cell_edges(cell_idx)
            for edge in cell_edges:
                if edge[0] > edge[1]:
                    edge_set.add((edge[1], edge[0]))
                else:
                    edge_set.add((edge[0], edge[1]))

        # Add edges as lines using folium
        m = folium.Map()
        extents = Extents()
        edges_list = []
        for edge in edge_set:
            point_list = [
                (ugrid_points[edge[0]][1], ugrid_points[edge[0]][0]),
                (ugrid_points[edge[1]][1], ugrid_points[edge[1]][0])
            ]
            edges_list.append(point_list)
            extents.add_xy(ugrid_points[edge[0]][0], ugrid_points[edge[0]][1])
            extents.add_xy(ugrid_points[edge[1]][0], ugrid_points[edge[1]][1])
        folium.vector_layers.PolyLine(locations=edges_list, weight=1, color='#000000').add_to(m)

        self._frame(extents, m)

        # Get HTML
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{mesh_name}_plot.html'))
        m.save(filepath)
        return os.path.basename(filepath)

    def plot_bc_coverage(self, coverage, plot_bc_atts, arc_ids_to_bc_ids, mesh_boundary, report_dir):
        """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            plot_bc_atts (:obj:`dict`): Dict of bc ids and bc attributes.
            arc_ids_to_bc_ids (:obj:`dict`): Dict of arc id -> bc_id.
            mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary. Optional.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Get all the points first so we can transform them all together if necessary
        arcs = coverage.arcs
        points = []
        for arc in arcs:
            arc_points = arc.get_points(FilterLocation.PT_LOC_ALL)
            for point in arc_points:
                points.append((point.x, point.y))

        extents = Extents()
        rv, points = self._transform_get_extents_and_swap_xy(points, coverage.projection, extents)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')
            return ''

        # Add the arcs to the plot
        m = folium.Map()
        start = 0
        for arc in arcs:
            bc_id = arc_ids_to_bc_ids.get(arc.id)
            if bc_id is None or bc_id not in plot_bc_atts:
                continue
            count = len(arc.get_points(FilterLocation.PT_LOC_ALL))
            end = start + count
            html = self._get_arc_bc_table_html(arc.id, bc_id, plot_bc_atts)  # Use of arc_id and arc.id is right
            popup = folium.Popup(html=html, min_width=275, max_width=275)
            polyline = folium.vector_layers.PolyLine(
                locations=points[start:end], popup=popup, tooltip=f'Arc ID: {arc.id}', color='#ff5233'
            )
            polyline.add_to(m)
            folium.plugins.PolyLineTextPath(
                polyline=polyline, text=plot_bc_atts[bc_id]['Type'], center=True, offset=10
            ).add_to(m)
            start += count

        if mesh_boundary:
            self._add_mesh_boundary(mesh_boundary, coverage.projection, m, extents)

        self._frame(extents, m)

        # Get HTML
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{coverage.name}_plot.html'))
        m.save(filepath)
        return os.path.basename(filepath)

    def plot_monitor_coverage(self, coverage, mesh_boundary, report_dir, arc_labels=None, pt_labels=None):
        """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary. Optional.
            report_dir (:obj:`str`): Path to directory where report files are created.
            arc_labels (:obj:'list'): List of arc labels.
            pt_labels (:obj:'list'): List of point labels.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Get all the points first so we can transform them all together if necessary
        # Arcs
        arcs = coverage.arcs
        arc_points = []
        arc_ids = []
        arc_labels_tmp = []
        for arc in arcs:
            arc_ids.append(arc.id)
            arc_labels_tmp.append(f'LN{arc.id}')
            do_arc_points = arc.get_points(FilterLocation.PT_LOC_ALL)
            for point in do_arc_points:
                arc_points.append((point.x, point.y))

        if arc_labels is None:
            arc_labels = arc_labels_tmp

        extents = Extents()
        rv, arc_points = self._transform_get_extents_and_swap_xy(arc_points, coverage.projection, extents)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')
            return ''

        # Points
        do_points = coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        points = []
        pt_labels_tmp = []
        for point in do_points:
            points.append((point.x, point.y))
            pt_labels_tmp.append(f'PT{point.id}')
        if pt_labels is None:
            pt_labels = pt_labels_tmp

        rv, points = self._transform_get_extents_and_swap_xy(points, coverage.projection, extents)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')
            return ''

        # Add the arcs to the plot
        m = folium.Map()
        start = 0
        for idx, arc in enumerate(arcs):
            count = len(arc.get_points(FilterLocation.PT_LOC_ALL))
            end = start + count
            polyline = folium.vector_layers.PolyLine(
                locations=arc_points[start:end], color='black', tooltip=arc_labels[idx]
            )
            polyline.add_to(m)
            folium.plugins.PolyLineTextPath(polyline=polyline, text=arc_labels[idx], center=True, offset=10).add_to(m)
            start += count

        for point, pt_label in zip(points, pt_labels):
            folium.vector_layers.CircleMarker(location=point, radius=2, color='black').add_to(m)
            # folium.vector_layers.Marker(location=point, tooltip=f'PT{pt_id}', icon=folium.Icon()).add_to(m)
            icon = folium.DivIcon(html=f'<div>{pt_label}</div>', icon_anchor=(2, 0))
            folium.vector_layers.Marker(location=point, icon=icon).add_to(m)

        if mesh_boundary:
            self._add_mesh_boundary(mesh_boundary, coverage.projection, m, extents)

        self._frame(extents, m)

        # Get HTML
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{coverage.name}_plot.html'))
        m.save(filepath)
        return os.path.basename(filepath)

    def plot_solution_plot(self, inf_file, report_dir, plot_func, args, filename):
        """Creates a 'Monitor Points WSE' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.
            plot_func (:obj:`str`): Name of SimulationPlots method to call.
            args (:obj:`tuple(str)`): Arguments to pass to plot_func.
            filename (:obj:`str`): Base name of image file.

        Returns:
            (:obj:`str`): Filepath to image file where the plot is saved.
        """
        if not self._simulation_plots:
            self._simulation_plots = SimulationPlots([inf_file])
        getattr(self._simulation_plots, plot_func)(*args)
        plot_filename = self._simulation_plots.get_unique_solution_plot_file_name(filename)
        filename = os.path.join(report_dir, plot_filename)
        self._simulation_plots.figure.savefig(filename)
        return os.path.basename(filename)

    def plot_monitor_points_wse(self, inf_file, report_dir):
        """Creates a 'Monitor Points WSE' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_monitor_point_plot', (None, None, None, None, False, 'wse'),
            'plot_monitor_points_wse.png'
        )

    def plot_monitor_points_z(self, inf_file, report_dir):
        """Creates a 'Monitor Points Z' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_monitor_point_plot', (None, None, None, None, False),
            'plot_monitor_points_z.png'
        )

    def plot_monitor_lines_q(self, inf_file, report_dir):
        """Creates a 'Monitor Lines Q' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_monitor_line_q_plot', (None, None), 'plot_monitor_lines_q.png'
        )

    def plot_monitor_lines_qs(self, inf_file, report_dir):
        """Creates a 'Monitor Lines QS' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_monitor_line_qs_plot', (None, None), 'plot_monitor_lines_qs.png'
        )

    def plot_net_q_inlet_q(self, inf_file, report_dir):
        """Creates a 'Net_Q/INLET_Q' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(inf_file, report_dir, 'create_net_q_plot', (None, None), 'plot_net_q.png')

    def plot_wet_elements(self, inf_file, report_dir):
        """Creates a 'Wet Elements' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_wet_elements_plot', (None, None), 'plot_wet_elements.png'
        )

    def plot_mass_balance(self, inf_file, report_dir):
        """Creates a 'Mass Balance' plot.

        Args:
            inf_file (:obj:`str`): Path to <case_name>_INF.dat file.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to image file where the plot is saved.
        """
        return self.plot_solution_plot(
            inf_file, report_dir, 'create_mass_balance_plot', (None, None), 'plot_mass_balance.png'
        )

    def plot_material_coverage(self, coverage, plot_polygon_atts, report_dir):
        """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            plot_polygon_atts (:obj:`dict`): Dict of polygon ids and polygon attributes.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Get all the points first so we can transform them all together if necessary
        do_polygons = coverage.polygons  # data_objects polygons
        polygons = {}  # Dict of polygon id -> polygon.
        #  "polygon" is a list of "poly", outer, then inners. "poly" is a list of xy points.
        for do_polygon in do_polygons:
            polygons[do_polygon.id] = polygon_orienteer.get_polygon_point_lists(do_polygon)

        # Transform, get extents, and swap xy
        extents = Extents()
        for polygon in polygons.values():
            for poly in polygon:
                rv, poly[:] = self._transform_get_extents_and_swap_xy(poly, coverage.projection, extents)
                if not rv:  # pragma: no cover
                    logger = logging.getLogger("Summary Report")
                    logger.warning('Could not get points in geographic coordinates.')
                    return ''

        # Add the polygons to the plot
        m = folium.Map()
        for polygon_id, polygon in polygons.items():
            if polygon_id in plot_polygon_atts:
                color_and_texture = plot_polygon_atts[polygon_id]['color']
                color = f'rgb({color_and_texture[0]}, {color_and_texture[1]}, {color_and_texture[2]})'
                name = plot_polygon_atts[polygon_id]['name']
                polygon_fo = folium.vector_layers.Polygon(
                    locations=polygon,
                    color='black',
                    weight=2,
                    fill=True,
                    fill_color=color,
                    fill_opacity=1.0,
                    popup=name
                )
                polygon_fo.add_to(m)

        self._frame(extents, m)

        # Get HTML
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{coverage.name}_plot.html'))
        m.save(filepath)
        return os.path.basename(filepath)

    @staticmethod
    def plot_bridge(xy_data, coverage_name, report_dir):
        """Creates a plot of the bridge profile lines (top, upstream, downstream).

        Args:
            xy_data (:obj:`dict`): Dict containing the x,y line data for the bridge profile lines.
            coverage_name (:obj:`str`): Name of the coverage (used to create the filename).
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Create the plot
        fig = go.Figure()
        fig.add_trace(go.Scatter(mode='lines', name='Top profile', x=xy_data['top']['x'], y=xy_data['top']['y']))
        fig.add_trace(go.Scatter(mode='lines', name='Upstream profile', x=xy_data['up']['x'], y=xy_data['up']['y']))
        fig.add_trace(
            go.Scatter(mode='lines', name='Downstream profile', x=xy_data['down']['x'], y=xy_data['down']['y'])
        )
        width = 500
        fig.update_layout(
            title=coverage_name, width=width, margin={
                "r": 0,
                "t": 30,
                "l": 0,
                "b": 0
            }, legend_orientation="h"
        )

        # Write the file
        filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{coverage_name}_plot.html'))
        fig.write_html(filepath, auto_open=False)
        return os.path.basename(filepath)

    def plot_structure(self, structures_list: list[dict], projection: Projection, report_dir: str) -> str:
        """Creates a plot of the structure locations.

        Args:
            structures_list: List of dicts containing the structure data.
            projection: The projection.
            report_dir: Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Create the plot
        m = folium.Map()
        extents = Extents()

        for data in structures_list:
            x, y = data['extents']
            points = [[x, y]]
            rv, location = report_util.transform_points_to_geographic(points, projection.well_known_text)
            extents.add_point([location[0][0], location[0][1]])
            if data['structure_type'] == 'Bridge':
                folium.Marker(
                    location=[location[0][1], location[0][0]],
                    tooltip=data['name'],
                    icon=folium.Icon(color='black', icon='bridge', prefix='fa')
                ).add_to(m)
            elif data['structure_type'] == 'Culvert':
                folium.Marker(
                    location=[location[0][1], location[0][0]],
                    tooltip=data['name'],
                    icon=folium.Icon(color='black', icon='circle', prefix='fa')
                ).add_to(m)
            folium.Marker(
                location=[location[0][1], location[0][0] + 0.0001],
                icon=folium.DivIcon(
                    html=f'<b><div style="font-size: 12px; color: black; '
                    f'white-space: nowrap;">{data["name"]}</div></b>'
                )
            ).add_to(m)
        self._frame(extents, m)

        # Write the file
        filepath = report_util.make_filename_unique(os.path.join(report_dir, 'structures_plot.html'))
        m.save(filepath)
        return os.path.basename(filepath)

    @staticmethod
    def plot_exit_h_cross_section(x_column, y_column, coverage_name, arc_id, report_dir):
        """Creates a plot of the mesh cross section at the exit-h BC and returns the path to the plot file saved.

        Args:
            x_column (:obj:`list[float]`): The x values of the cross section plot.
            y_column (:obj:`list[float]`): The y values of the cross section plot.
            coverage_name (:obj:`str`): Name of the coverage.
            arc_id (:obj:`int`): ID of the arc associated with the exit-h BC.
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            Filepath to html file where the plot is saved.
        """
        # Create the plot
        fig = go.Figure()
        fig.add_trace(go.Scatter(mode='lines', name='Cross section', x=x_column, y=y_column))
        width = 500
        fig.update_layout(
            title='Exit-H Cross Section', width=width, margin={
                "r": 0,
                "t": 30,
                "l": 0,
                "b": 0
            }, showlegend=False
        )

        # Write the file
        filepath = report_util.make_filename_unique(
            os.path.join(report_dir, f'{coverage_name}_exit-h_{arc_id}_plot.html')
        )
        fig.write_html(filepath, auto_open=False)
        return os.path.basename(filepath)

    @staticmethod
    def plot_arr_triangle(plot_points, triangle_xy, contour_dict, methods, mesh_name, report_dir):
        """Creates the ARR mesh quality plot.

        Args:
            plot_points (:obj:`list`): List of plot data points as xy pairs.
            triangle_xy (:obj:`dict`): Dict of x and y lists of coordinates forming the main triangle polygon.
            contour_dict (:obj:`dict`): Dict of method -> list of contours
            methods (:obj:`list[QualtiyMeasure]`): List of the plot types to include.
            mesh_name (:obj:`str`): Name of the mesh (used to create the filename).
            report_dir (:obj:`str`): Path to directory where report files are created.

        Returns:
            files (:obj:`list[str]`): Filepaths to html files containing the plot.
        """
        # Get list of x and y coordinates
        pt_x = [pt[0] for pt in plot_points]
        pt_y = [pt[1] for pt in plot_points]

        # Dashed triangle line
        dash_x = [50.0, 37.50]
        dash_y = [0.0, 21.65]

        # Axis and scale
        xrange = [-5.0, 55.0]
        yrange = [-5.0, 33.0]
        y_ratio = 0.63333
        width = 500
        height = width * y_ratio

        plot_titles = ['Q(alpha_min)', 'Q(Ll)', 'Q(ALS)', 'Q(Rr)', 'Q(Lr)', 'Q(Lh)']
        files = []
        fig = go.Figure()
        for method in methods:
            title = plot_titles[method]

            # Add points
            fig.add_trace(go.Scatter(mode='markers', x=pt_x, y=pt_y, marker=dict(color='Black', size=2)))

            # Add triangle lines
            fig.add_trace(
                go.Scatter(mode='lines', x=triangle_xy['x'], y=triangle_xy['y'], line=dict(color='Black', width=3))
            )
            fig.add_trace(go.Scatter(mode='lines', x=dash_x, y=dash_y, line=dict(color='Black', width=2, dash='dash')))

            # Add contours
            contours = contour_dict[method]
            contour_colors = ['rgb(255, 0, 0)', 'rgb(255, 255, 0)', 'rgb(0, 255, 0)']  # Red, Yellow, Green
            for i, contour in enumerate(contours):
                fig.add_trace(
                    go.Scatter(
                        mode='lines', x=contour['x'], y=contour['y'], line=dict(color=contour_colors[i], width=2)
                    )
                )

            # Set scale
            fig.update_xaxes(showticklabels=False, range=xrange)
            fig.update_yaxes(showticklabels=False, range=yrange, scaleanchor='y', scaleratio=y_ratio)
            fig.update_layout(
                title=title, height=height, width=width, showlegend=False, margin={
                    "r": 0,
                    "t": 30,
                    "l": 0,
                    "b": 0
                }
            )

            # Get HTML
            filepath = report_util.make_filename_unique(os.path.join(report_dir, f'{mesh_name}_{title}_plot_.html'))
            files.append(os.path.basename(filepath))
            fig.write_html(filepath, auto_open=False)
        return files

    @staticmethod
    def _get_arc_bc_table_html(arc_id, bc_id, plot_bc_atts):
        """Returns the html table containing the arc's BC data.

        Args:
            arc_id (:obj:`int`): The arc ID.
            bc_id (:obj:`int`): BC id.
            plot_bc_atts (:obj:`dict`): Dict of bc_id ->dict of arc BC data.

        Returns:
            See description.
        """
        html = ''
        if bc_id in plot_bc_atts:
            bc_jinja = plot_bc_atts[bc_id]
            html = '<table>'
            for key, value in bc_jinja.items():
                if 'plot' not in key and 'channel_calculator' not in key:
                    # If it's the 'Arc ID' and a structure, use the arc_id arg as 'Arc ID' may be the companion
                    if key == 'Arc ID' and 'arc_upstream' in bc_jinja:
                        value = arc_id
                    html += f'<tr><td>{key}:&nbsp;</td><td>{value}</td></tr>'
            html += '</table>'
        return html

    def _add_mesh_boundary(self, mesh_boundary, projection, folium_map, extents):
        """Adds the mesh boundary to the figure.

        Args:
            mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary.
            projection (:obj:`data_objects.parameters.Projection`): CRS (coordinate reference system)
                that the coverage is in.
            folium_map (:obj:`folium.Map`): The folium Map.
            extents (:obj:`Extents`): Extents of the data.
        """
        # Get all the points up front so we can transform them all together if necessary
        points = []
        ugrid_points = mesh_boundary.ugrid_points
        for _mesh_name, multi_poly in mesh_boundary.multi_polys.items():  # There should be only 1
            for polygon in multi_poly:  # If mesh is disjoint there may be more than 1
                for poly in polygon:  # If mesh has holes there will be more than 1
                    for point in poly:
                        points.append((ugrid_points[point][0], ugrid_points[point][1]))

        rv, points = self._transform_get_extents_and_swap_xy(points, projection, extents)
        if not rv:  # pragma: no cover
            logger = logging.getLogger("Summary Report")
            logger.warning('Could not get points in geographic coordinates.')

        # Add the polylines to the plot
        start = 0
        for _mesh_name, multi_poly in mesh_boundary.multi_polys.items():  # There should be only 1
            for polygon in multi_poly:  # If mesh is disjoint there may be more than 1
                for poly in polygon:  # If mesh has holes there will be more than 1
                    count = len(poly)
                    end = start + count
                    polyline = folium.vector_layers.PolyLine(locations=points[start:end], tooltip='Mesh boundary')
                    polyline.add_to(folium_map)
                    start += count

    @staticmethod
    def _frame(extents, folium_map):
        """Frames the map based on the extents.

        Args:
            extents (:obj:`Extents`): The min and max coordinates of the data.
            folium_map(:obj:`folium.Map`): The folium map
        """
        # Get center point
        center_lon = (extents.max.x + extents.min.x) / 2.0
        center_lat = (extents.max.y + extents.min.y) / 2.0

        # Create the folium map
        folium_map.location = [center_lon, center_lat]
        folium_map.fit_bounds([(extents.min.y, extents.min.x), (extents.max.y, extents.max.x)])

    @staticmethod
    def _transform_get_extents_and_swap_xy(points, projection, extents):
        """Does 3 things: transforms the points to geographic (if necessary), gets the extents, and swaps x and y.

        Args:
            points (:obj:`list`): List of points.
            projection (:obj:`data_objects.parameters.Projection`): CRS (coordinate reference system) that the
                coverage is in.
            extents (:obj:`Extents`): Extents of the points

        Returns:
            (:obj:`tuple`): tuple containing:

                rv (:obj:`bool`): Return value. True if OK, False if errors.

                new_points (:obj:`list`): List of x,y points.
        """
        # Transform to geographic if necessary
        if projection and projection.coordinate_system != 'GEOGRAPHIC':
            rv, points = report_util.transform_points_to_geographic(points, projection.well_known_text)
            if not rv:
                return False, points

        # Get point extents and swap x and y
        new_points = []
        for point in points:
            extents.add_point(point)
            # Swap x and y because folium wants them latitude (y) then longitude (x)
            new_points.append((point[1], point[0]))

        return True, new_points

    @staticmethod
    def _get_ugrid_points_in_geographic(ugrid, projection):
        """Returns the UGrid point locations transformed (if necessary) into geographic space.

        Args:
            ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.
            projection (:obj:`data_objects.parameters.Projection`): Projection.

        Returns:
            (:obj:`tuple`): tuple containing:

                rv (:obj:`bool`): Return value. True if OK, False if errors.

                points (:obj:`list`): List of the UGrid points.
        """
        # Convert to geographic
        if projection and projection.coordinate_system != 'GEOGRAPHIC':
            rv, points = report_util.transform_points_to_geographic(ugrid.locations, projection.well_known_text)
        else:
            rv = True
            points = ugrid.locations
        return rv, points


def plot_scatter(ugrid, projection, ugrid_name, report_dir):  # pragma: no cover
    """Creates a plot of the points of the ugrid and returns the html.

    Args:
        ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.
        projection (:obj:`data_objects.parameters.Projection`): Projection.
        ugrid_name (:obj:`str`): Name of the UGrid (used to create the filename).
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_scatter(ugrid, projection, ugrid_name, report_dir)


def plot_mesh(ugrid, projection, mesh_name, report_dir):  # pragma: no cover
    """Creates a plot of the ugrid cells and returns the html.

    Args:
        ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.
        projection (:obj:`data_objects.parameters.Projection`): Projection.
        mesh_name (:obj:`str`): Name of the mesh (used to create the filename).
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_mesh(ugrid, projection, mesh_name, report_dir)


def plot_bc_coverage(coverage, plot_bc_atts, arc_ids_to_bc_ids, mesh_boundary, report_dir):  # pragma: no cover
    """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

    Args:
        coverage (:obj:`data_objects.parameters.Coverage`): The coverage.
        plot_bc_atts (:obj:`dict`): Dict of bc ids and bc attributes.
        arc_ids_to_bc_ids (:obj:`dict`): Dict of arc id -> bc_id.
        mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary. Optional.
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_bc_coverage(coverage, plot_bc_atts, arc_ids_to_bc_ids, mesh_boundary, report_dir)


def plot_monitor_coverage(coverage, mesh_boundary, report_dir, arc_labels, pt_labels):  # pragma: no cover
    """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

    Args:
        coverage (:obj:`data_objects.parameters.Coverage`): The coverage.
        mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary. Optional.
        report_dir (:obj:`str`): Path to directory where report files are created.
        arc_labels (:obj:`list`): List of arc labels.
        pt_labels (:obj:`list`): List of point labels.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_monitor_coverage(coverage, mesh_boundary, report_dir, arc_labels, pt_labels)


def plot_material_coverage(coverage, plot_polygon_atts, report_dir):  # pragma: no cover
    """Creates a plot of the arcs in the coverage with their labels, and the mesh boundary.

    Args:
        coverage (:obj:`data_objects.parameters.Coverage`): The coverage.
        plot_polygon_atts (:obj:`dict`): Dict of polygon ids and polygon attributes.
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_material_coverage(coverage, plot_polygon_atts, report_dir)


def plot_bridge(xy_data, coverage_name, report_dir):  # pragma: no cover
    """Creates a plot of the bridge profile lines (top, upstream, downstream).

    Args:
        xy_data (:obj:`dict`): Dict containing the x,y line data for the bridge profile lines.
        coverage_name (:obj:`str`): Name of the coverage (used to create the filename).
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_bridge(xy_data, coverage_name, report_dir)


def plot_structure(structures_list: list[dict], projection: Projection, report_dir: str) -> str:
    """Creates a plot of the structure locations.

    Args:
        structures_list: List of dicts containing the structure data.
        projection: The projection.
        report_dir: Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_structure(structures_list, projection, report_dir)


def plot_exit_h_cross_section(x_column, y_column, coverage_name, arc_id, report_dir):  # pragma: no cover
    """Creates a plot of the mesh cross section at the exit-h BC and returns the path to the plot file saved.

    Args:
        x_column (:obj:`list[float]`): The x values of the cross section plot.
        y_column (:obj:`list[float]`): The y values of the cross section plot.
        coverage_name (:obj:`str`): Name of the coverage.
        arc_id (:obj:`int`): ID of the arc associated with the exit-h BC.
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        Filepath to html file where the plot is saved.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_exit_h_cross_section(x_column, y_column, coverage_name, arc_id, report_dir)


def plot_arr_triangle(plot_points, triangle_xy, contour_dict, methods, mesh_name, report_dir):  # pragma: no cover
    """Creates the ARR mesh quality plot.

    Args:
        plot_points (:obj:`list`): List of plot data points as xy pairs.
        triangle_xy (:obj:`dict`): Dict of x and y lists of coordinates forming the main triangle polygon.
        contour_dict (:obj:`dict`): Dict of method -> list of contours
        methods (:obj:`list[QualtiyMeasure]`): List of the plot types to include.
        mesh_name (:obj:`str`): Name of the mesh (used to create the filename).
        report_dir (:obj:`str`): Path to directory where report files are created.

    Returns:
        filepath (:obj:`str`): Filepath to html file containing the plot.
    """
    plotter = SummaryReportPlotter()
    return plotter.plot_arr_triangle(plot_points, triangle_xy, contour_dict, methods, mesh_name, report_dir)
